/*
 * RageIRCd: an advanced Internet Relay Chat daemon (ircd).
 * (C) 2000-2005 the RageIRCd Development Team, all rights reserved.
 *
 * This software is free, licensed under the General Public License.
 * Please refer to doc/LICENSE and doc/README for further details.
 *
 * $Id: motd.c,v 1.29.2.1 2004/12/07 03:05:16 pneumatus Exp $
 */

#include "config.h"
#include "struct.h"
#include "common.h"
#include "sys.h"
#include "numeric.h"
#include "msg.h"
#include "channel.h"
#include "h.h"
#include "memory.h"
#include "dlink.h"
#include <sys/stat.h>
#include <fcntl.h>
#include <utmp.h>
#include <time.h>

MessageFile *motd = NULL;
MessageFile *opermotd = NULL;
MessageFile *rules = NULL;

static const char *types_short[] = { "MOTD", "OPERMOTD", "RULES" };
static const char *types_full[] = { "Message of the Day", "Operator Message of the Day", "Server Rules" };

static MessageFile *make_message_file(char *filename, short type)
{
	MessageFile *newmf;

	ASSERT(filename != NULL);

	newmf = (MessageFile *)MyMalloc(sizeof(MessageFile));
	DupString(newmf->filename, filename);
	newmf->type = type;
	newmf->changed_date[0] = '\0';

	return newmf;
}

MessageFile *load_message_file(char *filename, short type)
{
	MessageFile *mf;
	char line[128], *p;
	int fd = -1, len = 0;
	struct stat sb;
	struct tm *localtm;

	ASSERT(filename != NULL);

	if ((fd = open(filename, O_RDONLY)) == -1) {
		return NULL;
	}
	if (fstat(fd, &sb) == -1 || !sb.st_size) {
		close(fd);
		return NULL;
	}

	mf = make_message_file(filename, type);

	dgets(-1, NULL, 0);
	while ((len = dgets(fd, line, 128)) > 0) {
		if ((p = strchr(line, '\n')) != NULL) {
			*p = '\0';
		}
		if ((p = strchr(line, '\r')) != NULL) {
			*p = '\0';
		}

		line[len] = '\0';
		DupString(p, line);
		dlink_add_tail(&mf->data, p);
	}

	if ((localtm = localtime(&sb.st_mtime)) != NULL) {
		ircsprintf(mf->changed_date, "%d/%d/%d %02d:%02d", localtm->tm_mday, localtm->tm_mon + 1,
			localtm->tm_year + 1900, localtm->tm_hour, localtm->tm_min);
	}
	close(fd);

	return mf;
}

void destroy_message_file(MessageFile *mf)
{
	dlink_node *node, *next = NULL;
	char *line;

	ASSERT(mf != NULL);

	if (mf->filename != NULL) {
		MyFree(mf->filename);
	}
	DLINK_FOREACH_SAFE_DATA(mf->data.head, node, next, line, char) {
		dlink_del(&mf->data, NULL, node);
		MyFree(line);
	}

	MyFree(mf);
}

void init_message_files()
{
	motd = load_message_file(MOTD_FILE, MFTYPE_MOTD);
	opermotd = load_message_file(OPERMOTD_FILE, MFTYPE_OPERMOTD);
	rules = load_message_file(RULES_FILE, MFTYPE_RULES);
}

void rehash_message_files()
{
	if (motd != NULL) {
		destroy_message_file(motd);
	}
	if (opermotd != NULL) {
		destroy_message_file(opermotd);
	}
	if (rules != NULL) {
		destroy_message_file(rules);
	}

	init_message_files();
}

void send_message_file(aClient *sptr, MessageFile *mf)
{
	dlink_node *node;

	ASSERT(sptr != NULL);

	if (mf == NULL) {
		send_me_noticeNA(sptr, ":File is missing");
		return;
	}

	if (*mf->changed_date != '\0') {
		send_me_notice(sptr, ":%s last changed on %s.", types_full[mf->type],
			mf->changed_date);
	}

	DLINK_FOREACH(mf->data.head, node) {
		send_me_numeric(sptr, RPL_MOTD, (char *)node->data);
	}
	send_me_notice(sptr, ":End of %s", types_short[mf->type]);
}

void send_motd(aClient *sptr)
{
	dlink_node *node;

	ASSERT(sptr != NULL);

	if (motd == NULL) {
		send_me_numericNA(sptr, ERR_NOMOTD);
		return;
	}

	if (*motd->changed_date != '\0') {
		send_me_notice(sptr, ":Message of the Day last changed on %s.", motd->changed_date);
	}

	send_me_numeric(sptr, RPL_MOTDSTART, me.name);
	DLINK_FOREACH(motd->data.head, node) {
		send_me_numeric(sptr, RPL_MOTD, (char *)node->data);
	}
	send_me_numericNA(sptr, RPL_ENDOFMOTD);
}

