/*
 * RageIRCd: an advanced Internet Relay Chat daemon (ircd).
 * (C) 2000-2005 the RageIRCd Development Team, all rights reserved.
 *
 * This software is free, licensed under the General Public License.
 * Please refer to doc/LICENSE and doc/README for further details.
 *
 * $Id: ircd.c,v 1.239.2.14 2005/07/27 19:34:28 amcwilliam Exp $
 */

#include "struct.h"
#include "common.h"
#include "sys.h"
#include "numeric.h"
#include "msg.h"
#include "memory.h"
#include "h.h"
#include "patchlevel.h"
#include "fd.h"
#include "setup.h"
#include "isupport.h"
#include "channel.h"
#include "res.h"
#include "ssl.h"
#include "sbuf.h"
#include "conf2.h"
#include "config.h"
#include "hook.h"
#include "dlink.h"
#include "xmode.h"
#include "modules.h"
#include "user_ban.h"
#include "zlink.h"
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/resource.h>
#include <errno.h>
#include <pwd.h>
#include <signal.h>
#include <fcntl.h>

static size_t initial_vmem = 0;
static int ircd_running = 0;

static char *usage[] = {
	" ",
	"RageIRCd: an advanced Internet Relay Chat daemon (ircd).",
	"(C) 2000-2005 the RageIRCd Development Team, all rights reserved.",
	" ",
	"Usage: rageircd [flags]",
	" ",
	"Where [flags] can be any of the following:",
	" ",
#ifdef CMDLINE_CONFIG
	"    -c file   Load file instead of default config file",
#endif
	"    -h        Displays usage information",
#ifndef STATIC_MODULES
	"    -m        Rehash server modules (SIGUSR1)",
#endif
#if defined(USE_OPENSSL) && defined(OS_CYGWIN)
	"    -p PASS   SSL private key password",
#endif
	"    -r        Rehash server config file (SIGHUP)",
	"    -s        Stop server (SIGTERM)",
	"    -t        Run in terminal",
	"    -v        Show server version",
	"    -V        Be verbose",
#ifdef DEBUGMODE
	"    -x n      Run in debug level n",
#endif
	" ",
	NULL
};

static ConfigOption server_log_files[] = {
	{ LOGFILE_DEFAULT, LOG_DEFAULT },
	{ LOGFILE_ERROR, LOG_ERROR },
	{ LOGFILE_KILL, LOG_KILL },
	{ LOGFILE_CLIENT, LOG_CLIENT },
	{ LOGFILE_SERVER, LOG_SERVER },
	{ LOGFILE_OPER, LOG_OPER },
	{ LOGFILE_OVERRIDE, LOG_OVERRIDE },
	{ NULL, 0L }
};

char server_options[] = {
#ifdef DEBUGMODE
	'D',
#endif
#ifdef LOG_LIBADNS_DEBUG
	'A',
#endif
#ifdef STATIC_MODULES
	'S',
#endif
#ifdef USE_OPENSSL
	'E',
#endif
#ifdef USE_THROTTLE
	'T',
#endif
#ifdef USE_SELECT
	's',
#endif
#ifdef USE_POLL
	'p',
#endif
#ifdef USE_KQUEUE
	'k',
#endif
#ifdef USE_EPOLL
	'e',
#endif
#ifdef USE_RTSIGIO
	'R',
#endif
#ifdef USE_DEVPOLL
	'd',
#endif
#ifdef INCREASE_SOCK_BUFS
	'I',
#endif
#ifdef FLUD
	'F',
#endif
#ifdef OS_CYGWIN
	'W',
#endif
#ifdef CMDLINE_CONFIG
	'c',
#endif
#ifdef HAVE_MMAP
	'M',
#endif
#ifdef HIDE_SPOOFED_IPS
	'O',
#endif
#ifdef USE_ASSERT
	'a',
#endif
#ifdef USE_ZLIB
	'Z',
#endif
	'\0'
};

char **myargv;
char isupport_buf[ISUPPORT_LINECNT][ISUPPORT_LINELEN];

aClient me, *client = &me;
struct Counter Count;
IRCd Internal;
int signal_type = 0;

dlink_list lunknown_list = DLINK_LIST_INIT;
dlink_list lclient_list = DLINK_LIST_INIT;
dlink_list lserver_list = DLINK_LIST_INIT;
dlink_list oper_list = DLINK_LIST_INIT;
dlink_list listingcli_list = DLINK_LIST_INIT;

#ifdef RLIMIT_FDMAX
#define RLIMIT_FD_MAX RLIMIT_FDMAX
#else
#ifdef RLIMIT_NOFLE
#define RLIMIT_FD_MAX RLIMIT_NOFILE
#else
#ifdef RLIMIT_OPEN_MAX
#define RLIMIT_FD_MAX RLIMIT_OPEN_MAX
#else
#undef RLIMIT_FD_MAX
#endif
#endif
#endif

static size_t get_vmem()
{
	return (size_t)sbrk(0);
}

unsigned long get_max_rss()
{
	return (get_vmem() - initial_vmem);
}

static void display_usage()
{
	char **text;

	fprintf(stderr, LF"rageircd %s (%s)"LF, ircd_version, revision);
	fprintf(stderr, "Created %s, compile #%s"LF, creation, compile_number);

	for (text = usage; *text != NULL; text++) {
		fprintf(stderr, "%s"LF, *text);
	}
}

static void save_statsfile()
{
	FILE *fd = NULL;
	if ((fd = fopen(STATS_FILE, "w+")) != NULL) {
		fprintf(fd, "%d %d %ld %ld %ld %ld %ld %ld %ld"LF, Count.max_loc, Count.max_tot,
			Count.weekly, Count.monthly, Count.yearly, Count.start, Count.week,
			Count.month, Count.year);
		fprintf(fd, "%d %d %d %d %d", Count.local, Count.total, Count.cli_restart, Count.oper, Count.chan);
		fclose(fd);
	}
}

static void init_statsfile()
{
	FILE *fd = NULL;
	char buf[1024], *item, *p;
	int cnt = 0;

	if (GeneralConfig.statsfile_save_freq) {
		add_event("save_statsfile", save_statsfile, NULL, GeneralConfig.statsfile_save_freq, 1);
	}
	if ((fd = fopen(STATS_FILE, "r")) == NULL) {
		return;
	}

	fgets(buf, 1024, fd);
	if (BadPtr(buf)) {
		fclose(fd);
		return;
	}

	for (item = strtoken(&p, buf, " "); item != NULL; item = strtoken(&p, NULL, " "), cnt++) {
		switch (cnt) {
			case 0:
				Count.max_loc = atoi(item);
				break;
			case 1:
				Count.max_tot = atoi(item);
				break;
			case 2:
				Count.weekly = atol(item);
				break;
			case 3:
				Count.monthly = atol(item);
				break;
			case 4:
				Count.yearly = atol(item);
				break;
			case 5:
				Count.start = atol(item);
				break;
			case 6:
				Count.week = atol(item);
				break;
			case 7:
				Count.month = atol(item);
				break;
			case 8:
				Count.year = atol(item);
				break;
			default:
				ircdlog(LOG_ERROR, "init_statsfile(): ignoring extra parameter (cnt %d) in %s",
					cnt, STATS_FILE);
		}
	}
	fclose(fd);

	if (Internal.verbose) {
		fprintf(stderr, "Loaded IRC statistics file."LF);
	}
}

void set_time()
{
	struct timeval new_time;

	new_time.tv_sec = 0;
	new_time.tv_usec = 0;

	if (gettimeofday(&new_time, NULL) == -1) {
		ircdlog(LOG_ERROR, "Clock failure (%d), TS may become corrupt!", errno);
		sendto_realops("Clock failure (%d), TS may become corrupt!", errno);
		restart("Clock failure!");
	}

	if (new_time.tv_sec < timeofday) {
		set_back_events(timeofday - new_time.tv_sec);
	}

	Internal.ircd_time.tv_sec = new_time.tv_sec;
	Internal.ircd_time.tv_usec = new_time.tv_usec;
}

static void write_pidfile()
{
	int fd = -1, len;
	char buf[20];

	if ((fd = open(PID_FILE, O_CREAT|O_WRONLY, 0600)) == -1) {
		Debug((DEBUG_NOTICE, "Error opening pid file %s", PID_FILE));
		ircdlog(LOG_ERROR, "Error opening pid file %s", PID_FILE);
		return;
	}

	len = ircsprintf(buf, "%5d", (int)getpid());
	if (write(fd, buf, len) == -1) {
		Debug((DEBUG_NOTICE, "Error writing to pid file %s", PID_FILE));
		ircdlog(LOG_ERROR, "Error opening pid file %s", PID_FILE);
	}
	close(fd);
}

static int check_pid()
{
	char buf[20];
	pid_t pid = 0;
	int fd, retval = 0;

	if ((fd = open(PID_FILE, O_RDONLY)) == -1) {
		if (errno != ENOENT) {
			fprintf(stderr, "WARNING: problem opening %s: %s"LF, PID_FILE, strerror(errno));
		}
		return 0;
	}

	if (read(fd, buf, sizeof(buf)) > 0) {
		if ((pid = atoi(buf)) != (int)getpid() && !kill(pid, 0)) {
			retval = (int)pid;
		}
	}
	close(fd);

	return retval;
}

static void init_internal()
{
	memset(&Internal, '\0', sizeof(Internal));

	Internal.conf_file = CONFIG_FILE;
	Internal.debug_level = Internal.next_event = -1;
	Internal.max_con_count = Internal.max_cli_count = 1;
	Internal.default_capabs = DEFAULT_CAPABS;
}

static void init_daemon()
{
	int pid;

	if (!isatty(0)) {
		return;
	}

	if ((pid = fork()) < 0) {
		fprintf(stderr, "ERROR: couldn't fork (%s)"LF, strerror(errno));
		exit(EXIT_FAILURE);
	}
	else if (pid > 0) {
		exit(EXIT_SUCCESS);
	}

	setsid();
	close(0);
}

static void init_stdio()
{
	int i, fd = -1;

	for (i = 0; i < MAXCONNECTIONS; i++) {
		if (fd_table[i].open) {
			fd_close(i);
			continue;
		}
		close(i);
	}

	for (i = 0; i < LOWEST_FD; i++) {
		fd = -1;

#ifdef DEBUGMODE
		if (Internal.debug_level >= 0 && i == 2 && !Internal.run_in_terminal) {
			truncate(LOGFILE_DEBUG, 0);
			if ((fd = open(LOGFILE_DEBUG, O_CREAT|O_WRONLY, 0600)) >= 0) {
				Debug((DEBUG_FATAL, "Debug: file %s level %d at %s", LOGFILE_DEBUG,
					Internal.debug_level, myctime(timeofday)));
				if (fd != 2) {
					dup2(fd, 2);
					close(fd);
				}
			}
		}
#endif
		if (fd < 0) {
			if ((fd = open("/dev/null", O_RDWR)) < 0) {
				exit(EXIT_FAILURE);
			}
		}
	}
}


void restart(char *message)
{
	static int been_here = 0;
	if (!been_here) {
		ircdlog(LOG_DEFAULT, "Server restarting: %s (%lu)", message, get_max_rss());
		been_here = 1;
		server_reboot();
	}
}

void server_reboot()
{
	char *errmsg;
	int i;

	sendto_realops("Aieeeeee! Restarting server... (%lu)", get_max_rss());
	Debug((DEBUG_DEBUG, "Restarting server..."));
	ircdlog(LOG_DEFAULT, "Restarting server...");

	for (i = 3; i < MAXCONNECTIONS; i++) {
		close(i);
	}

	execv(IRCD_EXE, myargv);

	errmsg = strerror(errno);
	ircdlog(LOG_ERROR, "FATAL: execv(%s,%s) failed (%s)", IRCD_EXE, myargv[0], errmsg);
	Debug((DEBUG_FATAL, "Couldn't restart server: %s", errmsg));

	exit(EXIT_FAILURE);
}

static void parse_args(int argc, char *argv[])
{
	extern char *optarg;
	int i = -1;

	while ((i = getopt(argc, argv, "c:hmp:rstvVx:")) != -1) {
		switch (i) {
			case 'c':
#ifdef CMDLINE_CONFIG
				Internal.conf_file = optarg;
#else
				fprintf(stderr, "ERROR: the -c option is not enabled."LF);
				exit(EXIT_FAILURE);
#endif
				break;
			case 'h':
				display_usage();
				exit(EXIT_SUCCESS);
				break;
#ifndef STATIC_MODULES
			case 'm':
				signal_type = 3;
				break;
#endif
#if defined(USE_OPENSSL) && defined(OS_CYGWIN)
			case 'p':
				strncpyzt(Internal.ssl_privkey_passwd, optarg, 512);
				break;
#endif
			case 'r':
				signal_type = 1;
				break;
			case 's':
				signal_type = 2;
				break;
			case 't':
				Internal.run_in_terminal = 1;
				break;
			case 'v':
				fprintf(stderr, "rageircd %s (%s)"LF, ircd_version, revision);
				fprintf(stderr, "Created %s, compile #%s"LF, creation, compile_number);
				exit(EXIT_SUCCESS);
				break;
			case 'V':
				Internal.verbose = 1;
				break;
#ifdef DEBUGMODE
			case 'x':
				if (optarg != NULL) {
					Internal.debug_level = atoi(optarg);
					Internal.debug_mode = optarg;
				}
				break;
#endif
			default:
				display_usage();
				exit(EXIT_SUCCESS);
		}
	}
}

static inline void build_version()
{
	ircsprintf(ircd_version, "%s(%s)-%d.%d.%d%s", BASENAME, CODENAME, IRCD_VERSION,
		MAJOR_VERSION, RELEASE_VERSION, IRCD_PATCHES);
}

void build_isupport()
{
	ircsnprintf(isupport_buf[0], ISUPPORT_LINELEN, FEATURES1, VALUES1);
	ircsnprintf(isupport_buf[1], ISUPPORT_LINELEN, FEATURES2, VALUES2);
}

static void init_core()
{
#ifdef RLIMIT_CORE
	struct rlimit ircd_lim;
	if (!getrlimit(RLIMIT_CORE, &ircd_lim)) {
		ircd_lim.rlim_cur = ircd_lim.rlim_max;
		if (setrlimit(RLIMIT_CORE, &ircd_lim)) {
			fprintf(stderr, "WARNING: unlimit core size failed (%s)"LF, strerror(errno));
		}
	}
#endif
}

static void init_vmem()
{
	initial_vmem = get_vmem();
}

static void init_counter()
{
	set_time();

	Count.server = 1;
	Count.oper = 0;
	Count.chan = 0;
	Count.local = 0;
	Count.total = 0;
	Count.invisi = 0;
	Count.unknown = 0;
	Count.max_loc = 0;
	Count.max_tot = 0;
	Count.cli_restart = 0;
	Count.today = 0;
	Count.weekly = 0;
	Count.monthly = 0;
	Count.yearly = 0;
	Count.start = timeofday;
	Count.day = timeofday;
	Count.week = timeofday;
	Count.month = timeofday;
	Count.year = timeofday;
}

static void init_hash_tables()
{
	clear_client_hash_table();
	clear_channel_hash_table();
	clear_scache_hash_table();
}

static void init_maximum_fds()
{
#ifdef RLIMIT_FD_MAX
	struct rlimit limit;

	if (!getrlimit(RLIMIT_FD_MAX, &limit)) {
		if (limit.rlim_max < MAXCONNECTIONS) {
			fprintf(stderr, "ERROR: rageircd fd table is too big, please fix MAXCONNECTIONS!"LF);
			fprintf(stderr, "       system fd limit: %ld current MAXCONNECTIONS: %d"LF,
				(long)limit.rlim_max, MAXCONNECTIONS);
			exit(EXIT_FAILURE);
		}

		limit.rlim_cur = limit.rlim_max;
		if (setrlimit(RLIMIT_FD_MAX, &limit) == -1) {
			fprintf(stderr, "ERROR: failed to set max fd limit to %ld"LF, (long)limit.rlim_max);
			exit(EXIT_FAILURE);
		}

#ifdef USE_SELECT
		if (MAXCONNECTIONS > FD_SETSIZE) {
			fprintf(stderr, "ERROR: FD_SETSIZE: %d MAXCONNECTIONS: %d"LF, FD_SETSIZE,
				MAXCONNECTIONS);
			fprintf(stderr, "       Make sure your system supports a large FD_SETSIZE,"LF);
			fprintf(stderr, "       then reconfigure with a max fd limit of %ld."LF,
				MAXCONNECTIONS);
			exit(EXIT_FAILURE);
		}
#endif
	}
#endif
}

static void calculate_bandwidth()
{
	static time_t last_performed = 0;
	static long last_sendK = 0, last_recvK = 0;
	long delta = (timeofday - last_performed);

	last_performed = timeofday;

	Internal.curr_sendK += ((float)(me.localClient->sendK - last_sendK) / (float)delta);
	Internal.curr_sendK /= 2;
	last_sendK = me.localClient->sendK;

	Internal.curr_recvK += ((float)(me.localClient->receiveK - last_recvK) / (float)delta);
	Internal.curr_recvK /= 2;
	last_recvK = me.localClient->receiveK;
}

static void io_loop()
{
	time_t next_expire = 0;
	int expire_stage = 0;

	while (ircd_running) {
		if (Internal.next_event == -1) {
			get_next_event();
		}
		if (Internal.next_event <= timeofday) {
			run_events();
		}

		if (timeofday >= next_expire) {
			next_expire = timeofday + 13;
			Debug((DEBUG_DEBUG, "Ban expire: stage %d, next expire %ld",
				expire_stage, next_expire));

			if (!expire_stage) {
				expire_userbans();
				expire_stage++;
			}
			else if (expire_stage == 1) {
				expire_simbans();
#ifdef USE_THROTTLE
				expire_stage++;
#else
				expire_stage = 0;
#endif
			}
#ifdef USE_THROTTLE
			else if (expire_stage == 2) {
				throttle_timer(timeofday);
				expire_stage = 0;
			}
#endif
		}

		hook_run(h_pre_netio, NULL);

		engine_do_netio(1000);
		send_queued_buffers();

		hook_run(h_post_netio, NULL);

		if (Internal.do_rehash) {
			conf_rehash();

			Internal.do_rehash = 0;
			sendto_realops("Config file reloaded.");
		}

#ifndef STATIC_MODULES
		if (Internal.do_module_rehash) {
			unload_all_modules();
			load_all_modules();
			
			check_core_commands();

			Internal.do_module_rehash = 0;
			sendto_realops("Modules reloaded.");
		}
#endif
	}
}

int main(int argc, char *argv[])
{
	int i = 0;

	if (!geteuid()) {
		fprintf(stderr, "ERROR: never run rageircd as root."LF);
		return -1;
	}

	init_core();
	init_vmem();
	init_internal();
	init_counter();
	init_sbuf();
	init_signal();

	build_version();

	memset((char *)&me, '\0', sizeof(me));
	myargv = argv;
	umask(077);
	parse_args(argc, argv);

	if ((i = check_pid()) > 0) {
		switch (signal_type) {
			case 1:
				fprintf(stderr, "rageircd: rehashing config file"LF);
				kill(i, SIGHUP);
				exit(EXIT_SUCCESS);
			case 2:
				fprintf(stderr, "rageircd: stopping server"LF);
				kill(i, SIGTERM);
				exit(EXIT_SUCCESS);
#ifndef STATIC_MODULES
			case 3:
				fprintf(stderr, "rageircd: rehashing modules"LF);
				kill(i, SIGUSR1);
				exit(EXIT_SUCCESS);
#endif
			default:
				fprintf(stderr, "ERROR: rageircd already running under process id %d."LF, i);
				exit(EXIT_FAILURE);
		}
	}
	else if (signal_type) {
		fprintf(stderr, "ERROR: rageircd not currently running"LF);
		exit(EXIT_FAILURE);
	}

#ifdef HAVE_SETLINEBUF
	setlinebuf(stderr);
#endif

	fd_init();

#ifdef USE_OPENSSL
	if (init_dh() == -1) {
		fprintf(stderr, "ERROR: failed to load SSL encryption layer."LF);
		exit(EXIT_FAILURE);
	}
#endif

	/* We chdir to the configuration directory early on */
	if (chdir(CONF_DIR) == -1) {
		fprintf(stderr, "ERROR: cannot change to conf directory: %s\n"
						"       CONF_DIR: %s", strerror(errno), CONF_DIR);
		exit(EXIT_FAILURE);
	}

	set_time();
	init_hash_tables();
	init_blockheap(0);
	dlink_init();
	init_blockheap(1);
	init_ip_hash();
	init_userban();
	hook_init();
	init_clients();
	init_user_mode();
	init_chan_mode();
	init_channels();
	init_channelbans();
	init_link();
	connauth_init();
#ifdef USE_THROTTLE
	init_throttle();
#endif
#ifdef FLUD
	init_fludbot();
#endif
#ifdef STATIC_MODULES
	init_whowas();
#endif
	init_command_table();
	init_modules();
	init_message_files();
	init_stats();

	init_conf();

#ifdef USE_OPENSSL
	if (!Internal.conf_errors) {
		char *ssl_error = init_ssl();
		if (ssl_error != NULL) {
			fprintf(stderr, "ERROR: %s"LF, ssl_error);
			Internal.conf_errors++;
		}
	}
#endif
	if (!Internal.conf_errors) {
		set_time();
		init_statsfile();
	}

	if (Internal.conf_errors) {
		fprintf(stderr, "Encountered %d fatal errors, terminating program."LF, Internal.conf_errors);
		exit(EXIT_FAILURE);
	}

#ifndef STATIC_MODULES
	load_all_modules();
	if (!check_core_commands()) {
		exit(EXIT_FAILURE);
	}
#endif

	fprintf(stderr, "rageircd: initialisation complete, attempting to boot."LF);

	set_time();
	init_maximum_fds();
	build_isupport();

	strncpyzt(me.name, ServerInfo->name, sizeof(me.name));
	strncpyzt(me.info, ServerInfo->desc, sizeof(me.info));

	me.localClient = make_localclient();
	me.localClient->fd = -1;
	me.hopcount = 0;
	me.next = NULL;
	me.user = NULL;
	me.from = &me;
	SetMe(&me);
	make_server(&me);
	me.serv->up = me.name;
	me.lasttime = me.since = me.firsttime = timeofday;
	add_to_client_hash_table(me.name, &me);

	if (!init_identity(ServerInfo->identity)) {
		fprintf(stderr, "ERROR: failed to initialise identity engine."LF);
		exit(EXIT_FAILURE);
	}

	if (!Internal.run_in_terminal) {
		init_daemon();
		init_stdio();
	}

	engine_preinit();
	engine_init();

	if (!init_resolver()) {
		ircdlog(LOG_ERROR, "FATAL: init_resolver() failed to load nameserver from %s (%s)",
			RESOLVER_FILE, strerror(errno));
		exit(EXIT_FAILURE);
	}

	write_pidfile();
	open_listeners();
	init_masking_keys(); /* Masking keys cannot be rehashed. */

	/* Setup some basic events */
	add_event("calculate_bandwidth", calculate_bandwidth, NULL, 2, 1);
	
	set_time();
	
	/* And we're ready... */
	ircd_running = 1;

#ifdef USE_ZLIB
	Debug((DEBUG_DEBUG, "Server using zlib %s", zlibVersion()));
#endif
#ifdef USE_OPENSSL
	Debug((DEBUG_DEBUG, "Server using %s", OPENSSL_VERSION_TEXT));
#endif
	Debug((DEBUG_DEBUG, "Server ready, version: %s revision: %s compile: %s)",
		ircd_version, revision, compile_number));
	
	/* Try connecting to something, then begin I/O */
	try_connections(&me);
	io_loop();

	return 0;
}

void ircdlog(unsigned int log_flag, char *pattern, ...)
{
	FILE *fd;
	va_list vl;
	char buffer[1024], *current_time, *file = NULL;
	ConfigOption *opt;
	int default_log = 0;

	if (ServerInfo == NULL) {
		default_log = 1;
	}
	else if (!(ServerInfo->logs & log_flag)) {
		return;
	}

	current_time = myctime(timeofday);
	va_start(vl, pattern);
	ircvsprintf(buffer, pattern, vl);
	va_end(vl);

	if (default_log) {
		file = LOGFILE_DEFAULT;
	}
	else {
		for (opt = server_log_files; opt->token != NULL; opt++) {
			if (opt->flag == log_flag) {
				file = opt->token;
				break;
			}
		}
	}
	if (file == NULL) {
		return;
	}

	if ((fd = fopen(file, "a")) == NULL) {
		return;
	}

	fprintf(fd, "[%s] %s"LF, current_time, buffer);
#ifdef DEBUGMODE
	if (log_flag == LOG_ERROR) {
		sendto_realops_lev(DEBUG_LEV, "[error log] %s", buffer);
	}
#endif

	fclose(fd);
}

#ifdef USE_ASSERT
void fatal(char *file, int line, char *statement)
{
	Debug((DEBUG_ERROR, "FATAL ERROR: false statement at %s:%d: (%s)", file, line, statement));
	ircdlog(LOG_ERROR, "FATAL ERROR: false statement at %s:%d: (%s)", file, line, statement);

	abort();
}
#endif
