/*
 * RageIRCd: an advanced Internet Relay Chat daemon (ircd).
 * (C) 2000-2005 the RageIRCd Development Team, all rights reserved.
 *
 * This software is free, licensed under the General Public License.
 * Please refer to doc/LICENSE and doc/README for further details.
 *
 * $Id: flud.c,v 1.24.2.1 2004/12/07 03:05:11 pneumatus Exp $
 */

#include "config.h"
#include "struct.h"
#include "common.h"
#include "sys.h"
#include "numeric.h"
#include "msg.h"
#include "channel.h"
#include "h.h"
#include "blalloc.h"
#include "memory.h"
#include "xmode.h"

#ifdef FLUD

BlockHeap *fludbot_heap = NULL;

int flud_num = FLUD_NUM;
int flud_time = FLUD_TIME;
int flud_block = FLUD_BLOCK;

struct fludbot *remove_fluder_reference(struct fludbot **, aClient*);
SLink *remove_fludee_reference(SLink **, void *);
void free_fluders(aClient *, aChannel *);
void free_fludees(aClient *);

void init_fludbot()
{
	fludbot_heap = BlockHeapCreate(sizeof(struct fludbot), CLIENT_HEAP_SIZE);
}

struct fludbot *remove_fluder_reference(struct fludbot **fluders, aClient *fluder)
{
	struct fludbot *current, *prev = NULL, *next = NULL;

	for (current = *fluders; current; current = next) {
		next = current->next;
		if (current->fluder != fluder) {
			prev = current;
			continue;
		}

		if (prev != NULL) {
			prev->next = current->next;
		}
		else {
			*fluders = current->next;
		}
		BlockHeapFree(fludbot_heap, current);
	}
        return *fluders;
}

SLink *remove_fludee_reference(SLink **fludees, void *fludee)
{
	SLink *current, *prev = NULL, *next = NULL;

	for (current = *fludees; current; current = next) {
		next = current->next;
		if (current->value.cptr != (aClient *)fludee) {
			prev = current;
			continue;
		}

		if (prev != NULL) {
			prev->next = current->next;
		}
		else {
			*fludees = current->next;
		}
		BlockHeapFree(slink_heap, current);
	}
	return *fludees;
}

int check_for_flud(aClient *fluder, aClient *cptr, aChannel *chptr)
{
	time_t now;
	struct fludbot *current, *prev = NULL, *next = NULL;
	int blocking, count = 0, found = 0;
	SLink *newfludee;

	if (!flud_block || (chptr == NULL && cptr == NULL)) {
		return 0;
	}

	if (cptr != NULL) {
		if (cptr->localUser == NULL) {
			return 0;
		}
		ASSERT(MyConnect(cptr));
	}

	time(&now);
	if (!(blocking = cptr ? cptr->localUser->fludblock : chptr->fludblock > now - flud_block)) {
		for (current = cptr ? cptr->localUser->fluders : chptr->fluders; current; current = next) {
			next = current->next;
			if (current->last_msg >= (now - flud_time)) {
				prev = current;
				continue;
			}

			remove_fludee_reference(&current->fluder->user->fludees, cptr ? (void *)cptr : (void *)chptr);
			if (prev != NULL) {
				prev->next = current->next;
			}
			else if (cptr != NULL) {
				cptr->localUser->fluders = current->next;
			}
			else {
				chptr->fluders = current->next;
			}
			BlockHeapFree(fludbot_heap, current);
		}
	}

	count = found = 0;
	for (current = cptr ? cptr->localUser->fluders : chptr->fluders; current; current = current->next) {
		if (current->fluder == fluder) {
			current->last_msg = now;
			current->count++;
			found = 1;
		}
		if (current->first_msg < now - flud_time) {
			count++;
		}
		else {
			count += current->count;
		}
	}
	if (!found) {
		current = (struct fludbot *)BlockHeapAlloc(fludbot_heap);
		current->fluder = fluder;
		current->count = 1;
		current->first_msg = current->last_msg = now;
		current->next = cptr ? cptr->localUser->fluders : chptr->fluders;
		if (cptr != NULL) {
			cptr->localUser->fluders = current;
		}
		else {
			chptr->fluders = current;
		}
		count++;

		newfludee = (SLink *)BlockHeapAlloc(slink_heap);
		newfludee->flags = cptr ? 0 : 1;

		if (cptr != NULL) {
			newfludee->value.cptr = cptr;
		}
		else {
			newfludee->value.chptr = chptr;
		}

		newfludee->next = fluder->user->fludees;
		fluder->user->fludees = newfludee;

		if (blocking) {
			sendto_realops_lev(FLOOD_LEV, "Flooder %s (%s@%s) on %s target: %s", fluder->name,
				fluder->username, MaskedHost(fluder),
				fluder->user->server, cptr ? cptr->name : chptr->chname);
		}
	}

	if (!blocking && (found > flud_num)) {
		blocking = 1;
		ircstp->is_flud++;
		if (cptr != NULL) {
			send_me_notice(cptr, ":*** Notice -- Server flood protection for %s activated.",
				chptr->chname);
		}
		else {
			sendto_channel_local_msg_butone(NULL, &me, chptr, ALL_MEMBERS, &CMD_NOTICE,
				"%s :*** Notice -- Server flood protection for %s activated.",
				chptr->chname, chptr->chname);
		}
		for (current = cptr ? cptr->localUser->fluders : chptr->fluders; current; current = current->next) {
			sendto_realops_lev(FLOOD_LEV, "Flooder %s (%s@%s) on %s target: %s", current->fluder->name,
				current->fluder->username, MaskedHost(current->fluder),
				current->fluder->user->server, cptr ? cptr->name : chptr->chname);
		}
	}
	if (blocking) {
		if (cptr != NULL) {
			cptr->localUser->fludblock = now;
		}
		else {
			chptr->fludblock = now;
		}
	}
	return blocking;
}

void free_fluders(aClient *cptr, aChannel *chptr)
{
	struct fludbot *fluders, *next = NULL;

	if (cptr == NULL && chptr == NULL) {
		return;
	}
	if (cptr != NULL) {
		if (cptr->localUser == NULL) {
			return;
		}
		ASSERT(MyConnect(cptr));
	}
	for (fluders = cptr ? cptr->localUser->fluders : chptr->fluders; fluders; fluders = next) {
		next = fluders->next;
		remove_fludee_reference(&fluders->fluder->user->fludees, cptr ? (void *)cptr : (void *)chptr);
		BlockHeapFree(fludbot_heap, fluders);
	}
}

void free_fludees(aClient *cptr)
{
	SLink *fludees, *next = NULL;

	if (cptr == NULL || cptr->user == NULL) {
		return;
	}
	for (fludees = cptr->user->fludees; fludees; fludees = next) {
		next = fludees->next;
		if (fludees->flags) {
			remove_fluder_reference(&fludees->value.chptr->fluders, cptr);
		}
		else {
			ASSERT(MyConnect(fludees->value.cptr));
			remove_fluder_reference(&fludees->value.cptr->localUser->fluders, cptr);
		}
		BlockHeapFree(slink_heap, fludees);
	}
}

#endif
