/*
 * RageIRCd: an advanced Internet Relay Chat daemon (ircd).
 * (C) 2000-2005 the RageIRCd Development Team, all rights reserved.
 *
 * This software is free, licensed under the General Public License.
 * Please refer to doc/LICENSE and doc/README for further details.
 *
 * $Id: engine_devpoll.c,v 1.5.2.1 2004/12/07 03:05:10 pneumatus Exp $
 */

#include "setup.h"
#include "struct.h"
#include "common.h"
#include "sys.h"
#include "h.h"
#include "fd.h"

#ifdef USE_DEVPOLL

#include <sys/devpoll.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#ifndef POLLRDNORM
#define POLLRDNORM POLLIN
#endif
#ifndef POLLWRNORM
#define POLLWRNORM POLLOUT
#endif

#define POLL_LENGTH HARD_FDLIMIT

static int devpoll_fd;
static short fdmask[POLL_LENGTH];

static void devpoll_write_update(int, int);
static void devpoll_update_events(int, short, FDCB *);

static void devpoll_write_update(int fd, int events)
{
	struct pollfd poll_fds[1];
	int retval;

	poll_fds[0].revents = 0;
	poll_fds[0].fd = fd;
	poll_fds[0].events = events;

	if ((retval = write(devpoll_fd, &poll_fds[0], sizeof(struct pollfd))) != sizeof(struct pollfd)) {
		ircdlog(LOG_ERROR, "devpoll_write_update() devpoll_fd write failed %d: %s",
			errno, strerror(errno));
	}
}

static void devpoll_update_events(int fd, unsigned short filter, FDCB *cb)
{
	int update_required = 0, current_mask = fdmask[fd];
	FDCB *curcb;

	fdmask[fd] = 0;

	ASSERT((filter & (FDEV_READ|FDEV_WRITE)) != 0);

	if (filter == FDEV_READ) {
		curcb = TheCallback(fd_table[fd].read);
		if (curcb != NULL) {
			fdmask[fd] |= POLLRDNORM;
		}
		else {
			fdmask[fd] &= ~POLLRDNORM;
		}
		if (TheCallback(fd_table[fd].write) != NULL) {
			fdmask[fd] |= POLLWRNORM;
		}
	}
	else if (filter == FDEV_WRITE) {
		curcb = TheCallback(fd_table[fd].write);
		if (curcb != NULL) {
			fdmask[fd] |= POLLWRNORM;
		}
		else {
			fdmask[fd] &= ~POLLWRNORM;
		}
		if (TheCallback(fd_table[fd].read) != NULL) {
			fdmask[fd] |= POLLRDNORM;
		}
	}

	if (((curcb == NULL) && (cb != NULL)) || ((curcb != NULL) && (cb == NULL))) {
		update_required++;
	}
	if (current_mask != fdmask[fd]) {
		update_required++;
	}

	if (update_required) {
		devpoll_write_update(fd, POLLREMOVE);

		if (fdmask[fd]) {
			devpoll_write_update(fd, fdmask[fd]);
		}
	}
}

void engine_init()
{
	if ((devpoll_fd = open("/dev/poll", O_RDWR)) == -1) {
		ircdlog(LOG_ERROR, "init_engine() failed: couldn't open /dev/poll: %d (%s)",
			errno, strerror(errno));
		exit(EXIT_FAILURE);
	}
	memset(&fdmask, '\0', sizeof(fdmask));
}

void engine_set_call(int fd, unsigned short flags, FDCB *cb, void *data, time_t timeout)
{
	fd_entry *fde = &fd_table[fd];

	ASSERT(fde->fd > FD_UNUSED);
	ASSERT(fde->open);

	if (flags & FDEV_READ) {
		devpoll_update_events(fd, FDEV_READ, cb);
		SetCallback(fde->read, cb, data);
	}
	if (flags & FDEV_WRITE) {
		devpoll_update_events(fd, FDEV_WRITE, h);
		SetCallback(fde->write, cb, data);
	}
	if (timeout > 0) {
		fde->timeout_time = timeofday + (timeout / 1000);
	}
}

void engine_do_netio(time_t delay)
{
	int cnt, i, fd;
	struct pollfd poll_fds[POLL_LENGTH];
	struct dvpoll dopoll;
	fd_entry *fde;
	FDCB *cb;
	void *data;

	do {
		for (;;) {
			dopoll.dp_timeout = delay;
			dopoll.dp_nfds = POLL_LENGTH;
			dopoll.dp_fds = &poll_fds[0];

			if ((cnt = ioctl(devpoll_fd, DP_POLL, &dopoll)) >= 0) {
				break;
			}
			if (engine_ignore_errno(errno)) {
				continue;
			}

			set_time();
			return;
		}

		set_time();

		if (!cnt) {
			continue;
		}

		for (i = 0; i < cnt; i++) {
			fd = dopoll.dp_fds[i].fd;
			fde = &fd_table[fd];

			if ((dopoll.dp_fds[i].revents & (POLLRDNORM|POLLIN|POLLHUP|POLLERR))
			  && (dopoll.dp_fds[i].events & (POLLRDNORM|POLLIN))) {
				GetCallback(fde->read, cb, data);
				ClrCallback(fde->read);

				if (cb != NULL) {
					cb(fd, data, ENGINE_OK);
					devpoll_update_events(fd, FDEV_READ, TheCallback(fde->read));
				}
			}
			if ((dopoll.dp_fds[i].revents & (POLLWRNORM|POLLOUT|POLLHUP|POLLERR))
			  && (dopoll.dp_fds[i].events & (POLLWRNORM|POLLOUT))) {
				GetCallback(fde->write, cb, data);
				ClrCallback(fde->write);

				if (cb != NULL) {
					cb(fd, data, ENGINE_OK);
					devpoll_update_events(fd, FDEV_WRITE, TheCallback(fde->write));
				}
			}
			if (pfd->revents & POLLNVAL) {
				ircdlog(LOG_ERROR, "engine_do_netio() encountered invalid revents for fd %d", fd);
			}
		}
		break;
	} while (0);
}

#endif
