/*
 * RageIRCd: an advanced Internet Relay Chat daemon (ircd).
 * (C) 2000-2005 the RageIRCd Development Team, all rights reserved.
 *
 * This software is free, licensed under the General Public License.
 * Please refer to doc/LICENSE and doc/README for further details.
 *
 * $Id: dh.c,v 1.23.2.2 2004/12/18 16:41:28 amcwilliam Exp $
 */

#include "setup.h"

#ifdef USE_OPENSSL

#include "struct.h"
#include "h.h"
#include "common.h"
#include "memory.h"
#include "ssl.h"

static unsigned int dh_gen_1024;
static unsigned char dh_prime_1024[];
static char *hex_to_string[256];
static BIGNUM *ircd_prime;
static BIGNUM *ircd_generator;

#ifdef __OpenBSD__
#define RAND_SRC "/dev/arandom"
#else
#define RAND_SRC "/dev/random"
#endif

static int verify_is_hex(char *string)
{
	int l = strlen(string);
	char tmp[4] = { '\0', '\0', '\0', '\0' };
	int tmpidx = 0;

	if (l & 0x01) {
		l++;
		tmp[tmpidx++] = '0';
	}
	while (*string != '\0') {
		tmp[tmpidx++] = *string++;
		if (tmpidx == 2) {
			char *eptr;
			unsigned char x;

			tmpidx = 0;
			x = strtol(tmp, &eptr, 16);
			if (*eptr != '\0') {
				return 0;
			}
		}
	}
	return 1;
}

int dh_hexstr_to_raw(char *string, unsigned char *hexout, int *hexlen)
{
	int l = strlen(string);
	char tmp[3] = { '\0', '\0', '\0' };
	int tmpidx = 0, hexidx = 0;

	if (l & 0x01) {
		l++;
		tmp[tmpidx++] = '0';
	}
	while (*string != '\0') {
		tmp[tmpidx++] = *string++;
		if (tmpidx == 2) {
			char *eptr;
			unsigned char x;

			tmpidx = 0;
			x = strtol(tmp, &eptr, 16);

			if (*eptr != '\0') {
				return 0;
			}

			hexout[hexidx++] = (unsigned char)x;
		}
	}
	*hexlen = hexidx;
	return 1;
}

static inline void entropy_error()
{
	printf("Couldn't generate entropy from %s: %s"LF, RAND_SRC, strerror(errno));
	printf("ircd needs a %d byte random seed for encryption. You can place suitably\n", RAND_BYTES);
	printf("random data into a file called %s along side %s.\n", ENTROPY_FILE, CONFIG_FILE);
}

static int make_entropy()
{
	char randbuf[RAND_BYTES * 4];
	FILE *fd;
	int i, cv;

	printf("No random state, generating entropy from %s."LF, RAND_SRC);

	if ((fd = fopen(RAND_SRC, "r")) == NULL) {
		entropy_error();
		return 0;
	}
	
	for (i = 0; i < (RAND_BYTES * 4); i++) {
		if ((cv = fgetc(fd)) != EOF) {
			randbuf[i] = cv;
			continue;
		}
		
		if (ferror(fd)) {
			entropy_error();
			fclose(fd);
			return 0;
		}
		
		clearerr(fd);
		usleep(100);
		i--; /* Try again */
	}
	
	fclose(fd);

	if ((fd = fopen(ENTROPY_FILE, "w")) == NULL) {
		printf("Couldn't open %s for writing: %s"LF, ENTROPY_FILE, strerror(errno));
		return 0;
	}

	fwrite(randbuf, RAND_BYTES * 4, 1, fd);
	fclose(fd);

	RAND_load_file(ENTROPY_FILE, -1);

	return 1;
}

static int init_random()
{
	int ret;
	time_t now;

	if ((ret = RAND_load_file(ENTROPY_FILE, -1)) <= 0) {
		if (!make_entropy()) {
			return -1;
		}
	}

	now = timeofday;
	RAND_seed(&now, 4);
	RAND_write_file(ENTROPY_FILE);

	return 0;
}

static void create_prime()
{
	char buf[PRIME_BYTES_HEX];
	int i, bufpos = 0;

	for (i = 0; i < PRIME_BYTES; i++) {
		char *x = hex_to_string[dh_prime_1024[i]];
		while (*x != '\0') {
			buf[bufpos++] = *x++;
		}
	}
	buf[bufpos] = '\0';

	ircd_prime = NULL;
	BN_hex2bn(&ircd_prime, buf);
	ircd_generator = BN_new();
	BN_set_word(ircd_generator, dh_gen_1024);
}

int init_dh()
{
	ERR_load_crypto_strings();
	create_prime();
	return init_random();
}

int dh_generate_shared(void *session, char *public_key)
{
	BIGNUM *tmp = NULL;
	int len;
	SessionInfo *si = (SessionInfo *)session;

	if (!verify_is_hex(public_key) || si == NULL || si->session_shared != NULL) {
		return 0;
	}

	BN_hex2bn(&tmp, public_key);

	if (tmp == NULL) {
		return 0;
	}

	si->session_shared_length = DH_size(si->dh);
	si->session_shared = (char *)MyMalloc(si->session_shared_length);
	len = DH_compute_key(si->session_shared, tmp, si->dh);
	BN_free(tmp);

	if (len < 0) {
		return 0;
	}

	si->session_shared_length = len;
	return 1;
}

void *dh_start_session()
{
	SessionInfo *si = (SessionInfo *)MyMalloc(sizeof(SessionInfo));

	si->dh = DH_new();
	si->dh->p = BN_dup(ircd_prime);
	si->dh->g = BN_dup(ircd_generator);

	if (!DH_generate_key(si->dh)) {
		DH_free(si->dh);
		MyFree(si);
		return NULL;
	}

	return (void *)si;
}

void dh_end_session(void *session)
{
	SessionInfo *si = (SessionInfo *)session;

	if (si->dh != NULL) {
		DH_free(si->dh);
		si->dh = NULL;
	}
	if (si->session_shared != NULL) {
		memset(si->session_shared, '\0', si->session_shared_length);
		MyFree(si->session_shared);
		si->session_shared = NULL;
	}

	MyFree(si);
}

char *dh_get_s_public(char *buf, int maxlen, void *session)
{
	SessionInfo *si = (SessionInfo *)session;
	char *tmp;

	if (si == NULL || si->dh == NULL || si->dh->pub_key == NULL) {
		return NULL;
	}
	if ((tmp = BN_bn2hex(si->dh->pub_key)) == NULL) {
		return NULL;
	}
	if (strlen(tmp) + 1 > maxlen) {
		OPENSSL_free(tmp);
		return NULL;
	}
	strcpy(buf, tmp);
	OPENSSL_free(tmp);
	return buf;
}

int dh_get_s_shared(char *buf, int *maxlen, void *session)
{
	SessionInfo *si = (SessionInfo *)session;

	if (si == NULL || si->session_shared == NULL || *maxlen < si->session_shared_length) {
		return 0;
	}

	*maxlen = si->session_shared_length;
	memcpy(buf, si->session_shared, *maxlen);
	return 1;
}

static char *hex_to_string[256] = {
	"00", "01", "02", "03", "04", "05", "06", "07", 
	"08", "09", "0a", "0b", "0c", "0d", "0e", "0f", 
	"10", "11", "12", "13", "14", "15", "16", "17", 
	"18", "19", "1a", "1b", "1c", "1d", "1e", "1f", 
	"20", "21", "22", "23", "24", "25", "26", "27", 
	"28", "29", "2a", "2b", "2c", "2d", "2e", "2f", 
	"30", "31", "32", "33", "34", "35", "36", "37", 
	"38", "39", "3a", "3b", "3c", "3d", "3e", "3f", 
	"40", "41", "42", "43", "44", "45", "46", "47", 
	"48", "49", "4a", "4b", "4c", "4d", "4e", "4f", 
	"50", "51", "52", "53", "54", "55", "56", "57", 
	"58", "59", "5a", "5b", "5c", "5d", "5e", "5f", 
	"60", "61", "62", "63", "64", "65", "66", "67", 
	"68", "69", "6a", "6b", "6c", "6d", "6e", "6f", 
	"70", "71", "72", "73", "74", "75", "76", "77", 
	"78", "79", "7a", "7b", "7c", "7d", "7e", "7f", 
	"80", "81", "82", "83", "84", "85", "86", "87", 
	"88", "89", "8a", "8b", "8c", "8d", "8e", "8f", 
	"90", "91", "92", "93", "94", "95", "96", "97", 
	"98", "99", "9a", "9b", "9c", "9d", "9e", "9f", 
	"a0", "a1", "a2", "a3", "a4", "a5", "a6", "a7", 
	"a8", "a9", "aa", "ab", "ac", "ad", "ae", "af", 
	"b0", "b1", "b2", "b3", "b4", "b5", "b6", "b7", 
	"b8", "b9", "ba", "bb", "bc", "bd", "be", "bf", 
	"c0", "c1", "c2", "c3", "c4", "c5", "c6", "c7", 
	"c8", "c9", "ca", "cb", "cc", "cd", "ce", "cf", 
	"d0", "d1", "d2", "d3", "d4", "d5", "d6", "d7", 
	"d8", "d9", "da", "db", "dc", "dd", "de", "df", 
	"e0", "e1", "e2", "e3", "e4", "e5", "e6", "e7", 
	"e8", "e9", "ea", "eb", "ec", "ed", "ee", "ef", 
	"f0", "f1", "f2", "f3", "f4", "f5", "f6", "f7", 
	"f8", "f9", "fa", "fb", "fc", "fd", "fe", "ff"
};

static unsigned int dh_gen_1024 = 2;
static unsigned char dh_prime_1024[] = {
	0xF4, 0x88, 0xFD, 0x58, 0x4E, 0x49, 0xDB, 0xCD,
	0x20, 0xB4, 0x9D, 0xE4, 0x91, 0x07, 0x36, 0x6B,
	0x33, 0x6C, 0x38, 0x0D, 0x45, 0x1D, 0x0F, 0x7C,
	0x88, 0xB3, 0x1C, 0x7C, 0x5B, 0x2D, 0x8E, 0xF6,
	0xF3, 0xC9, 0x23, 0xC0, 0x43, 0xF0, 0xA5, 0x5B,
	0x18, 0x8D, 0x8E, 0xBB, 0x55, 0x8C, 0xB8, 0x5D,
	0x38, 0xD3, 0x34, 0xFD, 0x7C, 0x17, 0x57, 0x43,
	0xA3, 0x1D, 0x18, 0x6C, 0xDE, 0x33, 0x21, 0x2C,
	0xB5, 0x2A, 0xFF, 0x3C, 0xE1, 0xB1, 0x29, 0x40,
	0x18, 0x11, 0x8D, 0x7C, 0x84, 0xA7, 0x0A, 0x72,
	0xD6, 0x86, 0xC4, 0x03, 0x19, 0xC8, 0x07, 0x29,
	0x7A, 0xCA, 0x95, 0x0C, 0xD9, 0x96, 0x9F, 0xAB,
	0xD0, 0x0A, 0x50, 0x9B, 0x02, 0x46, 0xD3, 0x08,
	0x3D, 0x66, 0xA4, 0x5D, 0x41, 0x9F, 0x9C, 0x7C,
	0xBD, 0x89, 0x4B, 0x22, 0x19, 0x26, 0xBA, 0xAB,
	0xA2, 0x5E, 0xC3, 0x55, 0xE9, 0x2F, 0x78, 0xC7
};

#endif
