/*
 * RageIRCd: an advanced Internet Relay Chat daemon (ircd).
 * (C) 2000-2005 the RageIRCd Development Team, all rights reserved.
 *
 * This software is free, licensed under the General Public License.
 * Please refer to doc/LICENSE and doc/README for further details.
 *
 * $Id: m_sjoin.c,v 1.77.2.2 2005/01/15 23:53:30 amcwilliam Exp $
 */

#include "config.h"
#include "struct.h"
#include "common.h"
#include "sys.h"
#include "numeric.h"
#include "msg.h"
#include "channel.h"
#include "h.h"
#include "memory.h"
#include "modules.h"
#include "conf2.h"
#include <time.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>

Module MOD_HEADER(m_sjoin) = {
	"m_sjoin",
	"SJOIN and RESYNCH protocol",
	6, "$Revision: 1.77.2.2 $"
};

int MOD_LOAD(m_sjoin)()
{
	if (register_command(&MOD_HEADER(m_sjoin), &CMD_SJOIN, m_sjoin) == NULL) {
		return MOD_FAILURE;
	}
	if (register_command(&MOD_HEADER(m_sjoin), &CMD_RESYNCH, m_resynch) == NULL) {
		return MOD_FAILURE;
	}
	return MOD_SUCCESS;
}

int MOD_UNLOAD(m_sjoin)()
{
	return MOD_SUCCESS;
}

/* These are to make life a little easier during mode synching */
#define ADDED(xx)		((mode.mode & (xx)) && !(oldmode->mode & (xx)))
#define REMOVED(xx)		(!(mode.mode & (xx)) && (oldmode->mode & (xx)))
#define BOTHSET(xx)		((mode.mode & (xx)) && (oldmode->mode & (xx)))

/* If the sign is new, append it to modebuf and set change accordingly */
#define ADD_SIGN(xx)		if (change != xx) { ADD_MODE((xx == XMODE_ADD) ? '+' : '-'); change = xx; }

/* If our parabuf is full, send it */
#define CHECK_PCOUNTC		if (pcount >= MAXMODEPARAMS) { sjoin_sendit(sptr, chptr); pcount = change = 0; }

static inline void sjoin_sendit(aClient *sptr, aChannel *chptr)
{
	/* Send MODE command to local channel */
	sendto_channel_local_msg_butone(NULL, sptr, chptr, ALL_MEMBERS, &CMD_MODE,
		"%s %s %s", chptr->chname, modebuf, parabuf);
	midx = pidx = idpidx = 0;
	modebuf[0] = parabuf[0] = idparabuf[0] = '\0';
}

/*
 * m_sjoin
 *	parv[0] = sender prefix;
 *	parv[1] = timestamp
 *	parv[2] = channel name
 *
 *	if (parc == 3) - Client-Server SJOIN
 *
 *	if (parc > 3) - Server-Server SSJ3
 *		parv[3] = modes + n parameters
 *		parv[4 + n] = nicklist
 */
int m_sjoin(aClient *cptr, aClient *sptr, int parc, char *parv[])
{
	aChannel *chptr;
	long newts, oldts, tstosend;
	int isnew = 0, created = 0, buffer_it;

	xMode *t;
	chanMode *m;
	int mindex = 0, args = 0, pcount = 0, people = 0;
	short change = XMODE_NONE;

	char *s, *s0, *p;
	static Mode mode, *oldmode;
	int keepours = 1, keeptheirs = 1;
	char saved_modebuf[REALMODEBUFLEN], saved_parabuf[REALMODEBUFLEN];

	chanMember *cm = NULL;
	aClient *acptr;
	long flags;

	char *pf, prefix[5], nick[NICKLEN + 5];
	static char sjbuf[BUFSIZE], idsjbuf[BUFSIZE];
	int sjidx, idsjidx;

	if (MyClient(sptr) || (parc < 5 && IsServer(sptr)) || (*parv[2] != '#')) {
		return 0;
	}

	isnew = ChannelExists(parv[2]) ? 0 : 1;
	chptr = get_channel(sptr, parv[2], CREATE, &created);

	newts = get_ts(parv[1]);
	oldts = chptr->channelts;

	if (parc == 3) {
		if (isnew) {
			chptr->channelts = tstosend = newts;
		}
		else if (!newts || !oldts) {
			chptr->channelts = tstosend = 0;
		}
		else if (newts == oldts) {
			tstosend = oldts;
		}
		else if (oldts > newts) {
			chptr->channelts = tstosend = newts;
		}
		else {
			tstosend = oldts;
		}

		if (!IsMember(sptr, chptr)) {
			add_user_to_channel(chptr, sptr, 0);
			sendto_channel_local_msg_butone(NULL, sptr, chptr, ALL_MEMBERS,
				&CMD_JOIN, "%s", chptr->chname);
		}

		sendto_serv_msg_butone(cptr, sptr, &CMD_SJOIN, "%ld %s", tstosend,
			chptr->chname);

		if (created) {
			sendto_realops_lev(DEBUG_LEV, "Requesting resynch of %s from %s "
				"(created by %s!%s@%s)", chptr->chname, cptr->name,
				sptr->name, sptr->username, MaskedHost(sptr));
			sendto_one_client_nopostfix(cptr, &me, &CMD_RESYNCH, "%s",
				chptr->chname);
		}
		return 0;
	}

	/* Build up an idea of what modes *they* have. First character of the
	 * mode string is *always* a +, so make sure it's skipped!
	 */
	/* Not anymore -- modebuf of "0" means this isn't our first SJOIN
	 * from the link. This also means we *should* be synched already,
	 * and so we can skip modestuffs...
	 */
	memset((char *)&mode, '\0', sizeof(mode));
	for (s = parv[3]; *s != '\0'; s++) {
		/* Firs things first: if the very first character is 0, we break
		 * out instantly (read above)
		 */
		if (*s == *parv[3] && (*s == '0')) {
			break;
		}
		if (*s == '+' || (*s == '-')) {
			/* get_chan_mode_byflag() doesn't recognise + or -, so
			 * we need to make sure the string prefix is skipped, as
			 * we now parse this from the very beginning (read above)
			 */
			continue;
		}
		if ((mindex = chanmodes->map[(unsigned char)*s]) == -1) {
			/* Once Server-Server CHANMODES capabilty is in place, this
			 * shouldn't happen, as links with unknown chan modes wont
			 * be allowed to get this far (in theory ;)
			 */
			/* CHANMODES and USERMODES capability has been postponed
			 * until beta-3+. It's not critical, provided admins don't
			 * get cocky. >;)
			 */
			/* Annoying bloody message...
			sendto_realops("WARNING! Ignoring unknown mode flag %c in SJOIN for %s",
				*s, chptr->chname);
			*/
			continue;
		}

		t = &chanmodes->table[mindex];
		m = chanmode_tab[mindex];

		/* Skip over USER and LIST modes */
		if (m->paratype > CPTYP_NORMAL) {
			continue;
		}

		Debug((DEBUG_DEBUG, "sjoin %s: [THEIR MODES] got mode flag %c",
			chptr->chname, t->flag));

		/* Add the mode (can be either nopara, setonly or setunset!) */
		mode.mode |= t->mode;

		if (m->paratype == CPARA_NONE) {
			continue; /* No para modes can finish here */
		}

		ASSERT(m->sjoin_domode_pbuf != NULL); /* Paranoia */

		/* Right, we got past the above statemment, so we need a parameter.
		 * Since SJOIN is now dynamic, we cant presume to know how to set
		 * a mode parameter. So, call the mode's domode function, but
		 * dont buffer it!
		 */
		(*m->sjoin_domode_pbuf)(mindex, change, &mode, parv[4 + args], 0);
		args++;

		if ((args + 4) > (parc - 1)) {
			Debug((DEBUG_DEBUG, "sjoin %s: eaten all my args!", chptr->chname));
			return 0;
		}
	}

	/* We now know what they *want*, let's see what we have */

	oldmode = &chptr->mode;
	if (isnew) {
		/* New, merge (keep all modes) */
		chptr->channelts = tstosend = newts;
	}
	else if (!newts || !oldts) {
		chptr->channelts = tstosend = 0;
	}
	else if (newts == oldts) {
		/* We're equal, merge */
		tstosend = oldts;
	}
	else if (oldts > newts) {
		/* We're newest, keep theirs */
		chptr->channelts = tstosend = newts;
		/*
		Another annoying message...
		send_channel_msg_local(&me, chptr, &CMD_NOTICE, ":*** TS for %s changed "
			"from %lu to %lu", chptr->chname, oldts, newts);
		*/

		Debug((DEBUG_DEBUG, "sjoin %s: keep their modes", chptr->chname));
		keepours = 0;

		kill_ban_list(cptr, chptr, CMODE_BAN);
		kill_ban_list(cptr, chptr, CMODE_EXCEPT);
		kill_ban_list(cptr, chptr, CMODE_INVEX);
	}
	else {
		/* We're oldest, keep ours */
		tstosend = oldts;

		Debug((DEBUG_DEBUG, "sjoin %s: keep our modes", chptr->chname));
		keeptheirs = 0;
	}

	if (!keeptheirs) {
		/* We dont keep theirs -- synch mode to *us* */
		mode = *oldmode;
	}
	else if (keepours) {
		/* We keep our settings */
		/* If we won TS, keeptheirs=0 (THUS we wouldn't get here because of what's above),
		 * so this must be a channel merge, as keeptheirs=1 and keepours=1
		 */
		mode.mode |= oldmode->mode;
		for (mindex = 0; mindex <= chanmodes->highest; mindex++) {
			t = &chanmodes->table[mindex];
			m = chanmode_tab[mindex];

			if ((t->flag == '\0') || (m->paratype != CPTYP_NORMAL)) {
				continue;
			}

			ASSERT(m->sjoin_cmp_para != NULL);
			ASSERT(m->sjoin_synch_para != NULL);

			if ((*m->sjoin_cmp_para)(mindex, oldmode, &mode) == CM_SJCMP_KEEPOLD) {
				Debug((DEBUG_DEBUG, "sjoin %s: [PARA] keeping oldmode para",
					chptr->chname));
				(*m->sjoin_synch_para)(mindex, &mode, oldmode);
			}
		}
	}

	/* Mode is now synched. Now we do the buffering in two stages.
	 *
	 * Stage 1: Delete any old modes REMOVED during synch
	 * Stage 2: Add any:
	 *		- new modes ADDED during synch
	 *		- existing modes with an updated para during synch
	 */

	midx = pidx = idpidx = 0;
	modebuf[0] = parabuf[0] = idparabuf[0] = '\0';

	/* Stage 1: delete any old modes removed during synch
	 *		WHEN: !keepours, keeptheirs, !merge
	 */
	ADD_SIGN(XMODE_DEL)
	for (mindex = 0; !keepours && (mindex <= chanmodes->highest); mindex++) {
		t = &chanmodes->table[mindex];
		m = chanmode_tab[mindex];

		/* Skip over USER and LIST modes */
		if (t->flag == '\0' || (m->paratype > CPTYP_NORMAL) || !REMOVED(t->mode)) {
			continue;
		}

		ADD_SIGN(XMODE_DEL) /* Incase buffers are cleared */
		ADD_MODE(t->flag);

		/* If we need a parameter to unset, call the function */
		if (m->para == CPARA_SETUNSET) {
			ASSERT(m->sjoin_domode_pbuf != NULL);
			(*m->sjoin_domode_pbuf)(mindex, change, &mode, NULL, 1);
			pcount++;
			CHECK_PCOUNTC
		}
	}
	if (midx == 1) {
		/* Buffer ends in +/-, move backwards */
		midx--;
		change = XMODE_NONE;
	}

	/* Stage 2: add any new modes added or changed during synch
	 *		WHEN: !keepours, keeptheirs, merge
	 */
	ADD_SIGN(XMODE_ADD)
	for (mindex = 0; keeptheirs && (mindex <= chanmodes->highest); mindex++) {
		t = &chanmodes->table[mindex];
		m = chanmode_tab[mindex];

		/* Skip over USER and LIST modes */
		if ((t->flag == '\0') || (m->paratype > CPTYP_NORMAL)) {
			continue;
		}

		/* Most common mode is a no-para mode, so just parse them quickly */
		if (m->paratype == CPTYP_NONE) {
			if (ADDED(t->mode)) {
				ADD_SIGN(XMODE_ADD) /* Incase buffers are cleared */
				ADD_MODE(t->flag);
			}
			continue;
		}

		ASSERT(m->sjoin_cmp_para != NULL); /* .. even more paranoia! */
		ASSERT(m->sjoin_synch_para != NULL);

		buffer_it = 0;
		if (ADDED(t->mode) || (BOTHSET(t->mode) && !keepours)) {
			/* Either the mode has been added during synch,
			 * or we both have the mode *BUT* we're not keeping
			 * ours (ie, MERGE). Just accept theirs.
			 */
			buffer_it = 1;
		}
		else if (BOTHSET(t->mode)) {
			/* Mode conflict! Both sides have the mode, and were merging
			 * the two. Battle it out, biggest parameter wins.
			 *
			 *   EQUAL: Both parameters are identical, dont buffer, dont change
			 * KEEPOLD: Our parameter won, dont buffer, synch theirs
			 * KEEPNEW: Their parameter won, buffer it, synch ours
			 */
			/* Sanity: !keepours is actually parsed above, as parv[4+n] is
			 * parsed. This synchs our modes, so in reality all we need
			 * check for is whether it should be buffered to the channel.
			 */
			if ((*m->sjoin_cmp_para)(mindex, oldmode, &mode) == CM_SJCMP_KEEPNEW) {
				buffer_it = 1;
			}
		}

		if (buffer_it) {
			ADD_SIGN(XMODE_ADD)
			ADD_MODE(t->flag);
			
			(*m->sjoin_domode_pbuf)(mindex, change, &mode, NULL, 1);

			pcount++;
			CHECK_PCOUNTC
		}
	}
	if (modebuf[midx - 1] == '+' || modebuf[midx - 1] == '-') {
		/* Buffer ends in +/-, move backwards */
		midx--;
		modebuf[midx] = '\0';
		change = XMODE_NONE;
	}

	chptr->mode = mode; /* Set our new synched modes! */

	/* If we aren't keeping our modes, dont keep our user modes either */
	for (cm = chptr->members; !keepours && (cm != NULL); cm = cm->nextuser) {
		if (cm->flags & CMODE_CHANADMIN) {
			cm->flags &= ~CMODE_CHANADMIN;

			ADD_SIGN(XMODE_DEL)
			ADD_MODE('a');
			ADD_NPARA(cm->cptr->name);

			pcount++;
			CHECK_PCOUNTC
		}
		if (cm->flags & CMODE_CHANOP) {
			cm->flags &= ~CMODE_CHANOP;

			ADD_SIGN(XMODE_DEL)
			ADD_MODE('o');
			ADD_NPARA(cm->cptr->name);

			pcount++;
			CHECK_PCOUNTC
		}
		if (cm->flags & CMODE_HALFOP) {
			cm->flags &= ~CMODE_HALFOP;

			ADD_SIGN(XMODE_DEL)
			ADD_MODE('h');
			ADD_NPARA(cm->cptr->name);

			pcount++;
			CHECK_PCOUNTC
		}
		if (cm->flags & CMODE_VOICE) {
			cm->flags &= ~CMODE_VOICE;

			ADD_SIGN(XMODE_DEL)
			ADD_MODE('v');
			ADD_NPARA(cm->cptr->name);

			pcount++;
			CHECK_PCOUNTC
		}
	}

	/* At this stage, all modes should be synched, and users deopped if
	 * applicable. If we have anything in the buffers, send it along and
	 * prepare for the next wave of SJOIN: user parsing!
	 */
	if (midx) {
		sjoin_sendit(sptr, chptr);
		change = XMODE_NONE;
	}

	/* Save our synched modes and parameters, for propogation later */
	saved_modebuf[0] = saved_parabuf[0] = '\0';
	if (*parv[3] != '0' && keeptheirs) {
		get_chan_modes(sptr, chptr, saved_modebuf, saved_parabuf);
	}
	else {
		saved_modebuf[0] = '0';
		saved_modebuf[1] = '\0';
	}

	sjidx = idsjidx = 0;
	sjbuf[0] = idsjbuf[0] = '\0';

	Debug((DEBUG_DEBUG, "sjoin(%s) (argcount:%d) got user list [%s]", chptr->chname, args, parv[4+args]));

	for (s = s0 = strtoken(&p, parv[4 + args], " "); s != NULL; s = s0 = strtoken(&p, NULL, " ")) {
		/* Parse the nickname prefix, if any, and get the flags if
		 * we aren't ignoring them. Build a prefix buffer, too
		 */
		flags = 0;
		pf = prefix;

		if (*s == '*') {
			flags |= (keeptheirs) ? CMODE_CHANADMIN : 0;
			*pf++ = *s++;
		}
		if (*s == '@') {
			flags |= (keeptheirs) ? CMODE_CHANOP : 0;
			*pf++ = *s++;
		}
		if (*s == '%') {
			flags |= (keeptheirs) ? CMODE_HALFOP : 0;
			*pf++ = *s++;
		}
		if (*s == '+') {
			flags |= (keeptheirs) ? CMODE_VOICE : 0;
			*pf++ = *s++;
		}
		*pf = '\0';

		/* We have now parsed any prefixes. s=nick, s0=@nick */
		if (*s == '\0') {
			Debug((DEBUG_DEBUG, "sjoin %s: blank user with flags %04x",
				chptr->chname, flags));
			continue;
		}
		if (!keeptheirs) {
			*prefix = '\0';
		}

		if ((acptr = find_target_chasing(sptr, s, "SJOIN")) == NULL) {
			Debug((DEBUG_DEBUG, "sjoin %s: unknown user %s", chptr->chname, s));
			continue;
		}
		if (acptr->from != cptr) {
			Debug((DEBUG_DEBUG, "sjoin %s: invalid user %s from %s, wrong direction",
				chptr->chname, acptr->name, cptr->name));
			continue;
		}

		/* Add the user, if they're not already in */
		if (!IsMember(acptr, chptr)) {
			if ((cm = add_user_to_channel(chptr, acptr, flags)) == NULL) {
				Debug((DEBUG_DEBUG, "sjoin %s: no chanMember link for %s",
					chptr->chname, acptr->name));
				continue;
			}
			sendto_channel_local_msg_butone(NULL, acptr, chptr, ALL_MEMBERS,
				&CMD_JOIN, "%s", chptr->chname);
		}

		people++; /* At this stage, they WILL be put in the SJOIN buffers */

		/* No flags? No prefix! */
		if (!flags) {
			ADD_STRING(acptr->name, sjbuf, sjidx);
			ADD_STRING(get_id(acptr), idsjbuf, idsjidx);
		}
		else {
			ircsprintf(nick, "%s%s", prefix, acptr->name);
			ADD_STRING(nick, sjbuf, sjidx);
			ircsprintf(nick, "%s%s", prefix, get_id(acptr));
			ADD_STRING(nick, idsjbuf, idsjidx);
		}

		/* We've got the nickname, calculated prefixes, and added the
		 * client to the SJOIN buffers. Time to set their flags.
		 */

		cm->flags = flags; /* Dont just add, we need to *set* here.. */
		if (flags & CMODE_CHANADMIN) {
			ADD_SIGN(XMODE_ADD)
			ADD_MODE('a');
			ADD_NPARA(acptr->name);

			pcount++;
			CHECK_PCOUNTC
		}
		if (flags & CMODE_CHANOP) {
			ADD_SIGN(XMODE_ADD)
			ADD_MODE('o');
			ADD_NPARA(acptr->name);

			pcount++;
			CHECK_PCOUNTC
		}
		if (flags & CMODE_HALFOP) {
			ADD_SIGN(XMODE_ADD)
			ADD_MODE('h');
			ADD_NPARA(acptr->name);

			pcount++;
			CHECK_PCOUNTC
		}
		if (flags & CMODE_VOICE) {
			ADD_SIGN(XMODE_ADD)
			ADD_MODE('v');
			ADD_NPARA(acptr->name);

			pcount++;
			CHECK_PCOUNTC
		}
	}

	/* Users should now be synched. If there's any data left in the buffers,
	 * send it and prepare for the final step: propogation!
	 */
	if (midx) {
		sjoin_sendit(sptr, chptr);
		change = XMODE_NONE;
	}

	/* This shouldn't happen, but just in case it does. */
	/* WRONG! This can (WILL) happen on nick collisions, if users on a remote
	 * server collide with our users, where we win.
	 */
	if (!people) {
		Debug((DEBUG_DEBUG, "sjoin %s: ERROR: no people?", chptr->chname));
		if (created && !chptr->users) {
			sub1_from_channel(chptr);
		}
		return 0;
	}

	if (*saved_parabuf != '\0') {
		sendto_serv_capab_msg_butone(cptr, sptr, NO_CAPS, ID_CAPS, &CMD_SJOIN,
			"%ld %s %s %s :%s", tstosend, chptr->chname, saved_modebuf,
			saved_parabuf, sjbuf);
		sendto_serv_capab_msg_butone(cptr, sptr, ID_CAPS, NO_CAPS, &CMD_SJOIN,
			"%B %s %s %s :%s", tstosend, chptr->chname, saved_modebuf,
			saved_parabuf, idsjbuf);
	}
	else {
		sendto_serv_capab_msg_butone(cptr, sptr, NO_CAPS, ID_CAPS, &CMD_SJOIN,
			"%ld %s %s :%s", tstosend, chptr->chname, saved_modebuf, sjbuf);
		sendto_serv_capab_msg_butone(cptr, sptr, ID_CAPS, NO_CAPS, &CMD_SJOIN,
			"%B %s %s :%s", tstosend, chptr->chname, saved_modebuf, idsjbuf);
	}

	return 0;
}

/*
 * m_resynch
 *	parv[0] = sender prefix
 *	parv[1] = channel name
 */
int m_resynch(aClient *cptr, aClient *sptr, int parc, char *parv[])
{
	aChannel *chptr;

	if (!MyConnect(sptr) || !IsServer(sptr) || parc < 2) {
		return 0;
	}

	chptr = find_channel(parv[1], NULL);

	sendto_realops_lev(DEBUG_LEV, "%s has requested a resynch of %s%s", sptr->name,
		parv[1], chptr == NULL ? " (failed)" : "");

	if (chptr != NULL) {
		synch_chan_modes(sptr, chptr);
	}

	return 0;
}
