/*
 * RageIRCd: an advanced Internet Relay Chat daemon (ircd).
 * (C) 2000-2005 the RageIRCd Development Team, all rights reserved.
 *
 * This software is free, licensed under the General Public License.
 * Please refer to doc/LICENSE and doc/README for further details.
 *
 * $Id: m_dccallow.c,v 1.48.2.2 2005/01/15 23:53:33 amcwilliam Exp $
 */

#include "config.h"
#include "struct.h"
#include "common.h"
#include "sys.h"
#include "numeric.h"
#include "msg.h"
#include "channel.h"
#include "h.h"
#include "memory.h"
#include "hook.h"
#include "dlink.h"
#include "modules.h"
#include "xmode.h"
#include "user_ban.h"
#include <time.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>

static int allow_dcc(HookData *);
static int del_dccallows(HookData *);

Module MOD_HEADER(m_dccallow) = {
	"m_dccallow",
	"/DCCALLOW command",
	6, "$Revision: 1.48.2.2 $"
};

int MOD_LOAD(m_dccallow)()
{
	Hook *h1, *h2, *h3;

	if ((h1 = hook_find("h_dccsend")) == NULL) {
		ircdlog(LOG_ERROR, "m_dccallow.so: couldn't find hook h_dccsend");
		return MOD_FAILURE;
	}
	if ((h2 = hook_find("h_nick_change_local")) == NULL) {
		ircdlog(LOG_ERROR, "m_dccallow.so: couldn't find hook h_nick_change_local");
		return MOD_FAILURE;
	}
	if ((h3 = hook_find("h_nick_change_remote")) == NULL) {
		ircdlog(LOG_ERROR, "m_dccallow.so: couldn't find hook h_nick_change_remote");
		return MOD_FAILURE;
	}

	if (register_command(&MOD_HEADER(m_dccallow), &CMD_DCCALLOW, m_dccallow) == NULL) {
		return MOD_FAILURE;
	}

	if (register_hook_event(&MOD_HEADER(m_dccallow), h1, allow_dcc) == NULL) {
		/* h_dccsend */
		return MOD_FAILURE;
	}
	if (register_hook_event(&MOD_HEADER(m_dccallow), h2, del_dccallows) == NULL) {
		/* h_nick_change_local */
		return MOD_FAILURE;
	}
	if (register_hook_event(&MOD_HEADER(m_dccallow), h3, del_dccallows) == NULL) {
		/* h_nick_change_remote */
		return MOD_FAILURE;
	}
	if (register_hook_event(&MOD_HEADER(m_dccallow), h_exit_user_local, del_dccallows) == NULL) {
		return MOD_FAILURE;
	}

	return MOD_SUCCESS;
}

int MOD_UNLOAD(m_dccallow)()
{
	return MOD_SUCCESS;
}

static int allow_dcc(HookData *hdata)
{
	aClient *from = hdata->sptr;
	aClient *to = hdata->acptr;
	char filename[BUFSIZE], *s;
	simBan *sban;
	int len = 0;

	if (to == from || !IsPerson(from) || IsULine(from) || HasMode(from, UMODE_OPER)) {
		return 0;
	}

	ASSERT(!BadPtr(hdata->c));
	strncpyzt(filename, hdata->c, BUFSIZE);
	s = filename + 8;

	while (IsSpace(*s)) {
		s++;
	}
	if (BadPtr(s)) {
		return 0;
	}

	while (*(s + len) != ' ') {
		if (*(s + len) == '\0') {
			break;
		}
		len++;
	}
	s[len] = '\0';

	if ((sban = find_simban_flags(s, SBAN_FILE)) == NULL) {
		return 0;
	}
	if (dlink_find(&to->user->dccallow_list, from) != NULL) {
		return 0;
	}

	send_me_notice(from, ":*** The transfer of file %s is not permitted on this server: %s", s, BanReason(sban));
	send_me_notice(to, ":*** %s attempted to send you file %s, but failed because this file is restricted: %s",
		from->name, s, BanReason(sban));

	if (!SeenDCCNotice(to)) {
		send_me_notice(to, ":*** If you trust %s and wish to receive this file, regardless "
			"of the restriction, add him/her to your dccallow list. Type /DCCALLOW ? for "
			"further details. You will not receive this message again.", from->name);
		SetDCCNotice(to);
	}

	sendto_realops_lev(DCCSEND_LEV, "%s (%s@%s) attempted to transfer restricted file %s to %s "
		"(%s@%s) [%s]", from->name, from->username, MaskedHost(from), s, to->name, to->username,
		MaskedHost(to), BanReason(sban));

	return FLUSH_BUFFER;
}

static void show_dccallow_list(aClient *cptr)
{
	dlink_node *node;
	aClient *acptr;
	char buf[BUFSIZE], *s;
	int len = 0, count = 0;
	int plen = strlen(cptr->name) + 10;

	DLINK_FOREACH_DATA(cptr->user->dccallow_list.head, node, acptr, aClient) {
		if ((plen + len + strlen(acptr->name) + 1) > BUFSIZE || count > 10) {
			send_me_numeric(cptr, RPL_DCCALLOWLIST, buf);
			buf[0] = '\0';
			len = count = 0;
		}
		if (len > 0) {
			buf[len++] = ' ';
		}
		for (s = acptr->name; *s != '\0'; s++) {
			buf[len++] = *s;
		}
		buf[len] = '\0';
		count++;
	}
	if (count > 0) {
		send_me_numeric(cptr, RPL_DCCALLOWLIST, buf);
	}
	send_me_numericNA(cptr, RPL_ENDOFDCCALLOW);
}

static void show_dccallow_help(aClient *cptr)
{
	char **p;
	static char *helptext[] = {
		"/DCCALLOW [-]nick[,[-]nick2[,[-]nick3[...]]]",
		"/DCCALLOW <*|?>",
		" ",
		"Transmission (DCC SEND) of some filetypes are banned in order to prevent",
		"the spread of trojans and viruses. However, you can choose to bypass this",
		"block. This is done by adding clients to your dcc allow list.",
		" ",
		"For example; client Bob is attempting to send you mirc.zip. However,",
		"zip files (*.zip) are banned from transfer. So, you would issue the",
		"following command.",
		" ",
		"/DCCALLOW Bob",
		" ",
		"After receiving the file, you may then want to remove Bob from your dcc",
		"allow list. Syntax is similar to that of adding clients, but you prefix",
		"the nickname with a - (subtract) char.",
		" ",
		"/DCCALLOW -Bob",
		" ",
		"You can add and remove more than one client to and from your dcc allow",
		"list at once. Just separate your client list with, (commas), as per the",
		"following example.",
		" ",
		"/DCCALLOW Bob,-Joe,-Lisa,Homer",
		" ",
		"To view all clients on your dcc allow list, issue a DCCALLOW command with",
		"an * (asterisk) as parameter. To view this help, issue a DCCALLOW command",
		"with a ? (question mark). You cannot mix *, ? and/or nicknames together.",
		NULL
	};
	for (p = helptext; *p != '\0'; p++) {
		send_me_numeric(cptr, RPL_DCCALLOWLIST, *p);
	}
	send_me_numericNA(cptr, RPL_ENDOFDCCALLOW);
}

static void add_dccallow(aClient *cptr, aClient *acptr)
{
	dlink_add(&cptr->user->dccallow_list, acptr);
	dlink_add(&acptr->user->on_dccallow_list, cptr);
}

static void del_dccallow(aClient *sptr, aClient *acptr)
{
	dlink_node *anode, *anext = NULL;
	dlink_node *node, *next = NULL;
	aClient *cptr;

	DLINK_FOREACH_SAFE_DATA(acptr->user->dccallow_list.head, anode, anext, cptr, aClient) {
		if (sptr != cptr) {
			continue;
		}

		dlink_del(&acptr->user->dccallow_list, NULL, anode);
		DLINK_FOREACH_SAFE_DATA(sptr->user->on_dccallow_list.head, node, next, cptr, aClient) {
			if (acptr == cptr) {
				dlink_del(&sptr->user->on_dccallow_list, NULL, node);
			}
		}
	}
}

static int del_dccallows(HookData *hdata)
{
	aClient *sptr = hdata->sptr;
	aClient *acptr;
	dlink_node *node, *next = NULL;

	DLINK_FOREACH_SAFE_DATA(sptr->user->dccallow_list.head, node, next, acptr, aClient) {
		if (acptr != NULL) {
			del_dccallow(acptr, sptr);
		}
	}
	DLINK_FOREACH_SAFE_DATA(sptr->user->on_dccallow_list.head, node, next, acptr, aClient) {
		if (acptr != NULL) {
			del_dccallow(sptr, acptr);
		}
	}
	return 0;
}

/*
 * m_dccallow
 *	parv[0] = sender prefix
 *	parv[1] = parameter
 */
int m_dccallow(aClient *cptr, aClient *sptr, int parc, char *parv[])
{
	char *s, *p;
	aClient *acptr;
	int del;

	if (!MyClient(sptr)) {
		return 0;
	}
	if (parc < 2 || BadPtr(parv[1])) {
		send_me_numeric(sptr, ERR_NEEDMOREPARAMS, "DCCALLOW");
		return 0;
	}
	if (*parv[1] == '*') {
		show_dccallow_list(sptr);
		return 0;
	}
	else if (*parv[1] == '?') {
		show_dccallow_help(sptr);
		return 0;
	}

	for (s = strtoken(&p, parv[1], ","); s != NULL; s = strtoken(&p, NULL, ",")) {
		while (IsSpace(*s)) {
			s++;
		}
		if (BadPtr(s)) {
			continue;
		}

		if (*s == '-') {
			del = 1;
			s++;
		}
		else {
			del = 0;
			if (*s == '+') {
				s++;
			}
		}

		/* Make sure we have enough space if we're adding */
		if (!del && (GeneralConfig.max_dccallow > 0)) {
			if (dlink_length(&sptr->user->dccallow_list) >= GeneralConfig.max_dccallow) {
				send_me_numeric(sptr, ERR_DCCALLOWFULL, GeneralConfig.max_dccallow);
				continue;
			}
		}

		if ((acptr = find_person(s, NULL)) == NULL) {
			send_me_numeric(sptr, ERR_NOSUCHNICK, s);
			continue;
		}
		else if (acptr == sptr) {
			continue;
		}

		if (del) {
			if (dlink_find(&sptr->user->dccallow_list, acptr) == NULL) {
				send_me_numeric_buf(sptr, ":%s is not on your dcc allow list",
					RPL_DCCALLOWINFO, acptr->name);
			}
			else {
				del_dccallow(acptr, sptr);
				send_me_numeric(sptr, RPL_DCCALLOWSTATUS, acptr->name, "removed from");
			}
		}
		else {
			if (dlink_find(&sptr->user->dccallow_list, acptr) != NULL) {
				send_me_numeric_buf(sptr, ":%s is already on your dcc allow list",
					RPL_DCCALLOWINFO, acptr->name);
			}
			else {
				add_dccallow(sptr, acptr);
				send_me_numeric(sptr, RPL_DCCALLOWSTATUS, acptr->name, "added to");
			}
		}
	}

	return 0;
}
