/*
 *  Ordinal: A Library of Ordinal Models
 *  Copyright (C) 1998, 1999, 2000, 2001  J.K. Lindsey and P.J. Lindsey
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  SYNOPSIS
 *
 *  void kord(double p[],int y[],double t[],double x[],int *nind,
 *            int nobs[],int *nbs,int *nc,int *np,int *npv,int *init,
 *            int *model,int *dep,int *torder,int inter[],int *tvc,
 *            double tvcov[],int *fit,double pred[],double rpred[],
 *            double cprob[],double ppred[],double cpred[],double creg[],
 *            int *rf,double bb[],double beta[],double bt[],double bt2[],
 *            double bet[],double cum[],double *like)
 *
 *  DESCRIPTION
 *
 *    Functions to compute the likelihood function for various
 * parameterization of the logistic distribution inserted in a beta
 * distribution with serial or frailty dependence using Kalman-type
 * update for ordinal longitudinal data.
 *
 *  SYNOPSIS
 *
 *  void pkord(double p[],int *nbs,int *nc,int *init,int *model,
 *             int *dep,int *preg,double cpred[],double pred[],
 *             double bb[],double beta[],double bt[],double bt2[],
 *             double bet[],double cum[]);
 *
 *  DESCRIPTION
 *
 *    Functions to compute the predicted marginal values for various
 * parameterization of the logistic distribution inserted in a beta
 * distribution with serial or frailty dependence using Kalman-type
 * update for ordinal longitudinal data.
 *
 */

#include <math.h>
#include <stddef.h>

void kord(double p[],int y[],double t[],double x[],int *nind,int nobs[],int *nbs,
          int *nc,int *np,int *npv,int *init,int *model,int *dep,int *torder,
          int inter[],int *tvc,double tvcov[],int *fit,double pred[],double rpred[],
          double cprob[],double ppred[],double cpred[],double creg[],int *rf,
          double bb[],double beta[],double bt[],double bt2[],double bet[],
          double cum[],double *like);

void pkord(double p[],int *nbs,int *nc,int *init,int *model,int *dep,
           int *preg,double cpred[],double pred[],double bb[],double beta[],
           double bt[],double bt2[],double bet[],double cum[]);

void kord(double p[],int y[],double t[],double x[],int *nind,int nobs[],int *nbs,
          int *nc,int *np,int *npv,int *init,int *model,int *dep,int *torder,
          int inter[],int *tvc,double tvcov[],int *fit,double pred[],double rpred[],
          double cprob[],double ppred[],double cpred[],double creg[],int *rf,
          double bb[],double beta[],double bt[],double bt2[],double bet[],
          double cum[],double *like) {
  int i,j,k,k1,k2,l,nm;
  double a,a1,a0a,a1a,b,b1,b0b,b1b,delta,omega,om,tmp,sum;

  *like=0;
  nm=0;
  if(*init)
    delta=exp(-p[*nc-1+*np+*npv+*tvc]);
  else
    delta=999999999.9999999;
  if(*dep>0&&*dep<3)
    omega=exp(p[*nc-1+*np+*npv+*tvc+*init])/(1+exp(p[*nc-1+*np+*npv+*tvc+*init]));
  for(i=0;i<*nind;i++){
    a=b=delta;
    if(*fit)
      a0a=b0b=delta;
    for(k=0;k<(*nc-1);k++) {
      beta[k]=p[k];
      if(!*rf)
        for(l=0;l<*np;l++)
          beta[k]+=p[*nc-1+l]*x[*nind*l+i];
      else
        if(*tvc==0)
          beta[k]+=bb[i];
    }
    if(*tvc==0&&*torder==0) {
      for(k=0;k<(*nc-1);k++)
        bt[k]=exp(beta[k]);
      switch(*model){
      case 1: /* binary */
        bt[0]=bt[0]/(1+bt[0]);
        break;
      case 2: /* multinomial */
        for(k=0,sum=0;k<(*nc-1);k++)
          sum+=bt[k];
        for(k=0;k<(*nc-1);k++) {
          bt[k]=bt[k]/(1+sum);
          cum[k]=bt[k];
        }
        for(k=1;k<(*nc-1);k++)
          for(l=0;l<k;l++)
            cum[k]+=bt[l];
        for(k=0;k<(*nc-1);k++)
          bt[k]=cum[k];
        break;
      case 3: /* downwards continuation-ratio */
        for(k=0;k<(*nc-1);k++)
          bt2[k]=bt[k]/(1+bt[k]);
        bt[0]=1;
        for(k=1;k<(*nc-1);k++)
          bt[k]=1-bt2[k-1];
        for(k=0;k<(*nc-1);k++)
          for(l=k;l<(*nc-1);l++)
            bt[k]*=bt2[l];
        for(k=0;k<(*nc-1);k++)
          cum[k]=bt[k];
        for(k=1;k<(*nc-1);k++)
          for(l=0;l<k;l++)
            cum[k]+=bt[l];
        for(k=0;k<(*nc-1);k++)
          bt[k]=cum[k];
        break;
      case 4: /* proportional-odds */
        for(k=0;k<(*nc-1);k++)
          bt[k]=bt[k]/(1+bt[k]);
        break;
      }
    }
    else
      for(k=0;k<(*nc-1);k++)
        bet[k]=beta[k];
    for(j=0;j<nobs[i];j++){
      a1=a+1;
      b1=b;
      if(*fit) {
        a1a=a0a+1;
        b1b=b0b;
      }
/* add in time-varying covariates */
      if(*torder){
        tmp=1;
        k1=k2=0;
        for(k=0;k<*npv;k++) {
          if(k<*torder)tmp*=t[nm];
          else {
            if(k2>inter[k1]) {
              k1++;
              k2=0;
            }
            if(k2==0) {
              tmp=x[*nind*k1+i]*t[nm];
              k2++;
            }
            else {
              tmp*=t[nm];
              k2++;
            }
          }
        }
        for(k=0;k<(*nc-1);k++) {
          beta[k]=bet[k];
          for(l=0;l<*npv;l++)
            beta[k]+=p[*nc-1+*np+l]*tmp;
          bt[k]=exp(beta[k]);
        }
      }
      if(*tvc>0){
        for(k=0;k<(*nc-1);k++) {
          beta[k]=bet[k];
          if(!*rf)
            for(l=0;l<*tvc;l++)
              beta[k]+=p[*nc-1+*np+*npv+l]*tvcov[nm+*nbs*l];
          else
            beta[k]+=bb[nm];
          bt[k]=exp(beta[k]);
        }
      }
      if(*torder||*tvc){
        switch(*model){
        case 1: /* binary */
          bt[0]=bt[0]/(1+bt[0]);
          break;
        case 2: /* multinomial */
          for(k=0,sum=0;k<(*nc-1);k++)
            sum+=bt[k];
          for(k=0;k<(*nc-1);k++) {
            bt[k]=bt[k]/(1+sum);
            cum[k]=bt[k];
          }
          for(k=1;k<(*nc-1);k++)
            for(l=0;l<k;l++)
              cum[k]+=bt[l];
          for(k=0;k<(*nc-1);k++)
            bt[k]=cum[k];
          break;
        case 3: /* downwards continuation-ratio */
          for(k=0;k<(*nc-1);k++)
            bt2[k]=bt[k]/(1+bt[k]);
          bt[0]=1;
          for(k=1;k<(*nc-1);k++)
            bt[k]=1-bt2[k-1];
          for(k=0;k<(*nc-1);k++)
            for(l=k;l<(*nc-1);l++)
              bt[k]*=bt2[l];
          for(k=0;k<(*nc-1);k++)
            cum[k]=bt[k];
          for(k=1;k<(*nc-1);k++)
            for(l=0;l<k;l++)
              cum[k]+=bt[l];
          for(k=0;k<(*nc-1);k++)
            bt[k]=cum[k];
          break;
        case 4: /* proportional-odds */
          for(k=0;k<(*nc-1);k++)
            bt[k]=bt[k]/(1+bt[k]);
          break;
        }
      }
/* calculate likelihood */
      switch(*model) {
      case 1: /* binary */
        if(y[nm]==0) {
          *like-=log(1-pow(b/(b-log(1-bt[0])),a));
        }
        else {
          b1-=log(1-bt[0]);
          if(*fit)
            b1b-=log(1-bt[0]);
          *like-=a*log(b/b1);
        }
        break;
      case 2: /* multinomial */
      case 3: /* downwards continuation-ratio */
      case 4: /* proportional-odds */
        if(y[nm]==0) {
          *like-=log(1-pow(b/(b-log(1-bt[0])),a));
        }
        else
          if(y[nm]==(*nc-1)) {
            b1-=log(1-bt[*nc-2]);
            if(*fit)
              b1b-=log(1-bt[*nc-2]);
            *like-=a*log(b/b1);
          }
          else {
            b1-=log(1-bt[y[nm]-1]);
            if(*fit)
              b1b-=log(1-bt[y[nm]-1]);
            *like-=log(pow(b/b1,a)-pow(b/(b-log(1-bt[y[nm]])),a));
          }
        break;
      }
/* calculate fitted values */
      if(*fit) {
        for(k=0;k<(*nc-1);k++)
          creg[*nbs*k+nm]=bt[k];
        for(k=0;k<(*nc-1);k++)
          cpred[*nbs*k+nm]=1-pow(b0b/(b0b-log(1-bt[k])),a0a);
        pred[nm]=0;
        if(*model==1) {
          if((1-cpred[nm])>cpred[nm])
            pred[nm]=1;
        }
        else {
          if((cpred[*nbs+nm]-cpred[nm])>cpred[nm])
            pred[nm]=1;
          for(k=2;k<(*nc-1);k++)
            if((cpred[*nbs*k+nm]-cpred[*nbs*(k-1)+nm])>(cpred[*nbs*(k-1)+nm]-cpred[*nbs*(k-2)+nm]))
              pred[nm]=k;
          if((1-cpred[*nbs*(*nc-2)+nm])>(cpred[*nbs*(*nc-2)+nm]-cpred[*nbs*(*nc-3)+nm]))
            pred[nm]=*nc-1;
        }
        if(*dep) {
          for(k=0;k<(*nc-1);k++)
            cprob[*nbs*k+nm]=1-pow(b/(b-log(1-bt[k])),a);
          ppred[nm]=0;
          if(*model==1) {
            if((1-cprob[nm])>cprob[nm])
              ppred[nm]=1;
          }
          else {
            if((cprob[*nbs+nm]-cprob[nm])>cprob[nm])
              ppred[nm]=1;
            for(k=2;k<(*nc-1);k++)
              if((cprob[*nbs*k+nm]-cprob[*nbs*(k-1)+nm])>(cprob[*nbs*(k-1)+nm]-cprob[*nbs*(k-2)+nm]))
                ppred[nm]=k;
            if((1-cprob[*nbs*(*nc-2)+nm])>(cprob[*nbs*(*nc-2)+nm]-cprob[*nbs*(*nc-3)+nm]))
              ppred[nm]=*nc-1;
          }
        }
        else {
          for(k=0;k<(*nc-1);k++)
            cprob[*nbs*k+nm]=cpred[*nbs*k+nm];
          ppred[nm]=pred[nm];
        }
        rpred[nm]=0;
        for(k=1;k<(*nc-1);k++)
          rpred[nm]+=k*(cprob[*nbs*k+nm]-cprob[*nbs*(k-1)+nm]);
        rpred[nm]+=(*nc-1)*(1-cprob[*nbs*(*nc-2)+nm]);
    }
/* update parameters */
      if(*dep) {
        if(*dep<3) {
          om=j?pow(omega,t[nm]-t[nm-1]):1;
          a=om*a1+(1-om)*delta;
          if(*dep==1)
            b=delta+om*(b1-b);
          else
            if(*dep==2)
              b=om*b1+(1-om)*delta;
        }
        else {
          a=a1;
          b=b1;
        }
      }
      nm++;
    }
  }
  return;
}

void pkord(double p[],int *nbs,int *nc,int *init,int *model,int *dep,
           int *preg,double cpred[],double pred[],double bb[],double beta[],
           double bt[],double bt2[],double bet[],double cum[]) {
  int j,k,l;
  double a,a1,b,b1,delta,tmp,sum;

  if(*init)
    delta=exp(-p[*nc-1+*preg]);
  else
    delta=999999999.9999999;
  a=b=delta;
  for(k=0;k<(*nc-1);k++)
    bet[k]=beta[k]=p[k];
  for(j=0;j<*nbs;j++){
    a1=a+1;
    b1=b;
/* add in time-varying covariates */
    for(k=0;k<(*nc-1);k++) {
      beta[k]=bet[k]+bb[j];
      bt[k]=exp(beta[k]);
    }
    switch(*model) {
    case 1: /* binary */
      bt[0]=bt[0]/(1+bt[0]);
      break;
    case 2: /* multinomial */
      for(k=0,sum=0;k<(*nc-1);k++)
        sum+=bt[k];
      for(k=0;k<(*nc-1);k++) {
        bt[k]=bt[k]/(1+sum);
        cum[k]=bt[k];
      }
      for(k=1;k<(*nc-1);k++)
        for(l=0;l<k;l++)
          cum[k]+=bt[l];
      for(k=0;k<(*nc-1);k++)
        bt[k]=cum[k];
      break;
    case 3: /* downwards continuation-ratio */
      for(k=0;k<(*nc-1);k++)
        bt2[k]=bt[k]/(1+bt[k]);
      bt[0]=1;
      for(k=1;k<(*nc-1);k++)
        bt[k]=1-bt2[k-1];
      for(k=0;k<(*nc-1);k++)
        for(l=k;l<(*nc-1);l++)
          bt[k]*=bt2[l];
      for(k=0;k<(*nc-1);k++)
        cum[k]=bt[k];
      for(k=1;k<(*nc-1);k++)
        for(l=0;l<k;l++)
          cum[k]+=bt[l];
      for(k=0;k<(*nc-1);k++)
        bt[k]=cum[k];
      break;
    case 4: /* proportional-odds */
      for(k=0;k<(*nc-1);k++)
        bt[k]=bt[k]/(1+bt[k]);
      break;
    }
    if(*model==1) {
      if(bt[0]>0.5) {
        b1-=log(1-bt[0]);
      }
    }
    else {
      tmp=0;
      if((bt[1]-bt[0])>bt[0])
        tmp=log(1-bt[0]);
      for(k=2;k<(*nc-1);k++) {
        if((bt[k]-bt[k-1])>(bt[k-1]-bt[k-2]))
          tmp=log(1-bt[k-1]);
      }
      if((1-bt[*nc-2])>(bt[*nc-2]-bt[*nc-3]))
        tmp=log(1-bt[*nc-2]);
      b1-=tmp;
    }
/* calculate fitted values */
    for(k=0;k<(*nc-1);k++)
      cpred[*nbs*k+j]=1-pow(b/(b-log(1-bt[k])),a);
    pred[j]=0;
    if(*model==1) {
      if((1-cpred[j])>cpred[j])
        pred[j]=1;
    }
    else {
      if((cpred[*nbs+j]-cpred[j])>cpred[j])
        pred[j]=1;
      for(k=2;k<(*nc-1);k++)
        if((cpred[*nbs*k+j]-cpred[*nbs*(k-1)+j])>(cpred[*nbs*(k-1)+j]-cpred[*nbs*(k-2)+j]))
          pred[j]=k;
      if((1-cpred[*nbs*(*nc-2)+j])>(cpred[*nbs*(*nc-2)+j]-cpred[*nbs*(*nc-3)+j]))
        pred[j]=*nc-1;
    }
/* update parameters */
    if(*dep)
      if(*dep<3)
        a=b=delta;
      else {
        a=a1;
        b=b1;
      }
  }
  return;
}
