\name{moprofile}
\title{Produce Marginal Ordinal Time Profiles for Plotting}
\usage{
plot(moprofile(z,curve.type="probability"),nind=1,observed=T,
     main=NULL,xlab=NULL,ylab=NULL,xlim=NULL,ylim=NULL,lty=NULL,
     pch=NULL,add=F,axes=F,bty="n",at=NULL,touch=F,...)
}
\alias{moprofile}
\alias{moprofile.default}
\alias{moprofile.lcr}
\alias{moprofile.kalordinal}
\alias{plot.moprofile}
\description{
  \code{mprofile} is used for plotting marginal ordinal profiles over
  time for for objects obtained from models obtained. It produces output
  for plotting highest probabilities and cumulative probabilities for
  marginal ordinal time profiles.

  See \code{\link{iprofile}} for plotting individual ordinal profiles
  from recursive fitted values.
}
\arguments{
  \item{z}{An object of class \code{lcr} or \code{kalordinal}
    (\code{\link{kalord}}).}
  \item{curve.type}{Specifies the type of curves to be plotted. Must
    either be "probability" for highest probabilities or "cumulative"
    for cumulative probabilities.}
  \item{nind}{Observation number(s) of individual(s) to be plotted.}
  \item{observed}{If TRUE, adds the corresponding observations to the
    plot. If cumulative curves have been chosen, they are added as a
    subtitle.}
  \item{main}{A main title for the plot.}
  \item{xlab}{A label for the x-axis.}
  \item{ylab}{A label for the y-axis.}
  \item{xlim}{The x limits (min,max) of the plot.}
  \item{ylim}{The y limits (min,max) of the plot.}
  \item{lty}{A vector of integers or character strings specifying the
    line type to be used as the default in plotting lines. For further
    information, see \code{\link{par}}.}
  \item{pch}{A vector of integers or single characters specifying
    symbols to be used as the default in plotting points. For further
    information, see \code{\link{par}}.}
  \item{add}{If TRUE, the graph is added to an existing plot.}
  \item{axes}{If FALSE, axes are not drawn around the plot.}
  \item{bty}{A character string which determined the type of box which
    is drawn about plots. For further information, see
    \code{\link{par}}.}
  \item{at}{The points at which tick-marks are to be drawn. For further
    information, see \code{\link{axis}}.}
  \item{touch}{If TRUE, the x-axis and y-axis will touch each other.}
}
\value{
  \code{moprofile} returns information ready for plotting by
  \code{plot.moprofile}.
}
\author{P.J. Lindsey}
\seealso{
  \code{\link{kalord}}, \code{\link{ioprofile}}, \code{\link{lcr}},
  \code{\link{plot.ordinal}}, \code{\link{poprofile}}.
}
\examples{
library(ordinal)

#
# Binary data
#
data(cardiac.indiv)

y <- restovec(cardiac.indiv[,1:4],type="ordinal")

cov <- tcctomat(as.matrix(cardiac.indiv[,5:10]))

w <- rmna(y,ccov=cov)

rm(cardiac.indiv,y,cov)

# Time-constant and time-varying covariate with a frailty dependence.
z <- kalord(w,distribution="binary",mu=~age+ren+cop+dia+sex+pmi+times,
            ptvc=c(4.43357,-0.03128,-0.62602,-0.37679,-0.32969,-0.17013,
                   -0.12209,-0.09095),pinit=0.1196,dep="frailty")

# Cumulative probability profiles.
par(mfrow=c(2,2))
plot(moprofile(z,"cum"),nind=1)
plot(moprofile(z,"cum"),nind=117)
plot(moprofile(z,"cum"),nind=c(1000,3000),add=T)
par(mfrow=c(1,1))

# Highest probability profiles.
par(mfrow=c(2,2))
plot(moprofile(z,"prob"),nind=2000)
plot(moprofile(z,"prob"),nind=3001)
plot(moprofile(z,"prob"),nind=c(3506,3521))
plot(moprofile(z,"prob"),nind=400)
par(mfrow=c(1,1))

rm(w,z)

#
# Ordinal data
#
data(obese)

resp <- cbind(codes(obese[,1])-1,codes(obese[,2])-1)
freq <- obese[,4]

age <- obese[,3]

rm(obese)

y <- restovec(resp,times=1:2,weights=freq,type="ordinal")

tcc <- tcctomat(age,name="age")

tvc <- tvctomat(matrix(times(y)^2,ncol=2),name="times2")

w <- rmna(y,ccov=tcc,tvcov=tvc)

rm(resp,freq,age,y,tcc,tvc)

z <- lcr(w,mu=~age*times+times2)

# Cumulative probability profiles.
par(mfrow=c(2,2))
plot(moprofile(z,"cum"),nind=1)
plot(moprofile(z,"cum"),nind=4)
plot(moprofile(z,"cum"),nind=8:9,add=T)
par(mfrow=c(1,1))

# Highest probability profiles.
par(mfrow=c(2,2))
plot(moprofile(z,"prob"),nind=1)
plot(moprofile(z,"prob"),nind=4)
plot(moprofile(z,"prob"),nind=c(8,9))
plot(moprofile(z,"prob"),nind=16)
par(mfrow=c(1,1))

rm(w,z)
}
\keyword{hplot}
