// evalif.c: if statements and for/repeat/while loops
//
// R : A Computer Language for Statistical Data Analysis
// Copyright (C) 1995, 1996    Robert Gentleman and Ross Ihaka
// Copyright (C) 1998--2006    The R Development Core Team.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, a copy is available at
// http://www.r-project.org/Licenses/

#ifdef HAVE_CONFIG_H
  #include "config.h"
#endif
#include "Defn.h"
#include "Print.h"
#define JIT_INTERNAL 1              // tell jit.h to include JIT tables etc.
#include "jit.h"
#include "jithash.h"
#include "printsxp.h"

#define SEXPV SEXPREC volatile *    // volatile pointer to SEXPREC

// these macros prevent "discarded qualifier" warnings from the
// compiler when p is volatile

#define VREPROTECT(p,i)             REPROTECT((SEXP)(p), (i))

#define VPROTECT_WITH_INDEX(p, i)   PROTECT_WITH_INDEX((SEXP)(p), (i))

//-----------------------------------------------------------------------------
// JIT manipulation functions which are defined inline here for speed
// instead of in jit.c where they strictly speaking belong.

static int R_INLINE inJitBlock(void)
{
    return jitState &
                (JITS_AWAITING_LOOP | JITS_IN_LOOP | JITS_COMPILING_STATES);
}

static Rboolean R_INLINE isWhileLoopJittable(CSEXP rhs)
{
    return inJitBlock() &&
           (TYPEOF(rhs) == LGLSXP ||
            TYPEOF(rhs) == INTSXP ||
            TYPEOF(rhs) == REALSXP);
}

// Called at the start of an R for/while/repeat loop.
// Returns the old jitState if caller must must call jitExitLoop, else 0

static R_INLINE unsigned jitEnterLoop(CSEXP s, CSEXP body)
{
    if (jitState & (JITS_AWAITING_LOOP | JITS_COMPILING_STATES))
        return jitEnterLoopAux(s, body);

    return 0;
}

static R_INLINE void genJitForPossibly(CSEXP rhs, CSEXP sym, CSEXP body,
                                       CSEXP call, CSEXP rho)
{
    // The check against jitState also prevents compile of outermost loop
    // (it is pointless to optimize the outer loop).

    if (jitDirective >= 2 &&
           (jitState & (JITS_IN_LOOP | JITS_COMPILING_STATES)) &&
           TYPEOF(rhs) == INTSXP) {

        genjitFor(rhs, sym, body, call, rho);
    }
}

static R_INLINE void traceLoopIteration(const char msg[],
                                        CSEXP indexVar, int i)
{
#if DEBUG_JIT > 1
    if (jitDirective && (traceEvalFlag || jitTrace >= 4))
        printLoopIteration(msg, indexVar, i);
#endif
}

//-----------------------------------------------------------------------------

static R_INLINE Rboolean bodyHasBraces(CSEXP body)
{
    return isLanguage(body) && CAR(body) == R_BraceSymbol;
}

static void debugLoopAux(CSEXP call, CSEXP op, CSEXP args,
                         CSEXP rho, Rboolean bgn)
{
    printf("debug: ");
    PrintValue(CAR(args));
    do_browser(call, op, args, rho);
}

static R_INLINE void debugLoop(CSEXP call, CSEXP op, CSEXP args,
                               CSEXP rho, Rboolean bgn)
{
    if (bgn && DEBUG(rho))
        debugLoopAux(call, op, args, rho, bgn);
}

static R_INLINE Rboolean asLogicalNoNA(CSEXP s, CSEXP call)
{
    Rboolean cond = NA_LOGICAL;

    if (length(s) > 1)
	warningcall(call,
		    _("the condition has length > 1 and only the first element will be used"));
    if (length(s) > 0) {
	/* inline common cases for efficiency */
	switch(TYPEOF(s)) {
	case LGLSXP: 
	    cond = LOGICAL(s)[0];
	    break;
	case INTSXP:
	    cond = INTEGER(s)[0]; /* relies on NA_INTEGER == NA_LOGICAL */
	    break;
	default:
	    cond = asLogical(s);
	}
    }

    if (cond == NA_LOGICAL) {
	char *msg = length(s) ? (isLogical(s) ?
				 _("missing value where TRUE/FALSE needed") :
				 _("argument is not interpretable as logical")) :
	    _("argument is of length zero");
	errorcall(call, msg);
    }
    return cond;
}

//-----------------------------------------------------------------------------
// do_break also handles "next"

SEXP attribute_hidden do_break(SEXP call, SEXP op, SEXP args, SEXP rho)
{
    findcontext(PRIMVAL(op), rho, RNIL);
    return RNIL;
}

SEXP attribute_hidden do_if(SEXP call, SEXP op, SEXP args, SEXP rho)
{
    SEXP result = RNIL;
    SEXP Cond; 
    PROTECT(Cond = eval(CAR(args), rho));
    if (jitCompiling())
        genjitIf(Cond, result, call, args, rho);
    if (asLogicalNoNA(Cond, call))
        result = eval(CADR(args), rho);
    else if (length(args) > 2)
        result = eval(CADDR(args), rho);
    UNPROTECT(1);
    jitUnsuspend();     // reallow generation of jit code
    if (result == RNIL)
        R_Visible = FALSE;
    return result;
}

//-----------------------------------------------------------------------------
// While loops.  For these we need to evaluate the condition once to
// determine if the loop can be jitted (based on the condition type).
// Thus on entry to whilejit and while1, the loop condition "cond"
// has already been evaluated once.
//
// General comments which apply to all loop code in this file:
// We use const variables where possible to help the compiler optimize
// and to reduce the risk of changing a non volatile variable in longjmp
// code.  Volatile vars are only used where strictly necessary, to avoid
// slowness --- if a var is in a setjmp block but is always re-inited before
// use even when there is a longjmp then it does not need to be volatile.
// For example "econd" below does not have to be volatile because it is
// revaluated in the CTXT_NEXT code if there is a longjump.

static R_INLINE SEXP while1(CSEXP call, CSEXP op, CSEXP args, CSEXP body,
                            CSEXP rho, CSEXP econd1,
                            const PROTECT_INDEX api, JMP_BUF cjmpbuf)
{
    const Rboolean bgn = bodyHasBraces(body);
    SEXPV ans = RNIL;                   // init in case loop isn't entered
    CSEXP cond = CAR(args);             // unevaluated loop condition
    SEXP  econd = econd1;               // local non const evaluated loop cond

    int jmpVal = SETJMP(cjmpbuf);
    if (jmpVal == CTXT_BREAK)
        return (SEXP)ans;               // note return
    else if (jmpVal == CTXT_NEXT)
        econd = eval(cond, rho);

    while (asLogicalNoNA(econd, call)) {
        debugLoop(call, op, args, rho, bgn);
        VREPROTECT(ans = eval(body, rho), api);
        econd = eval(cond, rho);
    }
    return (SEXP)ans;       // typecast prevents "discarded qualifier" warning
}

static void condTypeErr(SEXPTYPE condType, CSEXP econd)
{
    error(_("cannot change the type of a jitted loop condition\n"
            "Tried to change %s to %s"),
            type2char(condType), type2char(TYPEOF(econd)));
}

static R_INLINE void checkCondType(SEXPTYPE condType, CSEXP econd)
{
    if (condType != TYPEOF(econd))
        condTypeErr(condType, econd);
}

#define JIT_WHILE(TYPE)                                     \
    int jmpVal = SETJMP(cjmpbuf);                           \
    if (jmpVal == CTXT_BREAK)                               \
        return;                 /* note return */           \
    else if (jmpVal == CTXT_NEXT) {                         \
        econd = eval(cond, rho);                            \
        checkCondType(condType, econd);                     \
    }                                                       \
    while (TYPE(econd)[0]) {                                \
        eval(body, rho);                                    \
        econd = eval(cond, rho);                            \
        checkCondType(condType, econd);                     \
        if (TYPEOF(cond) == JITSXP) /* goto loop below */   \
            break;                                          \
    }                                                       \
    while (TYPE(econd)[0]) {                                \
        /* cond is jitted, body may not be */               \
        eval(body, rho);                                    \
        econd = evalJit(cond);                              \
        checkCondType(condType, econd);                     \
        if (TYPEOF(body) == JITSXP) /* goto loop below */   \
            break;                                          \
    }                                                       \
    while (TYPE(econd)[0]) {                                \
        /* cond and body are jitted */                      \
        evalJit(body);                                      \
        econd = evalJit(cond);                              \
        checkCondType(condType, econd);                     \
    }

// The function whilejit is used in JIT blocks, where the loop condition
// type can't change and integer NAs don't need to be supported.  It thus
// doesn't need to call asLogicalNoNA (which is quite slow).  It also
// doesn't call debugLoop, and doesn't need to assign eval(body) to ans.

static R_INLINE void whilejit(CSEXP call, CSEXP args, CSEXP body, CSEXP rho,
                              CSEXP econd1, JMP_BUF cjmpbuf)
{
    CSEXP cond = CAR(args);     // unevaluated loop condition
    SEXP  econd = econd1;       // local non const evaluated loop cond
    const SEXPTYPE condType = TYPEOF(econd);

    if (condType == LGLSXP) {
        JIT_WHILE(LOGICAL);     // may invoke "return"
    } else if (condType == INTSXP) {
        JIT_WHILE(INTEGER);
    } else {
        Dassert(condType == REALSXP);
        JIT_WHILE(REAL);
    }
    return;
}

SEXP attribute_hidden do_while(SEXP call, SEXP op, SEXP args, SEXP rho)
{
    const int dbg = DEBUG(rho);
    CSEXP body = CADR(args);
    const unsigned prevJitState = jitEnterLoop(call, body);
    SEXP ans = RNIL;        // init in case loop isn't entered
    PROTECT_INDEX api;
    PROTECT_WITH_INDEX(ans, &api);
    RCNTXT cntxt;
    checkArity(op, args);

    begincontext(&cntxt, CTXT_LOOP, RNIL, rho, R_BaseEnv, RNIL, RNIL);

    CSEXP econd = eval(CAR(args), rho); // eval loop condition

    if (isWhileLoopJittable(econd))
        whilejit(call, args, body, rho, econd, cntxt.cjmpbuf);
    else {
        ans = while1(call, op, args, body, rho, econd, api, cntxt.cjmpbuf);
        REPROTECT(ans, api);
    }
    endcontext(&cntxt);

    if (prevJitState)
       jitExitLoop(call, prevJitState);
    SET_DEBUG(rho, dbg);
    UNPROTECT(1);
    return (SEXP)ans;       // typecast prevents "discarded qualifier" warning
}

//-----------------------------------------------------------------------------
// Repeat loops.

SEXP attribute_hidden do_repeat(SEXP call, SEXP op, SEXP args, SEXP rho)
{
    RCNTXT cntxt;
    const int dbg = DEBUG(rho);
    CSEXP body = CAR(args);
    const Rboolean bgn = bodyHasBraces(body);
    const unsigned prevJitState = jitEnterLoop(call, body);
    SEXPV ans = RNIL;
    PROTECT_INDEX api;
    VPROTECT_WITH_INDEX(ans, &api);
    checkArity(op, args);

    begincontext(&cntxt, CTXT_LOOP, RNIL, rho, R_BaseEnv, RNIL, RNIL);

    if (inJitBlock()) {
        if (SETJMP(cntxt.cjmpbuf) != CTXT_BREAK)
            while(1)
                eval(body, rho);
    } else {
        if (SETJMP(cntxt.cjmpbuf) != CTXT_BREAK)
            while(1) {
                debugLoop(call, op, args, rho, bgn);
                VREPROTECT(ans = eval(body, rho), api);
            }
    }

    endcontext(&cntxt);
    if (prevJitState)
       jitExitLoop(call, prevJitState);
    SET_DEBUG(rho, dbg);
    UNPROTECT(1);
    return (SEXP)ans;       // typecast prevents "discarded qualifier" warning
}

//-----------------------------------------------------------------------------
// For loops.
//
// Notation: in the example loop expression "i in 11:13":
//     sym  is SYMSXP i                   (called "the loop variable")
//     rhs  is INTSXP with elems 11,12,13 (evaluated right hand side)
//     v    is INTSXP length 1, will be set successively to 11,12,13
//
// The approach used in forjit could be used in for1. But to be
// conservative we retain the original do_for code in for1 and use it
// when not jitting.

static SEXP for1(CSEXP call, CSEXP op, CSEXP args,
                 CSEXP sym, CSEXP rhs1, CSEXP body, CSEXP rho)
{
    const int bgn = bodyHasBraces(body);
    const int nm = NAMED(rhs1);
    RCNTXT cntxt;
    PROTECT_INDEX api, vpi;
    volatile int i;             // index into rhs
    SEXPV ans = RNIL;           // init in case loop is not entered
    SEXPV rhs = rhs1;           // volatile copy of rhs
    SEXP v;                     // in loop but does not need to be volatile
    int n;                      // length of right hand side i.e. nbr of iters

    VPROTECT_WITH_INDEX(ans, &api);
    if (isList(rhs1) || isNull(rhs1)) {
        n = length(rhs1);
        PROTECT_WITH_INDEX(v = RNIL, &vpi);  // dummy PROTECT to keep balance
    } else {
        n = LENGTH(rhs1);
        PROTECT_WITH_INDEX(v = allocVector(TYPEOF(rhs1), 1), &vpi);
    }
    begincontext(&cntxt, CTXT_LOOP, RNIL, rho, R_BaseEnv, RNIL, RNIL);
    switch (SETJMP(cntxt.cjmpbuf)) {
    case CTXT_BREAK: goto for_break;
    case CTXT_NEXT: goto for_next;
    }
    for (i = 0; i < n; i++) {
        debugLoop(call, op, args, rho, bgn);
        traceLoopIteration("for1", sym, i);
        switch (TYPEOF(rhs)) {
        case LGLSXP:
            REPROTECT(v = allocVector(TYPEOF(rhs), 1), vpi);
            LOGICAL(v)[0] = LOGICAL(rhs)[i];
            setVar(sym, v, rho);
            break;
        case INTSXP:
            REPROTECT(v = allocVector(TYPEOF(rhs), 1), vpi);
            INTEGER(v)[0] = INTEGER(rhs)[i];
            setVar(sym, v, rho);
            break;
        case REALSXP:
            REPROTECT(v = allocVector(TYPEOF(rhs), 1), vpi);
            REAL(v)[0] = REAL(rhs)[i];
            setVar(sym, v, rho);
            break;
        case CPLXSXP:
            REPROTECT(v = allocVector(TYPEOF(rhs), 1), vpi);
            COMPLEX(v)[0] = COMPLEX(rhs)[i];
            setVar(sym, v, rho);
            break;
        case STRSXP:
            REPROTECT(v = allocVector(TYPEOF(rhs), 1), vpi);
            SET_STRING_ELT(v, 0, STRING_ELT(rhs, i));
            setVar(sym, v, rho);
            break;
        case RAWSXP:
            REPROTECT(v = allocVector(TYPEOF(rhs), 1), vpi);
            RAW(v)[0] = RAW(rhs)[i];
            setVar(sym, v, rho);
            break;
        case EXPRSXP:
        case VECSXP:
            // make sure loop variable is a copy if needed
            if(nm > 0) SET_NAMED(VECTOR_ELT(rhs, i), 2);
            setVar(sym, VECTOR_ELT(rhs, i), rho);
            break;
        case LISTSXP:
            // make sure loop variable is a copy if needed
            if(nm > 0) SET_NAMED(CAR(rhs), 2);
            setVar(sym, CAR(rhs), rho);
            rhs = CDR(rhs);
            break;
        default:
            errorcall(call, _("invalid for() loop sequence"));
        }
        VREPROTECT(ans = eval(body, rho), api);
    for_next:
        ; // needed for strict ISO C compliance, according to gcc 2.95.2
    }
for_break:
    endcontext(&cntxt);
    UNPROTECT(2);
    return (SEXP)ans;       // typecast prevents "discarded qualifier" warning
}

// This big macro is needed for efficient type handling without
// type lookups in the loop.
//
// On the use of R_set_binding_value:
// R_set_binding_value is like setVar but is much faster because it does
// not need to look up the symbol (you pass it the binding location).
// We look up the binding location once at the start of the loop.
// It remains constant thoughout the loop, and this is enforced by
// by marking it as jitted (with setJittedBit).  This prevents the
// user changing the binding location. See jit.c:disallowIfJitting
// and related functions.
// The jitted bit will be cleared by clearJittedBits which is called
// at the end of the jit block or on error.
//
// Note that bodyJitted can change on the fly, see execution of JIT_eval.

#define FORJIT(ftype, FTYPE, FTYPE1)  /* "ftype" for "for type" */  \
{                                                                   \
    ftype * volatile pv = FTYPE(v);                                 \
    const ftype * const pRhs = FTYPE(rhs);                          \
    volatile Rboolean bodyJitted = (TYPEOF(body) == JITSXP);        \
    SEXP loc = findVarLoc(sym, rho);                                \
    setJittedBit(loc);  /* prevent user messing with loop index */  \
                                                                    \
    int jmpVal = SETJMP(cntxt.cjmpbuf);                             \
    if (jmpVal == CTXT_NEXT) {                                      \
        if (!bodyJitted)                                            \
            goto next##FTYPE;                                       \
        else                                                        \
            goto nextj##FTYPE;                                      \
    }                                                               \
    else if (jmpVal == CTXT_BREAK)                                  \
        goto break##FTYPE;                                          \
                                                                    \
    i = 0;                                                          \
    if (!bodyJitted) {                                              \
        /* body is not jitted so standard call to eval */           \
                                                                    \
        for (; i < n; i++) {                                        \
            traceLoopIteration("forjit", sym, i);                   \
            if (NAMED(v)) {                                         \
                VREPROTECT(v = allocVector(FTYPE1, 1), vpi);        \
                pv = FTYPE(v);                                      \
            }                                                       \
            pv[0] = pRhs[i];                                        \
            R_set_binding_value(loc, v);                            \
            eval(body, rho);                                        \
            if (TYPEOF(body) == JITSXP) {                           \
                bodyJitted = TRUE;                                  \
                i++;                                                \
                break;  /* goto for loop below */                   \
            }                                                       \
            next##FTYPE:;                                           \
        }                                                           \
    }                                                               \
                                                                    \
    /* will enter loop below only if the loop body is jitted */     \
                                                                    \
    for (; i < n; i++) {                                            \
        /* body is jitted, so quicker to call evalJit than eval */  \
                                                                    \
        traceLoopIteration("forjit-bodyJitted", sym, i);            \
        if (NAMED(v)) {                                             \
            VREPROTECT(v = allocVector(FTYPE1, 1), vpi);            \
            pv = FTYPE(v);                                          \
        }                                                           \
        pv[0] = pRhs[i];                                            \
        R_set_binding_value(loc, v);                                \
        evalJit(body);                                              \
        nextj##FTYPE:;                                              \
    }                                                               \
    break##FTYPE:;                                                  \
}

// macros needed when loop body is too complicated for FORJIT above

#define FORJIT_BEGIN(FTYPE)                     \
{                                               \
    int jmpVal = SETJMP(cntxt.cjmpbuf);         \
    if (jmpVal == CTXT_NEXT)                    \
        goto next9##FTYPE;                      \
    else if (jmpVal == CTXT_BREAK)              \
        goto break9##FTYPE;                     \
    for (i = 0; i < n; i++) {                   \
        traceLoopIteration("forjit", sym, i);

#define FORJIT_END(FTYPE)                       \
        eval(body, rho);                        \
        next9##FTYPE:;                          \
    }                                           \
    break9##FTYPE:;                             \
}

static SEXP forjit(CSEXP call, CSEXP sym, CSEXP rhs, CSEXP body, CSEXP rho)
{
    PROTECT_INDEX vpi;
    SEXP v;                     // evaluated loop index
    int n;                      // length of right hand side i.e. nbr of iters

    if (isList(rhs) || isNull(rhs)) {
        n = length(rhs);
        PROTECT_WITH_INDEX(v = RNIL, &vpi);  // dummy PROTECT to keep balance
    } else {
        n = LENGTH(rhs);
        PROTECT_WITH_INDEX(v = allocVector(TYPEOF(rhs), 1), &vpi);
    }
    if (n > 0) {
        const int nm = NAMED(rhs);
        volatile int i;         // index into rhs
        RCNTXT cntxt;

        begincontext(&cntxt, CTXT_LOOP, RNIL, rho, R_BaseEnv, RNIL, RNIL);
        setVar(sym, v, rho);

        switch (TYPEOF(rhs)) {
        case LGLSXP:
            FORJIT(int, LOGICAL, LGLSXP);
            break;
        case INTSXP:
            FORJIT(int, INTEGER, INTSXP);
            break;
        case REALSXP:
            FORJIT(double, REAL, REALSXP);
            break;
        case CPLXSXP:
            FORJIT(Rcomplex, COMPLEX, CPLXSXP);
            break;
        case RAWSXP:
            FORJIT(Rbyte, RAW, RAWSXP);
            break;
        case STRSXP:
            {
            CSEXP loc = findVarLoc(sym, rho);
            // set the jitted bit to prevent the user messing with the
            // loop index so we can safely use R_set_binding_value
            setJittedBit(loc);
            FORJIT_BEGIN(STRING);
                if (NAMED(v))
                    VREPROTECT(v = allocVector(STRSXP, 1), vpi);
                SET_STRING_ELT(v, 0, STRING_ELT(rhs, i));
                R_set_binding_value(loc, v);
            FORJIT_END(STRING);
            }
            break;
        case EXPRSXP:
        case VECSXP:
            {
            CSEXP loc = findVarLoc(sym, rho);
            setJittedBit(loc);
            FORJIT_BEGIN(VECSXP);
                if (nm > 0)
                    SET_NAMED(VECTOR_ELT(rhs, i), 2);
                R_set_binding_value(loc, VECTOR_ELT(rhs, i));
            FORJIT_END(VECSXP);
            }
            break;
        case LISTSXP:
            {
            SEXPV rhsv = rhs;           // volatile copy of rhs
            CSEXP loc = findVarLoc(sym, rho);
            setJittedBit(loc);
            FORJIT_BEGIN(LISTSXP);
                if (nm > 0)
                    SET_NAMED(CAR(rhsv), 2);
                R_set_binding_value(loc,  CAR(rhsv));
                rhsv = CDR(rhsv);
            FORJIT_END(LISTSXP);
            }
            break;
        default:
            errorcall(call, _("invalid for() loop sequence"));
        }
        endcontext(&cntxt);
    }
    UNPROTECT(1);
    return RNIL;            // value returned by jitted for is always RNIL
}

SEXP attribute_hidden do_for(SEXP call, SEXP op, SEXP args, SEXP rho)
{
    SEXP ans;
    unsigned prevJitState;
    SEXP rhs;
    CSEXP sym = CAR(args);
    CSEXP body = CADDR(args);
    const int dbg = DEBUG(rho);
    if (!isSymbol(sym))
        errorcall(call, _("non-symbol loop variable"));
    PROTECT(rhs = eval(CADR(args), rho));
    defineVar(sym, RNIL, rho);
    prevJitState = jitEnterLoop(call, body);

    if (inJitBlock())
        PROTECT(ans = forjit(call, sym, rhs, body, rho));
    else
        PROTECT(ans = for1(call, op, args, sym, rhs, body, rho));

    if (prevJitState)
        jitExitLoop(call, prevJitState);
    genJitForPossibly(rhs, sym, body, call, rho);
    SET_DEBUG(rho, dbg);
    UNPROTECT(2);
    return (SEXP)ans;       // typecast prevents "discarded qualifier" warning
}
