/***************************************************************************
 *   Copyright (C) 2009 by fra74   *
 *   francesco.b74@gmail.com   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "schedalistatreni.h"
#include "qviaggiatreno.h"

int SchedaListaTreni::s_count = 0;

SchedaListaTreni::SchedaListaTreni(QViaggiaTreno* parent, const unsigned int intervalloStandard):
        SchedaViaggiaTreno(parent, tsListaTreni, intervalloStandard)
{
    s_count++;
    
    m_stato = statoNuovaScheda;
    m_modificata = false;
    m_titoloLista = "";
    
    //imposta il widget
    m_widget = new WidgetListaTreni(this);
    addWidget(m_widget);
    m_widget->impostaTitolo(titolo(true));
    m_widget->ridimensionaColonne();
    
    //imposta connessioni
    connect(this, SIGNAL(statoCambiato(quint32)), parent, SLOT(aggiornaStatoScheda(quint32)));
    connect(this, SIGNAL(nomeSchedaCambiato(quint32)), parent, SLOT(aggiornaNomeScheda(quint32)));
    connect(this, SIGNAL(apriSchedaStazione(const QString&)), parent, SLOT(nuovaStazione(const QString&)));
    connect(this, SIGNAL(apriSchedaTreno(const QString&)), parent, SLOT(nuovoTreno(const QString&)));
    connect(this, SIGNAL(GuiNonSincronizzata(quint32)), parent, SLOT(sincronizzaGUI(quint32)));
	connect(this, SIGNAL(messaggioStatus(const QString&)), parent, SLOT(mostraMessaggioStatusBar(const QString&)));
}

//restituisce una stringa con il "titolo" di questa scheda, che viene ad esempio usato come etichetta della TabBar
// se titoloItem == true restituisce un titolo per l'item dell'albero
QString SchedaListaTreni::titolo(bool titoloItem) const
{
    if (m_titoloLista.isEmpty())
    {
        if (titoloItem)
			return(QString::fromUtf8("Lista di treni senza titolo #%1").arg(s_count));
        else
			return(QString::fromUtf8("Lista di treni: senza titolo #%1").arg(s_count));
    }
    else
    {
        if (titoloItem)
            return m_titoloLista;
        else
            return(QString::fromUtf8("Lista di treni: %1").arg(m_titoloLista));
    }
}

void SchedaListaTreni::impostaTitolo()
{
    bool ok;
    QString titolo = QInputDialog::getText(this, QString::fromUtf8("Impostare titolo della lista"),
                                           QString::fromUtf8("Inserire il nuovo titolo della lista di treni"),
                                           QLineEdit::Normal, m_titoloLista, &ok);
    if (ok)
    {
        m_titoloLista = titolo;
        m_widget->impostaTitolo(titolo);
        m_modificata = true;
        
        emit GuiNonSincronizzata(idScheda());
        emit nomeSchedaCambiato(idScheda());
    }
    
}


void SchedaListaTreni::avvia()
{
    SchedaViaggiaTreno::avvia();
}

void SchedaListaTreni::ferma()
{
    SchedaViaggiaTreno::ferma();
    
    m_codatreni.clear();
    m_trenoAttuale = "";
    
}

void SchedaListaTreni::aggiorna()
{
    //ottiene una lista di tutti i numeri dei treni che si stanno attualmente controllando
    QStringList listaNumeri = m_listatreni.keys();
    
    //e li aggiunge alla coda
    QStringListIterator it(listaNumeri);
    while (it.hasNext())
        m_codatreni.enqueue(it.next());
    
    //preleva il primo elemento dalla coda e procedi...
    prossimoTreno();
}

void SchedaListaTreni::downloadFinito(QNetworkReply *risposta)
{
	Lista::DatiTreno* treno;
    //procedi solo se nel frattempo non è stato cancellato il numerod el treno attuale
	//questo succede ad esempio se nel frattempo il treno è stato rimosso dalla lista dei treni
	if (m_trenoAttuale != "")
	{
		treno = m_listatreni.value(m_trenoAttuale);
		//preleva il testo della pagina di risposta di VT
		QString rispostaVT = QString::fromUtf8(risposta->readAll());
		//elimina gli spazi e corregge gli errori nei tag
		rispostaVT = correggiOutputVT(rispostaVT);

		//sono possibili 4 risposte
		//1) il numero del treno non è valido
		//2) il treno non è partito ancora
		//3) il treno è in viaggio
		//4) il treno è già arrivato
		//nel caso 1) non è necessario fare il parsing della risposta...
		//verifichiamo quindi se siamo in questa situazione e nel caso aggiorniamo immediatamente il treno
		if (rispostaVT.contains("non valido"))
		{
			treno->cancella();
			treno->impostaStatoTreno(TrenoNonPrevisto);
		}
		else
			analizzaRispostaVT(rispostaVT, treno);

		//aggiorna la riga del treno attuale nel widget
		m_widget->aggiornaTreno(treno);

		emit messaggioStatus(QString::fromUtf8("Aggiornati dati treno %1 della lista '%2' Rimangono %3 su %4 treni da analizzare")
							 .arg(treno->numero()).arg(titolo(true)).arg(m_codatreni.count()).arg(m_listatreni.count()));
	}
    
    //non ci sono altri treni in coda?
    if (m_codatreni.isEmpty())
        //no, cambia stato e memorizza l'ora di fine aggiornamento
    {
		m_trenoAttuale = "";
        m_ultimoAgg = QDateTime::currentDateTime();
		m_widget->impostaAggiornamento(m_ultimoAgg.toString(Qt::DefaultLocaleShortDate));
        cambiaStato(statoMonitoraggioAttivo);
    }
    else
        //si, preleva il treno successivo
        prossimoTreno();

	risposta->deleteLater();
}

//slot
//questo metodo risponde all'attivazione dell'azione per l'apertura di un file

void SchedaListaTreni::apri()
{
	QString filename = QFileDialog::getOpenFileName(this, QString::fromUtf8("Apri lista treni"),
												QDir::homePath(), QString::fromUtf8("File QViaggiaTreno (*.qvt)"));

	if (filename != "")
		apriFile(filename);
}

void SchedaListaTreni::apriFile(const QString& filename)
{
	QFile fileaperto(filename);
	if (!fileaperto.open(QIODevice::ReadOnly))
	{
		//errore durante l'apertura del file
		QMessageBox::critical(this, QString::fromUtf8("Errore apertura file"),
							  QString::fromUtf8("Impossibile aprire il file '%1'").arg(filename));
		return;
	}

	QString errore;
	int riga, col;
	if (!m_dom.setContent(&fileaperto, &errore, &riga, &col))
	{
		//errore nel parsing
		QMessageBox msgBox;
		msgBox.setIcon(QMessageBox::Critical);
		msgBox.setText(QString::fromUtf8("Errore durante la lettura della lista di treni."));
		msgBox.setDetailedText(QString::fromUtf8("Ricevuto messaggio di errore %1 alla riga %2, colonna %3.").arg(errore).arg(riga).arg(col));
		msgBox.exec();
		return;
	}

	//legge il file xml
	QDomElement radice = m_dom.documentElement();
	//verifica che il tag dell'elemento radice sia "qviaggiatreno"
	if (radice.tagName() != "qviaggiatreno")
	{
		QMessageBox::warning(this, QString::fromUtf8("Errore nel file"),
							 QString::fromUtf8("Il file non sembra essere un file dati di QViaggiaTreno"));
		return;
	}

	//in futuro si dovrà verificare la versione del file, ora non è necessario
	if (radice.attribute("tipo") != "lista_treni")
	{
		debugStringa(radice.attribute("tipo"));
		QMessageBox::warning(this, QString::fromUtf8("Errore nel file"),
							 QString::fromUtf8("Il file è un file di dati di QViaggiaTreno,"
											   " ma non sembra contenere una lista treni"));
		return;
	}

	// il file contiene una lista di treni...
	//recupera il titolo
	m_titoloLista = radice.firstChildElement("titolo").text();
	m_widget->impostaTitolo(titolo(true));
	emit nomeSchedaCambiato(idScheda());
	//ottiene una lista di tutti gli elementi con tag name "treno"
	QDomNodeList listaTreni = m_dom.elementsByTagName("treno");
	//scorre la lista ed aggiunge i singoli treni
	for (int i = 0; i < listaTreni.count(); i++)
	{
		aggiungiTreno(listaTreni.at(i).toElement().attribute("numero"));
	}
	fileaperto.close();
	//forza l'aggiornamento
	aggiorna();
	m_modificata = false;
	m_nomefile = filename;
	//sincronizza la GUI
	emit GuiNonSincronizzata(idScheda());
}

//slot
//viene richiamato quando viene scelto da menu o toolbar l'azione epr salvare il file
void SchedaListaTreni::salva()
{
	if (m_nomefile != "")
		salvaFile(m_nomefile);
	else
		salvaConNome();

}

void SchedaListaTreni::salvaConNome()
{
	QString filename = QFileDialog::getSaveFileName(this, QString::fromUtf8("Salva file con nome"),
													QDir::homePath(), QString::fromUtf8("File QViaggiaTreno (*.qvt)"));
	if (filename != "")
		salvaFile(filename);

}

void  SchedaListaTreni::salvaFile(const QString& filename)
{
	//prova ad aprire il file
	QFile fileDaSalvare(filename);
	//l'apertura non è andata a buon fine
	if (!fileDaSalvare.open(QIODevice::WriteOnly))
	{
		QMessageBox::critical(this, QString::fromUtf8("Errore salvataggio file"),
							  QString::fromUtf8("Errore nell'apertura in scrittura del file %1").arg(filename));
		return;
	}

	//cancella il documento dom
	m_dom = QDomDocument("qviaggiatreno");
	//crea l'elemento radice
	QDomElement radice = m_dom.createElement("qviaggiatreno");
	m_dom.appendChild(radice);
	//imposta versione e tipo file
	radice.setAttribute("versione", "1");
	radice.setAttribute("tipo", "lista_treni");
	//aggiunge la radice al documento


	//crea l'elemento per il titolo
	QDomElement titolo = m_dom.createElement("titolo");
	QDomText testo = m_dom.createTextNode(m_titoloLista);
	titolo.appendChild(testo);
	radice.appendChild(titolo);
	//scorre la lista dei treni monitorati
	QDomElement lista = m_dom.createElement("listatreni");
	radice.appendChild(lista);
	QStringListIterator it(m_listatreni.keys());
	while (it.hasNext())
	{
		QString numero = it.next();
		QDomElement elementoTreno = m_dom.createElement("treno");
		elementoTreno.setAttribute("numero", numero);
		lista.appendChild(elementoTreno);
	}

	//salva l'albero DOM
	QTextStream stream(&fileDaSalvare);
	m_dom.save(stream, 4);
	fileDaSalvare.close();

	//cambia nome file
	m_nomefile = filename;
	m_modificata = false;
	emit GuiNonSincronizzata(idScheda());
}

//aggiunge un singolo treno alla lista dei treni da monitorare
void SchedaListaTreni::aggiungiTreno(const QString& numero)
{
    //verifica se si sta già controllando questo treno,
    //in caso positivo esci dalla funzione senza fare niente
    if (m_listatreni.contains(numero))
        return;
    
	//crea una nuova istanza di Lista::DatiTreno per memorizzare i dati del treno
	Lista::DatiTreno *treno = new Lista::DatiTreno(numero);
    //lo aggiunge alla mappa dei treni
    m_listatreni.insert(numero, treno);
    //e lo aggiunge al widget
    m_widget->aggiungiTreno(treno);
    
}

//rimuove un singolo treno alla lista dei treni da monitorare
void SchedaListaTreni::rimuoviTreno(const QString& numero)
{
	//rimuove il treno dalla lista dei treni
	m_listatreni.remove(numero);

	//verifica che non sia nella coda dei treni, se c'è rimuovilo
	m_codatreni.removeAll(numero);
	if (m_trenoAttuale == numero)
		m_trenoAttuale = "";

	m_widget->rimuoviTreno(numero);
}

//slot
//questo slot è connesso all'azione per aggiungere treni
void SchedaListaTreni::aggiungiTreni()
{
    DialogoAggiuntaTreni dialogo(this);
    if (dialogo.exec() == QDialog::Accepted)
    {
        if (!dialogo.listaVuota())
        {
            QStringListIterator it(dialogo.listaTreni());
            while (it.hasNext())
                aggiungiTreno(it.next());
        }
    }
    
    //sono stati aggiunti tutti i treni, aggiornare la larghezza delle colonne
    m_widget->ridimensionaColonne();

	m_modificata = true;

    //c'è stato un cambiamento nella scheda che ha influenza sull'aspetto della GUI
    //va quindi emesso il segnale per sincronizzare la gui
    emit(GuiNonSincronizzata(idScheda()));

	//forza un aggiornamento immediato dei treni
	aggiorna();
    
}

void SchedaListaTreni::rimuoviTreni()
{
	DialogoRimozioneTreni dialogo(m_listatreni.keys(), this);
	if (dialogo.exec() == QDialog::Accepted)
	{
		QStringListIterator it(dialogo.listaTreni());
		while (it.hasNext())
			rimuoviTreno(it.next());
	}

	m_modificata = true;

	//c'è stato un cambiamento nella scheda che ha influenza sull'aspetto della GUI
	//va quindi emesso il segnale per sincronizzare la gui
	emit(GuiNonSincronizzata(idScheda()));
}

//slot
//questo slot è connesso all'azione per rimuovere tutti i treni dalla lista
void SchedaListaTreni::rimuoviTuttiITreni()
{
	//rimuovi tutti i treni dalla tabella
	m_widget->rimuoviTuttiITreni();

	//rimuovi tutti i treni dalla coda
	m_codatreni.clear();
	m_trenoAttuale = "";

	//elimina la lista di treni
	m_listatreni.clear();

	m_modificata = true;

	//sincronizza la gui
	emit(GuiNonSincronizzata(idScheda()));
}

//questa funziona preleva il prossimo treno dalla coda dei treni
//e richiede i dati a viaggiatreno
void SchedaListaTreni::prossimoTreno()
{
    //verifica che ci siano elementi nella coda, altrimenti esci
    if (m_codatreni.isEmpty())
        return;
    
    m_trenoAttuale = m_codatreni.dequeue();
    
    QNetworkRequest request;
    QString dati;
    //costruisce i dati da inviare al server sottoforma di richiesta POST
    dati = QString("numeroTreno=%1&tipoRicerca=numero&lang=IT").arg(m_trenoAttuale);
    request.setUrl(QUrl("http://mobile.viaggiatreno.it/viaggiatreno/mobile/numero"));
    request.setRawHeader("Content-type", "application/x-www-form-urlencoded");
    networkAM()->post(request, dati.toUtf8());
}

void SchedaListaTreni::analizzaRispostaVT(const QString& risposta, Lista::DatiTreno *treno)
{
	//QMessageBox::information(this, "Funzione non implementata", QString("Analisi risposta di viaggiatreno per il treno %1").arg(treno->numero()));
	QDomDocument documentoDOM;
	int riga, colonna, idx;
	QString errore, temp;

	treno->cancella();

	//analizza il testo della risposta di viaggiatreno
	if (!documentoDOM.setContent(risposta, &errore, &riga, &colonna ))
	{
		//errore nel parsing
		cambiaStato(statoErrore);
		QMessageBox msgBox;
		msgBox.setIcon(QMessageBox::Warning);
		msgBox.setText(QString::fromUtf8("Errore durante l'analisi della risposta di ViaggiaTreno. Se il problema persiste contattare l'autore"));
		msgBox.setDetailedText(QString::fromUtf8("Ricevuto messaggio di errore %1 alla riga %2, colonna %3.\nTesto:\n %4").arg(errore).arg(riga).arg(colonna).arg(risposta));
		msgBox.exec();
		treno->impostaStatoTreno(DatiSconosciuti);
		return;
	}

	//inizia l'analisi dell'albero DOM

	//rintraccia il tag body
	QDomElement body = documentoDOM.documentElement().firstChildElement("body");
	//rintraccia il tag H1 che contiene categoria e numero treno
	temp = body.firstChildElement("h1").text();
	treno->impostaDato(Lista::dtCategoria, temp.section(' ', 0, 0));
	//ottiene una lista di tutti gli elementi div
	QDomNodeList elementiDiv = body.elementsByTagName("div");

	QDomElement div;
	//imposta lo stato del treno
	if (risposta.contains("ancora partito"))
		treno->impostaStatoTreno(TrenoNonPartito);

	if (risposta.contains("viaggia "))
		treno->impostaStatoTreno(TrenoInViaggio);

	if (risposta.contains("arrivato"))
		treno->impostaStatoTreno(TrenoArrivato);

	//rintraccia l'elemento DIV con i dati della stazione di partenza
		for (int i = 0; i < elementiDiv.count(); i++)
			if (elementiDiv.at(i).toElement().text().contains("Partenza"))
				div = elementiDiv.at(i).toElement();

		//estrai i dati sulla stazione di partenza dall'elemento appena ottenuto
		treno->impostaDato(Lista::dtOrigine, div.firstChildElement("h2").text());
		treno->impostaDato(Lista::dtPartenzaProgrammata, div.firstChildElement("p").firstChildElement("strong").text());
		//non cercare l'orario di partenza effettivo se il treno risulta non ancora partito
		if (!risposta.contains("ancora partito"))
			treno->impostaDato(Lista::dtPartenzaEffettiva, div.firstChildElement("p").nextSiblingElement("p")
							   .firstChildElement("strong").text());

		//rintraccia l'elemento DIV con i dati della stazione di arrivo
		for (int i = 0; i < elementiDiv.count(); i++)
			if (elementiDiv.at(i).toElement().text().contains("Arrivo"))
				div = elementiDiv.at(i).toElement();

		//estrai i dati sulla stazione di arrivo dall'elemento appena ottenuto
		treno->impostaDato(Lista::dtDestinazione, div.firstChildElement("h2").text());
		treno->impostaDato(Lista::dtArrivoProgrammato, div.firstChildElement("p").firstChildElement("strong").text());
		//cerca l'orario di arrivo effettivo solo se il treno risulta già arrivato
		if (risposta.contains("arrivato"))
			treno->impostaDato(Lista::dtArrivoEffettivo, div.firstChildElement("p").nextSiblingElement("p")
							   .firstChildElement("strong").text());

		//cerca solo per i treni ancora in viaggio il dato sull'ultima fermata effettuata
		if (risposta.contains("viaggia "))
		{
			idx = -1;
			for (int i = 0; i < elementiDiv.count(); i++)
				if (elementiDiv.at(i).toElement().text().contains("Ultima fermata effettuata"))
					idx = i+1;
			//è stata trovato? allora estrai i dati
			if (idx != -1)
			{
				div = elementiDiv.at(idx).toElement();

				treno->impostaDato(Lista::dtUltimaFermata, div.firstChildElement("h2").text());
				treno->impostaDato(Lista::dtOrarioFermataProgrammato, div.firstChildElement("p").firstChildElement("strong").text());
				treno->impostaDato(Lista::dtOrarioFermataEffettivo, div.firstChildElement("p").nextSiblingElement("p")
								   .firstChildElement("strong").text());
			}
		}

		//cerca il ritardo in arrivo per un treno già arrivato
		if (risposta.contains("arrivato"))
		{
			idx = -1;
			for (int i = 0; i < elementiDiv.count(); i++)
				if (elementiDiv.at(i).toElement().text().contains("Il treno e' arrivato"))
					idx = i;
			if (idx != -1)
			{
				div = elementiDiv.at(idx).toElement();
				//il treno è arrivato in orario, non c'è ragione di estrarre il ritardo
				if (div.text().contains("orario"))
					treno->impostaDato(Lista::dtRitardoTransito, QString::fromUtf8("In orario"));
				else
				{
					QString minuti = div.text().section(' ', 6, 6);
					QString ritOAnticipo = div.text().section(' ', 9, 9);
					if (minuti == "1")
						temp = QString::fromUtf8("1 minuto in %1").arg(ritOAnticipo);
					else
						temp = QString::fromUtf8("%1 minuti in %2").arg(minuti).arg(ritOAnticipo);

					treno->impostaDato(Lista::dtRitardoTransito, temp);
				}
			}
		}

		//cerca il ritardo per i treni ancora in viaggio
		if (risposta.contains("viaggia "))
		{
			idx = -1;
			for (int i = 0; i < elementiDiv.count(); i++)
				if (elementiDiv.at(i).toElement().text().contains("Il treno viaggia"))
					idx = i;


			if (idx != -1)
			{
				QString ritardo, transito;
				div = elementiDiv.at(idx).toElement();
				temp = div.firstChildElement("strong").text();
				idx = temp.indexOf("Ultimo");
				if (idx == -1)
				{
					ritardo = temp;
					transito = "";
				}
				else
				{
					ritardo = temp.left(idx);
					transito = temp.mid(idx);
				}

				//esistono i dati sull'ora e la località dell'ultimo transito
				if (transito != "")
				{
					treno->impostaDato(Lista::dtOrarioTransito, transito.section(' ', -2, -2));
					treno->impostaDato(Lista::dtUltimoRilevamento, transito.section(' ', 3, -5));
				}

				if (ritardo.contains("orario"))
					treno->impostaDato(Lista::dtRitardoTransito, QString::fromUtf8("in orario"));
				else
				{
					temp = ritardo.section(' ', 5, 5);
					if (temp == "1")
						treno->impostaDato(Lista::dtRitardoTransito, QString::fromUtf8("1 minuto in %1").arg(ritardo.section(' ', 8, 8)));
					else
						treno->impostaDato(Lista::dtRitardoTransito, QString::fromUtf8("%1 minuti in %2").arg(temp).arg(ritardo.section(' ', 8, 8)));
				}
			}
		}
}

Lista::DatiTreno::DatiTreno(const QString& numero)
{
    m_numero = numero;
    
    m_stato = DatiSconosciuti;
}

//cancella tutti i dati del treno tranne numero
void Lista::DatiTreno::cancella()
{
	m_dati.clear();
	m_stato = DatiSconosciuti;
}

//restituisce una stringa con lo stato del treno
QString Lista::DatiTreno::stringaStatoTreno() const
{
    QString stringa;
    switch(m_stato)
    {
                case DatiSconosciuti: stringa = QString::fromUtf8("Stato sconosciuto"); break;
                case TrenoNonPrevisto: stringa = QString::fromUtf8("Non previsto oggi"); break;
                case TrenoNonPartito: stringa = QString::fromUtf8("Non ancora partito"); break;
                case TrenoInViaggio: stringa = QString::fromUtf8("In viaggio"); break;
                case TrenoArrivato: stringa = QString::fromUtf8("Arrivato"); break;
                }
    
    return stringa;
}

void SchedaListaTreni::salvaScheda(QSettings& settings)
{
	//la scheda è modificata, proponi di salvare il contenuto
	if (modificata())
	{

	int risposta = QMessageBox::warning(this, QString::fromUtf8("La scheda è modificata"),
						 QString::fromUtf8("La scheda <b>'%1'</b> è stata modificata, ma la lista di treni non è ancora stata salvata e non ne sarà possibile il ripristino.<br>"
										   "Si desidera salvare la lista?").arg(titolo(true)), QMessageBox::Yes, QMessageBox::No);

	if (risposta == QMessageBox::Yes)
		salva();
	}

	//salva semplicemente il nome del file
	settings.setValue("tipo scheda", "lista treni");
	settings.setValue("nome file", m_nomefile);
}
