


#include <stdio.h>
#include <gwenhywfar/xml.h>
#include <gwenhywfar/db.h>
#include <gwenhywfar/debug.h>
#include <gwenhywfar/text.h>

#include <ctype.h>



GWEN_DB_NODE *findAccount(GWEN_DB_NODE *dbAccounts, const char *id) {
  GWEN_DB_NODE *dbT;

  dbT=GWEN_DB_GetGroup(dbAccounts, GWEN_PATH_FLAGS_NAMEMUSTEXIST,
                       "categories");
  if (!dbT) {
    return 0;
  }
  dbT=GWEN_DB_FindFirstGroup(dbT, "category");
  while(dbT) {
    const char *s;
    GWEN_DB_NODE *dbC;

    s=GWEN_DB_GetCharValue(dbT, "id", 0, 0);
    if (s && strcasecmp(s, id)==0) {
      return dbT;
    }
    dbC=findAccount(dbT, id);
    if (dbC)
      return dbC;

    dbT=GWEN_DB_FindNextGroup(dbT, "category");
  }

  return 0;
}



char *getCharValue(GWEN_XMLNODE *n, const char *name,
                   char *buffer, unsigned int size) {
  GWEN_BUFFER *nbuf;
  GWEN_BUFFER *obuf;

  nbuf=GWEN_Buffer_new(0, 256, 0, 1);
  n=GWEN_XMLNode_FindFirstTag(n, name, 0, 0);
  if (n) {
    n=GWEN_XMLNode_GetFirstData(n);
    while(n) {
      const char *s;

      s=GWEN_XMLNode_GetData(n);
      if (GWEN_Buffer_GetUsedBytes(nbuf))
        GWEN_Buffer_AppendString(nbuf, " ");
      GWEN_Buffer_AppendString(nbuf, s);
      n=GWEN_XMLNode_GetNextData(n);
    }
  }
  obuf=GWEN_Buffer_new(0, 256, 0, 1);
  GWEN_Text_UnescapeXmlToBuffer(GWEN_Buffer_GetStart(nbuf),
                                obuf);
  GWEN_Buffer_free(nbuf);
  assert(GWEN_Buffer_GetUsedBytes(obuf)<size-1);
  strcpy(buffer, GWEN_Buffer_GetStart(obuf));
  GWEN_Buffer_free(obuf);
  return buffer;
}


void toUtf8(const char *s, GWEN_BUFFER *buf) {
  const char *p;

  p=s;
  while(*p) {
    if (*p=='&') {
      if (p[1]=='#') {
        unsigned char num=0;

        p+=2;
        while(isdigit(*p)) {
          num*=10;
          num+=(*p)-'0';
          p++;
        }
        p++;
        switch(num) {
        case 0xc4: /* AE */
          GWEN_Buffer_AppendByte(buf, 0xc3);
          GWEN_Buffer_AppendByte(buf, 0x84);
          break;
    
        case 0xe4: /* ae */
          GWEN_Buffer_AppendByte(buf, 0xc3);
          GWEN_Buffer_AppendByte(buf, 0xa4);
          break;
    
        case 0xd6: /* OE */
          GWEN_Buffer_AppendByte(buf, 0xc3);
          GWEN_Buffer_AppendByte(buf, 0x96);
          break;
    
        case 0xf6: /* oe */
          GWEN_Buffer_AppendByte(buf, 0xc3);
          GWEN_Buffer_AppendByte(buf, 0xb6);
          break;
    
        case 0xdc: /* UE */
          GWEN_Buffer_AppendByte(buf, 0xc3);
          GWEN_Buffer_AppendByte(buf, 0x9c);
          break;
    
        case 0xfc: /* ue */
          GWEN_Buffer_AppendByte(buf, 0xc3);
          GWEN_Buffer_AppendByte(buf, 0xbc);
          break;
    
        case 0xdf: /* sz */
          GWEN_Buffer_AppendByte(buf, 0xc3);
          GWEN_Buffer_AppendByte(buf, 0x9f);
          break;
    
        case 0xa7: /* section sign */
          GWEN_Buffer_AppendByte(buf, 0xc3);
          GWEN_Buffer_AppendByte(buf, 0x67);
          break;
    
          /* english chars */
        case 0xa3: /* pound swign */
          GWEN_Buffer_AppendByte(buf, 0xc3);
          GWEN_Buffer_AppendByte(buf, 0x63);
          break;
    
          /* french chars */
        case 0xc7: /* C cedille */
          GWEN_Buffer_AppendByte(buf, 0xc3);
          GWEN_Buffer_AppendByte(buf, 0x87);
          break;
    
        case 0xe0: /* a accent grave */
          GWEN_Buffer_AppendByte(buf, 0xc3);
          GWEN_Buffer_AppendByte(buf, 0xa0);
          break;
    
        case 0xe1: /* a accent aigu */
          GWEN_Buffer_AppendByte(buf, 0xc3);
          GWEN_Buffer_AppendByte(buf, 0xa1);
          break;
    
        case 0xe2: /* a accent circumflex */
          GWEN_Buffer_AppendByte(buf, 0xc3);
          GWEN_Buffer_AppendByte(buf, 0xa2);
          break;
    
        case 0xe7: /* c cedille */
          GWEN_Buffer_AppendByte(buf, 0xc3);
          GWEN_Buffer_AppendByte(buf, 0xa7);
          break;
    
        case 0xe8: /* e accent grave */
          GWEN_Buffer_AppendByte(buf, 0xc3);
          GWEN_Buffer_AppendByte(buf, 0xa8);
          break;
    
        case 0xe9: /* e accent aigu */
          GWEN_Buffer_AppendByte(buf, 0xc3);
          GWEN_Buffer_AppendByte(buf, 0xa9);
          break;
    
        case 0xea: /* e accent circumflex */
          GWEN_Buffer_AppendByte(buf, 0xc3);
          GWEN_Buffer_AppendByte(buf, 0xaa);
          break;
    
        case 0xec: /* i accent grave (never heard of this) */
          GWEN_Buffer_AppendByte(buf, 0xc3);
          GWEN_Buffer_AppendByte(buf, 0xac);
          break;
    
        case 0xed: /* i accent aigu (never heard of this, either) */
          GWEN_Buffer_AppendByte(buf, 0xc3);
          GWEN_Buffer_AppendByte(buf, 0xad);
          break;
    
        case 0xee: /* i accent circumflex (never heard of this, either) */
          GWEN_Buffer_AppendByte(buf, 0xc3);
          GWEN_Buffer_AppendByte(buf, 0xae);
          break;
    
        case 0xf2: /* o accent grave */
          GWEN_Buffer_AppendByte(buf, 0xc3);
          GWEN_Buffer_AppendByte(buf, 0xb2);
          break;
    
        case 0xf3: /* o accent aigu */
          GWEN_Buffer_AppendByte(buf, 0xc3);
          GWEN_Buffer_AppendByte(buf, 0xb3);
          break;
    
        case 0xf4: /* o accent circumflex */
          GWEN_Buffer_AppendByte(buf, 0xc3);
          GWEN_Buffer_AppendByte(buf, 0xb4);
          break;
    
        case 0xf9: /* u accent grave */
          GWEN_Buffer_AppendByte(buf, 0xc3);
          GWEN_Buffer_AppendByte(buf, 0xb9);
          break;
    
        case 0xfa: /* u accent aigu */
          GWEN_Buffer_AppendByte(buf, 0xc3);
          GWEN_Buffer_AppendByte(buf, 0xba);
          break;
    
        case 0xfb: /* u accent circumflex */
          GWEN_Buffer_AppendByte(buf, 0xc3);
          GWEN_Buffer_AppendByte(buf, 0xbb);
          break;
    
        default:
          GWEN_Buffer_AppendByte(buf, '?');
        }

      }
      else {
        GWEN_Buffer_AppendByte(buf, *p);
        p++;
      }
    }
    else {
      GWEN_Buffer_AppendByte(buf, *p);
      p++;
    }
  }
}



void setCharValue(GWEN_DB_NODE *db,
                  uint32_t flags,
                  const char *name, const char *val) {
  GWEN_BUFFER *nbuf;

  nbuf=GWEN_Buffer_new(0, 256, 0, 1);
  toUtf8(val, nbuf);
  GWEN_DB_SetCharValue(db, flags, name,
                       GWEN_Buffer_GetStart(nbuf));
  GWEN_Buffer_free(nbuf);
}



int scanAccounts(GWEN_XMLNODE *node, GWEN_DB_NODE *dbAccounts) {
  GWEN_XMLNODE *n;
  int notAdded=0;
  int added=0;

  n=GWEN_XMLNode_FindFirstTag(node, "gnc:account", 0, 0);
  while (n) {
    const char *id;
    const char *name;
    const char *typ;
    const char *descr;
    GWEN_XMLNODE *next;
    GWEN_DB_NODE *dbCats;
    char idBuffer[256];
    char nameBuffer[256];
    char typBuffer[256];
    char descrBuffer[1024];

    next=GWEN_XMLNode_FindNextTag(n, "gnc:account", 0, 0);

    dbCats=GWEN_DB_GetGroup(dbAccounts, GWEN_DB_FLAGS_DEFAULT,
                            "categories");
    assert(dbCats);
    id=getCharValue(n, "act:id", idBuffer, sizeof(idBuffer));
    name=getCharValue(n, "act:name", nameBuffer, sizeof(nameBuffer));
    typ=getCharValue(n, "act:type", typBuffer, sizeof(typBuffer));
    descr=getCharValue(n, "act:description", descrBuffer, sizeof(descrBuffer));

    if (id && *id && name && *name && typ && *typ) {
      const char *parentId;
      int useThis=1;
      GWEN_DB_NODE *parentNode=dbCats;

      parentId=GWEN_XMLNode_GetCharValue(n, "act:parent", 0);
      if (parentId && *parentId) {
        parentNode=findAccount(dbAccounts, parentId);
        if (!parentNode)
          useThis=0;
        else {
          dbCats=GWEN_DB_GetGroup(parentNode, GWEN_DB_FLAGS_DEFAULT,
                                  "categories");
        }
      }
      if (useThis) {
        GWEN_DB_NODE *newNode;

        newNode=GWEN_DB_GetGroup(dbCats, GWEN_PATH_FLAGS_CREATE_GROUP,
                                 "category");
        assert(newNode);
        setCharValue(newNode, GWEN_DB_FLAGS_OVERWRITE_VARS,
                     "id", id);
        setCharValue(newNode, GWEN_DB_FLAGS_OVERWRITE_VARS,
                     "name", name);
        setCharValue(newNode, GWEN_DB_FLAGS_OVERWRITE_VARS,
                     "typ", typ);
        setCharValue(newNode, GWEN_DB_FLAGS_OVERWRITE_VARS,
                     "description", descr);

        if (strcasecmp(typ, "income")==0)
          GWEN_DB_SetIntValue(newNode, GWEN_DB_FLAGS_OVERWRITE_VARS,
                              "isIncome", 1);
        else if (strcasecmp(typ, "expense")==0) {
          GWEN_DB_SetIntValue(newNode, GWEN_DB_FLAGS_OVERWRITE_VARS,
                              "isIncome", 0);
        }
        GWEN_XMLNode_UnlinkChild(GWEN_XMLNode_GetParent(n), n);
        GWEN_XMLNode_free(n);
        added++;
      }
      else {
        notAdded++;
      }
    }
    n=next;
  } /* while */

  DBG_ERROR(0, "Added %d, not added %d", added, notAdded);

  if (notAdded) {
    if (!added)
      return -1;
    else
      return 1;
  }
  return 0;
}



void removeUnneeded(GWEN_DB_NODE *db) {
  GWEN_DB_NODE *dbT;

  dbT=GWEN_DB_GetGroup(db, GWEN_PATH_FLAGS_NAMEMUSTEXIST, "categories");
  if (dbT) {
    dbT=GWEN_DB_FindFirstGroup(dbT, "category");
    while(dbT) {
      GWEN_DB_NODE *dbN;
      const char *s;
      int del=0;

      dbN=GWEN_DB_FindNextGroup(dbT, "category");
      s=GWEN_DB_GetCharValue(dbT, "typ", 0, 0);
      if (s) {
        if ((strcasecmp(s, "income")!=0) &&
            (strcasecmp(s, "expense")!=0))
          del=1;
      }
      else {
        DBG_ERROR(0, "No type:");
        GWEN_DB_Dump(dbT, stderr, 2);
      }
      if (del) {
        GWEN_DB_UnlinkGroup(dbT);
        GWEN_DB_Group_free(dbT);
      }
      else
        removeUnneeded(dbT);
      dbT=dbN;
    }
  }
}



void assignId(GWEN_DB_NODE *db, int *id) {
  GWEN_DB_NODE *dbT;

  dbT=GWEN_DB_GetGroup(db, GWEN_PATH_FLAGS_NAMEMUSTEXIST, "categories");
  if (dbT) {
    dbT=GWEN_DB_FindFirstGroup(dbT, "category");
    while(dbT) {
      char numbuf[32];

      snprintf(numbuf, sizeof(numbuf), "%d", (*id)++);
      GWEN_DB_SetCharValue(dbT, GWEN_DB_FLAGS_OVERWRITE_VARS, "id", numbuf);
      GWEN_DB_DeleteVar(dbT, "typ");
      assignId(dbT, id);
      dbT=GWEN_DB_FindNextGroup(dbT, "category");
    }
  }
}



int main(int argc, char **argv) {
  GWEN_DB_NODE *dbAccounts;
  GWEN_XMLNODE *n;
  int rv;
  const char *fileName;
  int i;
  GWEN_XMLNODE *nRoot;
  GWEN_XMLNODE *nV2;

  if (argc<2) {
    fprintf(stderr, "Usage: %s INFILE1 ... INFILEn\n", argv[0]);
    return 1;
  }

  nRoot=GWEN_XMLNode_new(GWEN_XMLNodeTypeTag, "root");
  nV2=GWEN_XMLNode_new(GWEN_XMLNodeTypeTag, "gnc-v2");
  GWEN_XMLNode_AddChild(nRoot, nV2);
  dbAccounts=GWEN_DB_Group_new("accounts");
  for (i=1; i<argc; i++) {
    GWEN_XMLNODE *nTemp;
    GWEN_XMLNODE *nTv2;

    fileName=argv[i];

    nTemp=GWEN_XMLNode_new(GWEN_XMLNodeTypeTag, "temp");
    rv=GWEN_XML_ReadFile(nTemp, fileName,
                         GWEN_XML_FLAGS_DEFAULT |
                         GWEN_XML_FLAGS_HANDLE_HEADERS);
    if (rv) {
      DBG_ERROR(0, "Error reading file \"%s\"", fileName);
      return 2;
    }
    nTv2=GWEN_XMLNode_FindFirstTag(nTemp, "gnc-v2", 0, 0);
    if (nTv2) {
      GWEN_XMLNode_AddChildrenOnly(nV2, nTv2, 0);
    }
    else {
      DBG_ERROR(0, "gnx-v2 not found:");
      GWEN_XMLNode_Dump(nTemp, stderr, 2);
    }
    GWEN_XMLNode_free(nTemp);
  }

  GWEN_XMLNode_WriteFile(nRoot, "out.xml", GWEN_XML_FLAGS_DEFAULT);

  n=GWEN_XMLNode_FindFirstTag(nRoot, "gnc-v2", 0, 0);
  if (n) {
    int j;
    for (j=0;j<100;j++) {
      fprintf(stderr, "Try %d\n", j);
      rv=scanAccounts(n, dbAccounts);
      if (rv==-1) {
        DBG_ERROR(0, "Error scanning accounts");
        return 3;
      }
      else if (rv==0)
        break;
    }
  }

  i=1;
  removeUnneeded(dbAccounts);
  assignId(dbAccounts, &i);
  if (GWEN_DB_WriteFile(dbAccounts, "out.db",
			GWEN_DB_FLAGS_DEFAULT,
			2000, 0)) {
    DBG_ERROR(0, "Could not write file.\n");
    return 4;
  }

  return 0;
}

