#-*- coding:utf-8 -*-

#  Pybik -- A 3 dimensional magic cube game.
#  Copyright © 2009-2012  B. Clausius <barcc@gmx.de>
#
#  This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.


# Ported from GNUbik
# Original filename: widget-set.c
# Original copyright and license: 2003, 2004  John Darrington, GPL3+


import os

import gobject
import gtk
from gtk import gdk

from .debug import *
from . import config
from . import dialogs
from . import dialogcolors
from . import plugins
import state

from . import glarea
from .confstore import confstore


class Application (object):
    def __init__(self):
        # create cube state
        self.solved = False
        self.current_movement = state.CurrentMovement()
        
        #
        confstore.init_notify()
        self.plugin_helper = plugins.PluginHelper(self)
        
        # UI
        ui_file = os.path.join(config.UI_DIR, 'pybik.ui')
        self.ui_xml = gtk.Builder()
        self.ui_xml.props.translation_domain = config.PACKAGE
        self.ui_xml.add_from_file(ui_file)
        debug('Connect signals:', ui_file)
        missing = self.ui_xml.connect_signals(self)
        if missing:
            debug(' ', len(missing), 'missing handlers:')
            for m in missing:
                debug('    def', m+'(self, unused_widget, *unused_args):')
                debug('        pass')
        else:
            debug('  no missing handlers')
        
        self.window = self.ui_xml.get_object('window_main')
        self.statusbar = self.ui_xml.get_object('statusbar_main')
        self.entry_formula = self.ui_xml.get_object('entry_formula')
        self.treeview = self.ui_xml.get_object('treeview_scripts')
        self.selected_tree_path = None
        
        self.window.set_icon_from_file(config.IMAGE_FILE)
        self.window.connect("delete-event", self.on_window_delete_event)
        
        self.create_menu()
        self.create_sidepane()
        
        self.cube_area = glarea.CubeArea()
        drawingarea_cube = self.ui_xml.get_object('drawingarea_cube')
        self.cube_area.init(drawingarea_cube)
        self.cube_area.connect('texture-changed', self.on_cubearea_texture_changed)
        self.cube_area.connect('end-animation', self.on_cubearea_end_animation)
        self.cube_area.connect('request-rotation', self.on_cubearea_request_rotation)
        
        if not os.path.exists(config.get_data_home()):
            os.mkdir(config.get_data_home())
        accel_map_file = os.path.join(config.get_data_home(), 'accels')
        gtk.accel_map_load(accel_map_file)
        gtk.accel_map_get().connect('changed', self.on_accel_map_changed, accel_map_file)
        
    def on_accel_map_changed(self, unused_map, unused_path, unused_key, unused_mods, filename):
        gtk.accel_map_save(filename)
        
    def on_cubearea_texture_changed(self, unused_widget):
        if dialogcolors.ColorSelectDialog.this_dlg:
            dialogcolors.ColorSelectDialog.this_dlg.redraw_swatches()
    def on_cubearea_end_animation(self, unused_cubearea, last_move):
        self.cube_area.apply_to_glmodel(self.current_movement.current_cube_state)
        self.cube_area.update_selection()
        
        if not last_move:
            # go again
            self.update_statusbar()
            move_data = self.current_movement.request_play()
            if not self.start_animation(move_data, stop_after=False):
                self.update_ui()
        else:
            self.update_ui()
    def on_cubearea_request_rotation(self, unused_cubearea, maxis, mslice, mdir):
        self.current_movement.request_rotation(maxis, mslice, mdir)
        self.update_statusbar()
        move_data = self.current_movement.request_play()
        self.start_animation(move_data, stop_after=False)
        
    def create_menu(self):
        ui_file = os.path.join(config.UI_DIR, 'menu.ui')
        
        self.main_actions = self.ui_xml.get_object('actiongroup_main')
        self.play_actions = self.ui_xml.get_object('actiongroup_play')
        
        self.uimanager = gtk.UIManager()
        self.uimanager.insert_action_group(self.main_actions)
        self.uimanager.insert_action_group(self.play_actions)
        self.uimanager.add_ui_from_file(ui_file)
        
        menubar = self.uimanager.get_widget('/menubar1')
        vbox = self.ui_xml.get_object('vbox1')
        vbox.pack_start(menubar, fill=False, expand=False)
        vbox.reorder_child(menubar, 0)
        
        accel = self.uimanager.get_accel_group()
        self.window.add_accel_group(accel)
    
        self.playbarstate = PlaybarState(self)
        self.playbarstate.empty()
        
    @debug_func
    def create_sidepane(self):
        column = gtk.TreeViewColumn(_('Name'), gtk.CellRendererText(), text=0)
        column.set_sizing(gtk.TREE_VIEW_COLUMN_FIXED)
        column.set_fixed_width(200)
        self.treeview.append_column(column)
        
        self.add_scripts_to_sidepane()
        
        self.treeview.connect('row-activated', self.on_treeview_row_activated)
        self.id_treeview_motion_notify_event = None
        active = confstore.dynamic_script_selection
        self.ui_xml.get_object('toggleaction_dynamic_script_selection').set_active(active)
        #self.treeview.expand_all()
        self.treeview.show_all()
        
    def add_scripts_to_sidepane(self):
        treestore = gtk.TreeStore(gobject.TYPE_STRING, gobject.TYPE_PYOBJECT)
        plugins = self.plugin_helper.load_plugins()
        tree_iters = {():None}
        for plugin_path, func in plugins:
            for i in xrange(1, 5):
                p_i = plugin_path[:i]
                if p_i not in tree_iters:
                    name = _(p_i[-1])
                    tree_iters[p_i] = treestore.append(
                                        tree_iters[plugin_path[:i-1]],
                                        [name, func if p_i == plugin_path else None])
        self.treeview.set_model(treestore)
        
    def is_animating(self):
        return self.cube_area.animation_in_progress
        
    def on_treeview_motion_notify_event(self, treeview, event):
        tpath_column = treeview.get_path_at_pos(int(event.x), int(event.y))
        if tpath_column is None or self.selected_tree_path == tpath_column[0]:
            return True
        tpath = tpath_column[0]
        treeview.collapse_all()
        treeview.expand_to_path(tpath)
        treeview.expand_row(tpath, False)
        treeview.set_cursor(tpath)
        self.selected_tree_path = tpath
        return True
        
    def on_treeview_row_activated(self, treeview, path, unused_view_column):
        if self.is_animating():
            return
        unused_name, func = treeview.get_model()[path]
        if self.plugin_helper.call(func):
            self.cube_area.apply_to_glmodel(self.current_movement.current_cube_state)
            self.update_ui()
            move_data = self.current_movement.request_play()
            if not self.start_animation(move_data, stop_after=False):
                self.cube_area.render()
                
    @debug_func
    def run_main_loop(self):
        self.window.show()
        gtk.quit_add(0, self.on_quit)
        gtk.main()
        
    def first_new_game(self, dimension, solved):
        if solved:
            self.solved = True
            
        if self.solved:
            if dimension is None:
                dimension = confstore.dimension
            self.current_movement.set_solved(dimension)
            self.cube_area.apply_to_glmodel(self.current_movement.current_cube_state)
        else:
            if dimension is None:
                self.current_movement.set_saved_game(
                                            confstore.dimension,
                                            confstore.saved_state,
                                            confstore.saved_moves,
                                            confstore.saved_pos)
                self.cube_area.apply_to_glmodel(self.current_movement.current_cube_state)
            else:
                self.current_movement.set_random(dimension)
                self.cube_area.apply_to_glmodel(self.current_movement.current_cube_state)
                
        self.cube_area.render()
        self.update_ui(first=True)
        
    @debug_func
    def new_game(self, new_size, solved):
        assert not self.is_animating()
        
        if solved is not None:
            self.solved = solved
        
        if new_size is not None:
            self.cube_area.re_initialize()
            if new_size <= 0:
                new_size = self.current_movement.current_cube_state.dimension
                
        if self.solved:
            self.current_movement.set_solved(new_size)
        else:
            self.current_movement.set_random(new_size)
        self.cube_area.apply_to_glmodel(self.current_movement.current_cube_state)
        self.cube_area.render()
        self.current_movement.all_moves.reset()
        self.update_ui()
        
        return False
    
    # Request that the game be restarted
    # If data is non zero,  then all it's data will be reallocated
    @debug_func
    def request_new_game(self, unused_widget=None, new_size=None, solved=None):
        #FIXME: fix abort_requested and use it here
        if self.is_animating():
            return
        gobject.idle_add(self.new_game, new_size, solved)
        
    def start_animation(self, move_data, stop_after):
        if move_data:
            blocks = list(self.current_movement.current_cube_state.identify_rotation_blocks(
                                move_data.axis, move_data.slice))
            self.cube_area.animate_rotation(move_data, blocks, stop_after)
            self.playbarstate.playing(self.current_movement.mark_before)
            return True
        return False
        
        
    # Action handlers
    @debug_func
    def on_quit(self):
        (unused_dimension, confstore.saved_state, confstore.saved_moves,
                    confstore.saved_pos) = self.current_movement.get_saved_game()
        return 0
    on_window_delete_event = gtk.main_quit
    def on_action_new_activate(self, *unused_args):
        self.request_new_game(solved=False)
    def on_action_new_solved_activate(self, *unused_args):
        self.request_new_game(solved=True)
    def on_action_colors_activate(self, unused_widget):
        dialogcolors.colorselect_dialog(self)
    def on_action_preferences_activate(self, unused_widget):
        dialogs.preferences_dialog(self)
    def on_toggleaction_play_toolbar_toggled(self, widget):
        self.play_toolbar.props.visible = widget.props.active
    def on_toggleaction_statusbar_toggled(self, widget):
        self.statusbar.props.visible = widget.props.active
    def on_action_reset_rotation_activate(self, unused_widget):
        self.cube_area.reset_rotation()
    def on_toggleaction_dynamic_script_selection_toggled(self, widget):
        confstore.dynamic_script_selection = widget.props.active
        if widget.props.active:
            self.id_treeview_motion_notify_event = (
                self.treeview.connect('motion-notify-event', self.on_treeview_motion_notify_event))
        else:
            self.treeview.disconnect(self.id_treeview_motion_notify_event)
            self.id_treeview_motion_notify_event = None
    def on_action_info_activate(self, unused_widget):
        dialogs.show_about(self.window)
        
    def on_action_rewind_activate(self, unused_widget):
        if not self.is_animating():
            needs_update = self.current_movement.do_rewind_to_mark()
            self.cube_area.apply_to_glmodel(self.current_movement.current_cube_state)
            if needs_update:
                self.cube_area.render()
                self.update_ui()
    def on_action_previous_activate(self, unused_widget):
        if not self.is_animating():
            move_data = self.current_movement.request_back()
            self.start_animation(move_data, stop_after=True)
    def on_action_stop_activate(self, unused_widget):
        self.cube_area.stop_requested = 1
    def on_action_play_activate(self, unused_widget):
        if not self.is_animating():
            move_data = self.current_movement.request_play()
            self.start_animation(move_data, stop_after=False)
    def on_action_next_activate(self, unused_widget):
        if not self.is_animating():
            move_data = self.current_movement.request_next()
            self.start_animation(move_data, stop_after=True)
    def on_action_forward_activate(self, unused_widget):
        if self.is_animating():
            return
        self.current_movement.do_fast_forward()
        self.cube_area.apply_to_glmodel(self.current_movement.current_cube_state)
        self.cube_area.render()
        self.update_ui()
    def on_action_add_mark_activate(self, unused_widget):
        self.current_movement.request_mark_move_queue(True)
        self.update_ui()
    def on_action_remove_mark_activate(self, unused_widget):
        self.current_movement.request_mark_move_queue(False)
        self.update_ui()
        
    def on_action_set_as_initial_state_activate(self, unused_widget):
        if self.is_animating():
            return
        self.current_movement.set_as_initial_state()
        self.update_ui()
    def on_action_reload_scripts_activate(self, unused_widget):
        self.plugin_helper = plugins.PluginHelper(self)
        self.add_scripts_to_sidepane()
    def on_action_invert_moves_activate(self, unused_widget):
        if self.is_animating():
            return
        self.current_movement.invert_moves()
        self.cube_area.apply_to_glmodel(self.current_movement.current_cube_state)
        self.cube_area.render()
        self.update_ui()
        
        
    #Callbacks for PreferencesDialog
    def on_button_size_reset_activate(self, unused_widget):
        del confstore.dimension
    def on_button_animspeed_reset_activate(self, unused_widget):
        del confstore.frameQty
    def on_button_lighting_reset_activate(self, unused_widget):
        del confstore.lighting
        
    def on_button_color_reset_activate(self, unused_widget):
        i = dialogcolors.ColorSelectDialog.this_dlg.active_face[0]
        del confstore.colors[i].color
    def on_button_pattern_reset_activate(self, unused_widget):
        i = dialogcolors.ColorSelectDialog.this_dlg.active_face[0]
        del confstore.colors[i].pattern
    def on_button_image_reset_activate(self, unused_widget):
        i = dialogcolors.ColorSelectDialog.this_dlg.active_face[0]
        del confstore.colors[i].facetype
        del confstore.colors[i].imagefile
        del confstore.colors[i].imagemode
    def on_button_background_color_set(self, button):
        confstore.background_color = str(button.get_color())
    def on_button_background_reset_activate(self, unused_widget):
        del confstore.background_color
        
        
    # Callbacks for editing formulas
    def on_entry_formula_activate(self, unused_widget=None):
        if self.is_animating():
            #TODO: Better: stop animating and proceed
            return
        code = self.entry_formula.get_text()
        pos = self.entry_formula.get_position()
        self.current_movement.set_from_formula(code, pos)
        self.cube_area.apply_to_glmodel(self.current_movement.current_cube_state)
        self.cube_area.render()
        self.update_ui()
        
    def on_entry_formula_icon_press(self, unused_widget, icon_info, event):
        if self.is_animating():
            #TODO: Better: stop animating and proceed
            return
        if event.button==1 and event.type==gdk.BUTTON_PRESS:
            if icon_info.value_nick == 'primary':
                return self.on_entry_formula_activate()
            if icon_info.value_nick == 'secondary':
                self.entry_formula.set_text('')
                self.on_entry_formula_activate()
        
    def set_entry_formula(self, code, pos, unused_markup=None):
        self.entry_formula.set_text(code)
        self.entry_formula.set_position(pos)
        #TODO: seems, that gtk.Entry doesn't work with pango markup, use one-line gtk.TextView?
        
    def update_statusbar(self, first=False):
        '''This function must be called before any action that change the move queue length
        to reflect total count of moves and after each single move'''
        # update statusbar
        # in cube.py        NOT_SOLVED = 0,  SOLVED = 1,  HALF_SOLVED = 2
        #TODO: HALF_SOLVED is broken in some ways
        #       * see comment in on_action_rewind_activate
        #       * should only shown if asymetric pattern or image
        
        #FIXME: cube.cube_status_check() is broken, dont use it now
        #status = cube.cube_status_check()
        status = 0
        
        if first:
            plural = 0
        else:
            plural = self.current_movement.all_moves.queue_length
        status_text = [_('not solved'), _('solved'), _('solved')][status]
        #status_text = [_('not solved'), _('solved'), _('rotated center')][status]
        pass
        
        mesg = ngettext("{current} / {total} move",
                        "{current} / {total} moves",
                        plural).format(
                            current=self.current_movement.all_moves.current_place,
                            total=self.current_movement.all_moves.queue_length)
        context = self.statusbar.get_context_id("move-count")
        self.statusbar.pop(context)
        self.statusbar.push(context, mesg)
        
    def update_ui(self, first=False):
        # update toolbar
        if self.current_movement.all_moves.at_start():
            if self.current_movement.all_moves.at_end():
                self.playbarstate.empty()
            else:
                self.playbarstate.first()
        else:
            if self.current_movement.all_moves.at_end():
                self.playbarstate.last(self.current_movement.all_moves.tail.mark_after)
            else:
                self.playbarstate.mid(self.current_movement.all_moves.prev().mark_after)
                
        # update formula
        code, pos, markup = self.current_movement.all_moves.format(
                                    self.current_movement.current_cube_state.dimension)
        self.set_entry_formula(code, pos, markup)
        
        self.update_statusbar(first=first)
            
    def error_dialog(self, message):
        '''Popup an error dialog box'''
        dialog = gtk.MessageDialog(self.window,
                            gtk.DIALOG_MODAL, gtk.MESSAGE_ERROR, gtk.BUTTONS_CLOSE,
                            message)
        
        dialog.set_transient_for(self.window)
        # Destroy the dialog when the user responds to it (e.g. clicks a button)
        #dialog.connect ("response", gtk.Widget.destroy)
        #dialog.show_all ()
        dialog.run()
        dialog.destroy()
        
        
class PlaybarState (object):
    def __init__(self, application):
        actions = [ 'action_rewind', 'action_previous',
                    'action_stop', 'action_play',
                    'action_next', 'action_forward',
                    'action_add_mark', 'action_remove_mark']
        self.play_button_list = [application.play_actions.get_action(a) for a in actions]
    #@debug_func
    def set_toolbar_state(self, sflags, vflags, mark):
        if mark:
            vflags ^= 0b11
        for a in reversed(self.play_button_list):
            a.set_sensitive(sflags & 1)
            a.set_visible(vflags & 1)
            sflags >>= 1
            vflags >>= 1
    def empty(self):          self.set_toolbar_state(0b00000000, 0b11011101, False)
    def first(self):          self.set_toolbar_state(0b00011100, 0b11011101, False)
    def mid(self, mark):      self.set_toolbar_state(0b11011111, 0b11011110, mark)
    def last(self, mark):     self.set_toolbar_state(0b11000011, 0b11011110, mark)
    def playing(self, mark):  self.set_toolbar_state(0b00100000, 0b11101110, mark)
    

