#-*- coding:utf-8 -*-

#  Pybik -- A 3 dimensional magic cube game.
#  Copyright © 2009, 2011  B. Clausius <barcc@gmx.de>
#
#  This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.


# Ported from GNUbik
# Original filename: guile-hooks.c
# Original copyright and license:
#/*
#    Copyright (C) 2004  Dale Mellor
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 3 of the License,  or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
#*/


import os
import sys

from .debug import *
from . import config
import move_queue


class PluginHelper (object):
    def __init__(self, app):
        self.app = app
        # When a script runs,  the first cube movement that is requested by the script
        # must flush the move queue after the place of current insertion; further move
        # requests must be appended to the end of the move queue. This variable is used
        # to track the first request.
        self.moved = False
        self.plugin_funcs = [
                ("plugin_cube_state",       self.plugin_cube_state),
                ("plugin_rotate_animated",  self.plugin_rotate_animated),
                ('plugin_rotate_flubrd',    self.plugin_rotate_flubrd),
                ("plugin_run_moves",        self.plugin_run_moves),
                ("plugin_error_dialog",     self.plugin_error_dialog),
                ('random',  lambda x:app.current_movement.current_cube_state.rand.randrange(x)),
            ]
            
    def __del__(self):
        self.app = None
        
    def call(self, func):
        if not callable(func):
            return False
        self.moved = False
        func()
        return self.moved
        
    def plugin_cube_state(self):
        '''Function callable from plugin which returns a structure
        reflecting the current state of the cube.'''
        return self.app.current_movement.current_cube_state.copy()
        
    def _start_moves_if_first(self):
        '''The first time a script makes a move on the cube,  the move_queue must be
        truncated to the current place,  and the place is marked so that the viewer
        can rewind the effects of the script. This script performs the necessary
        preparations.'''
        if not self.moved:
            self.moved = 1
            self.app.current_movement.all_moves.truncate()
            self.app.current_movement.all_moves.mark_current(True)
            
    def plugin_rotate_animated(self, moves):
        '''Function which, when called from plugins, causes one
        side of the cube to rotate on-screen.'''
        self._start_moves_if_first()
        for maxis, mslice, mdir in moves:
            data = move_queue.MoveData(maxis, mslice, mdir)
            self.app.current_movement.all_moves.push(data)
            
    def plugin_rotate_flubrd(self, code, size):
        self._start_moves_if_first()
        #TODO: here flubrd should be used, not the one in all_moves,
        # the converter in all_moves may be configurable in the future
        moves = []
        for move_data, unused_pos in self.app.current_movement.all_moves.parse_iter(code, len(code), size):
            moves.append((move_data.axis, move_data.slice, move_data.dir))
        return moves
        
    def plugin_run_moves(self):
        '''Function allowing a script to apply all its moves in one go to the cube,
        without creating animations on the display.'''
        #assert False
        self.app.current_movement.do_fast_forward()
        
    def plugin_error_dialog(self, message):
        '''Function to allow a plugin to display a message to the user.'''
        self.app.error_dialog(message)
        
    def _read_script_directory(self, dir_name):
        '''Function to scan the named directory for all files with a .py extension, and import them.'''
        script_items = []
        if os.path.isdir(dir_name):
            debug('Plugins from', dir_name)
            sys.path.insert(0, dir_name)
            for entry in os.listdir(dir_name):
                if ".py" == entry[-3:]:
                    module_name = entry[:-3]
                    #buffer = "%s/%s" % (dir_name, entry)
                    import imp
                    file_, path, desc = imp.find_module(module_name, [dir_name])
                    try:
                        module = imp.load_module(module_name, file_, path, desc)
                        debug('  import', module_name)
                    except SyntaxError, e:
                        debug(e)
                    except ImportError, e:
                        debug(e)
                    else:
                        for fname, func in self.plugin_funcs:
                            setattr(module, fname, func)
                        if hasattr(module,'scripts'):
                            script_items += module.scripts
                    finally:
                        if file_:
                            file_.close()
        return script_items
        
    def startup_plugins(self):
        '''This function initializes the plugins for us, and once the plugins have
        all been imported, it returns the requested menu structure to the caller.'''
        script_items = self._read_script_directory(config.SCRIPT_DIR)
        script_dir = os.path.join(config.get_data_home(), 'scripts')
        script_items += self._read_script_directory(script_dir)
        return script_items
        
