#-*- coding:utf-8 -*-

#  Pybik -- A 3 dimensional magic cube game.
#  Copyright © 2009-2011  B. Clausius <barcc@gmx.de>
#
#  This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.


# Ported from GNUbik
# Original filename: glarea-common.c
# Original copyright and license:
#/*
#    GNUbik -- A 3 dimensional magic cube game.
#    Copyright (C) 2003  John Darrington
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 3 of the License,  or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
#*/


global __name__, __file__, __package__


import cython

#pxd from gl cimport *
#pxd from glu cimport *
#pxd from math cimport *
#pxd from cube_c cimport *
#pxd from drwBlock_c cimport *
#px-5
from OpenGL.GL import *
from OpenGL.GLU import *
from math import cos, sin, tan, pi as M_PI
from cube import *
from drwBlock import *

import cube

from debug import debug

debug('Importing module:', __name__)
debug('  from package:', __package__)
debug('  compiled:', cython.compiled)


#pxd ctypedef float Quarternion[4]
#px-
Quarternion = None

# Start with the unit quarternion
cython.declare(cube_view = Quarternion)
#px-
cube_view = [0] * 4
cube_view[0] = 0
cube_view[1] = 0
cube_view[2] = 0
cube_view[3] = 1


j8 = [  (0.5625, 0.4375),
        (0.0625, 0.9375),
        (0.3125, 0.6875),
        (0.6875, 0.8125),
        (0.8125, 0.1875),
        (0.9375, 0.5625),
        (0.4375, 0.0625),
        (0.1875, 0.3125),
     ]


# Functions for rotation

#pxd cdef void quarternion_to_matrix(p_Vector M, float* q)
def quarternion_to_matrix(M, q):
    cython.declare(
        q00 = cython.float,
        q01 = cython.float,
        q02 = cython.float,
        q11 = cython.float,
        q12 = cython.float,
        q22 = cython.float,
        q03 = cython.float,
        q13 = cython.float,
        q23 = cython.float,
        q33 = cython.float,
        )
    q00 = q[0] * q[0]
    q01 = q[0] * q[1]
    q02 = q[0] * q[2]
    q03 = q[0] * q[3]
    q11 = q[1] * q[1]
    q12 = q[1] * q[2]
    q13 = q[1] * q[3]
    q22 = q[2] * q[2]
    q23 = q[2] * q[3]
    q33 = q[3] * q[3]
    
    M[0][0] = q33 + q00 - q11 - q22
    M[1][1] = q33 - q00 + q11 - q22
    M[2][2] = q33 - q00 - q11 + q22
    M[3][3] = q33 + q00 + q11 + q22
    
    M[0][1] = 2*(q01 - q23);  M[0][2] = 2*(q02 + q13);  M[1][2] = 2*(q12 - q03)
    M[1][0] = 2*(q01 + q23);  M[2][0] = 2*(q02 - q13);  M[2][1] = 2*(q12 + q03)
    
    M[0][3] = M[1][3] = M[2][3] = 0.0
    M[3][0] = M[3][1] = M[3][2] = 0.0
    
# Pre multiply q1 by q2
#pxd cdef void quarternion_pre_mult(float* q1, float* q2)
def quarternion_pre_mult(q1, q2):
    cython.declare(
        dot_product = cython.float,
        s1 = cython.float,
        s2 = cython.float,
        cross_product = cython.float[3])
    #px-
    cross_product = [0]*3
    
    dot_product = q1[0]*q2[0] + q1[1]*q2[1] + q1[2]*q2[2]
    cross_product[0] = q1[1]*q2[2] - q1[2]*q2[1]
    cross_product[1] = q1[2]*q2[0] - q1[0]*q2[2]
    cross_product[2] = q1[0]*q2[1] - q1[1]*q2[0]
    
    s1 = q1[3]
    s2 = q2[3]
    q1[0] = s1*q2[0] + s2*q1[0] + cross_product[0]
    q1[1] = s1*q2[1] + s2*q1[1] + cross_product[1]
    q1[2] = s1*q2[2] + s2*q1[2] + cross_product[2]
    q1[3] = s1 * s2 - dot_product
    
# Rotate cube about axis (screen relative)
def rotate_cube(axis, dir_):
    # how many degrees to turn the cube with each hit
    cython.declare(
        radians = cython.float,
        s = cython.float,
        rot = Quarternion)
    #px-
    rot = [0]*4
    
    # quarternion from rotation
    radians = (-M_PI if dir_ else M_PI) / 180.0
    s = sin(radians)
    rot[0] = 0
    rot[1] = 0
    rot[2] = 0
    rot[3] = cos(radians)
    if axis == 0:   rot[0] = s
    elif axis == 1: rot[1] = s
    elif axis == 2: rot[2] = s
    
    quarternion_pre_mult(cube_view, rot)


# display functions

cython.declare(
    fovy = cython.float,
    cp_near = cython.float,
    cp_far = cython.float,
    bounding_sphere_radius = cython.float)
fovy = 33.0     # field of view angle
cp_near = -1
cp_far = -1
bounding_sphere_radius = 0


# Wrapper to set the modelview matrix
#pxd cdef void modelViewInit()
def modelViewInit():
    cython.declare(
        m = Matrix)
    #px-
    m = Matrix()
    
    # Update viewer position in modelview matrix
    glMatrixMode(GL_MODELVIEW)
    glLoadIdentity()
    gluLookAt(0, 0, bounding_sphere_radius + cp_near,
              0, 0, 0,          # center
              0.0, 1.0, 0.0)    # up
    
    quarternion_to_matrix(m, cube_view)
    glMultMatrixM(m)
    
    # start with the cube slightly skew,  so we can see all its aspects
    glRotatef(15.0, 1, 0, 0) # horizontally
    glRotatef(15.0, 0, 1, 0) # vertically

    #
    # DM 3-Jan-2004
    #
    # Add a couple of 90 degree turns to get the top and right faces in their
    # logical positions when the program starts up.
    glRotatef(90.0, 1, 0, 0)
    glRotatef(-90.0, 0, 0, 1)

#pxd cdef inline int* glGetViewport(int* v):
#pxd    glGetIntegerv(GL_VIEWPORT, v)
#pxd    return v
#px-
def glGetViewport(unused_v):    return glGetIntegerv(GL_VIEWPORT)

# accFrustum and accPerspective are taken from p397 of the Red Book
#pxd cdef void accFrustum(double rad, double zNear, double zFar, double pixdx, double pixdy)
def accFrustum(rad, zNear, zFar, pixdx, pixdy):
    cython.declare(
        viewport_ = cython.int[4],
        viewport = cython.p_int,
        dx = cython.double,
        dy = cython.double,
        wsize = cython.double)
    #px-
    viewport_ = None
    
    viewport = glGetViewport(viewport_)
    wsize = 2*rad
    dx = -pixdx * wsize / viewport[2]
    dy = -pixdy * wsize / viewport[3]
    
    glFrustum(dx-rad, dy+rad, dy-rad, dy+rad, zNear, zFar)

#pxd cdef void projection_init(int jitter)
def projection_init(jitter):
    global bounding_sphere_radius, cp_near, cp_far
    bounding_sphere_radius = 2 * cube.get_cube_dimension()
    cp_near = bounding_sphere_radius / tan(fovy * M_PI / 360.0)
    cp_far = cp_near + 2*bounding_sphere_radius
    
    glEnable(GL_DEPTH_TEST)
    glClearColor(0, 0, 0, 0)
    
    glMatrixMode(GL_PROJECTION)
    glLoadIdentity()
    
    accFrustum(bounding_sphere_radius,  cp_near, cp_far, j8[jitter][0], j8[jitter][1])

def graphics_area_init(lighting):
    lighting_init(lighting)
    projection_init(0)
    modelViewInit()

def display():
    projection_init(0)
    modelViewInit()
    glClear(GL_ACCUM_BUFFER_BIT)
    
    for jitter in xrange(8):
        glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT)
        
        projection_init(jitter)
        modelViewInit()
        
        draw_cube()
        glAccum(GL_ACCUM, 1.0/8.0)
    glAccum(GL_RETURN, 1.0)
    
#pxd cdef enum:
#pxd    BUFSIZE = 512
#pxd ctypedef GLuint SelectBuf[BUFSIZE]
#px-2
BUFSIZE = 512
SelectBuf = lambda: [0]*BUFSIZE
cython.declare(
    selectBuf = SelectBuf
    )
#px-
selectBuf = SelectBuf()

#pxd cdef void glSelectBuffer_()
def glSelectBuffer_():
    #px-
    glSelectBuffer(BUFSIZE); return
    glSelectBuffer(BUFSIZE, selectBuf)
#pxd ctypedef unsigned char *p_uchar
#pxd cdef inline p_uchar glReadPixel(int x, int y, p_uchar pixel):
#pxd    glReadPixels(x, y, 1, 1, GL_RGB, GL_UNSIGNED_BYTE, pixel); return pixel
#px-3
p_uchar = cython.p_char
def glReadPixel(x, y, pixel):
    return glReadPixels(x, y, 1, 1, GL_RGB, GL_UNSIGNED_BYTE, [[[0, 0, 0]]])[0][0]

def pick_polygons(lighting, x, y, granularity, pick_mode=1):
    cython.declare(
        viewport_ = cython.int[4],
        viewport = cython.p_int,
        pixel = cython.uchar[3],
        pixel_ = p_uchar)
    #px-2
    viewport_ = None
    pixel = [0]*3
    
    viewport = glGetViewport(viewport_)
    if x < 0 or y < 0 or x >= viewport[2] or y >= viewport[3]:
        return None
        
    if lighting:
        set_lighting(False)
        
    glMatrixMode(GL_PROJECTION)
    glPushMatrix()
    glLoadIdentity()
    
    gluPickMatrix(x, y, granularity, granularity, viewport)
    gluPerspective(fovy, 1, cp_near, cp_far)
    
    modelViewInit()
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT)
    if pick_mode == 1:
        pick_cube()
    else:
        pick2_cube()
    glMatrixMode(GL_PROJECTION)
    glPopMatrix()
    
    pixel_ = glReadPixel(x, y, pixel)
    
    if lighting:
        set_lighting(True)
        
    if pixel_[2] == 0:
        return None
    if pick_mode == 1:
        return (((pixel_[0]-68)<<8) | pixel_[1]) >> 3, pixel_[1] & 0x7, pixel_[2] / 0x55
    else:
        return pixel_[0], pixel_[1], pixel_[2]
    

#pxd cdef void lighting_init(bint lighting)
def lighting_init(lighting):
    cython.declare(
        light_ambient = cython.float[4],
        mat_diffuse = cython.float[4],
        mat_specular = cython.float[4],
        mat_shininess = cython.float,
        light_position = cython.float[4])
    #px-4
    light_ambient = [0]*4
    mat_diffuse = [0]*4
    mat_specular = [0]*4
    light_position = [0]*4
    
    init_vector(light_ambient, [0.6, 0.6, 0.6, 1.0])
    init_vector(mat_diffuse, [0.8, 0.8, 0.8, 1.0])
    init_vector(mat_specular, [0.2, 0.2, 0.2, 1.0])
    mat_shininess = 1.0
    init_vector(light_position, [-1.0, 0.5, 0.0, 0.0])
    
    glShadeModel(GL_SMOOTH)
    glColorMaterial(GL_FRONT, GL_AMBIENT_AND_DIFFUSE)
    
    glMaterialfv(GL_FRONT, GL_SPECULAR, mat_specular)
    glMaterialfv(GL_FRONT, GL_DIFFUSE, mat_diffuse)
    glMaterialf(GL_FRONT, GL_SHININESS, mat_shininess)
    
    glLightfv(GL_LIGHT0, GL_POSITION, light_position)
    glLightfv(GL_LIGHT1, GL_AMBIENT, light_ambient)
    
    if lighting:
        glEnable(GL_LIGHTING)
    else:
        glDisable(GL_LIGHTING)
    glEnable(GL_COLOR_MATERIAL)
    glEnable(GL_LIGHT0)
    glEnable(GL_LIGHT1)

#pxd cpdef set_lighting(int enable)
def set_lighting(enable):
    if enable:
        glEnable(GL_LIGHTING)
    else:
        glDisable(GL_LIGHTING)


#pxd cpdef resize(int width, int height)
def resize(width, height):
    cython.declare(
        min_dim = cython.int)
    min_dim = min(width, height)
    # Ensure that cube is always the same proportions 
    glViewport((width-min_dim) / 2, (height-min_dim) / 2, min_dim, min_dim)

