/*
    ProjectType.h

    Protocol declaration of the ProjectType protocol for the
    ProjectManager application.

    Copyright (C) 2005, 2006  Saso Kiselkov

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#import <Foundation/NSObject.h>
#import <Foundation/NSString.h>
#import <Foundation/NSDictionary.h>

@class NSArray, NSDictionary, NSImage;

@class ProjectDocument;

/**
 * This protocol is adopted by project type support objects.
 *
 * Each project has a 'type' which is represented by it's
 * projectType object. This object defines what kind of file
 * categories exist in the project, how to construct the
 * makefile and a bunch of other things.
 */
@protocol ProjectType <NSObject>

// ======== Class Methods =========

/// Should return the project type ID.
+ (NSString *) projectTypeID;

/**
 * Should return a human-readable, perhaps localized descriptive
 * name of the project type.
 */
+ (NSString *) humanReadableProjectTypeName;

/// Should return a more detailed description of the project type.
+ (NSString *) projectTypeDescription;

/// Should return an icon of the project type (max 48x48 pixels).
+ (NSImage *) projectTypeIcon;

/**
 * Should return an OR'ed set of project capabilities provided by this
 * project type.
 */
+ (NSArray *) projectModules;

/**
 * The project type should return a dictionary where keys are
 * project template names and values are descriptions of the
 * templates.
 */
+ (NSDictionary *) projectTemplateDescriptions;

/**
 * The receiver should return the path to the location of the
 * project template named `templateName'.
 */
+ (NSString *) pathToProjectTemplate: (NSString *) templateName;


// ========= Instance Methods ==========

/**
 * Designated initializer.
 *
 * This message is sent to a project type object when it is initialized.
 *
 * @param aProject The project which owns the project type. The project
 *      type should establish a weak-reference to the project, since
 *      the project retains the project type object and a retain cycle
 *      would result.
 * @param infoDict The project type's saved private info dictionary.
 * @param modules The project's modules.
 *
 * @see [ProjectType infoDictionary]
 */
- initWithDocument: (ProjectDocument *) aProject
    infoDictionary: (NSDictionary *) infoDict
    projectModules: (NSArray *) modules;

/**
 * Should return the project's info dictionary which is suitable for
 * being written into the project file when saving.
 *
 * @return The project type's private information dictionary.
 */
- (NSDictionary *) infoDictionary;

/**
 * Instructs the project to regenerate any of it's derived files,
 * if necessary. This step invoked in situations such as the following:
 *
 * - a project is opened
 * - a project is saved
 *
 * @return YES if regenerating derived files succeeded, NO if it didn't.
 */
- (BOOL) regenerateDerivedFiles;

/**
 * In case the project type supports subprojects, this method should
 * return a path to a directory where subprojects of the project can
 * be stored.
 *
 * @return A path to the project's subproject directory.
 */
- (NSString *) pathToSubprojectsDirectory;

@end

/**
 * Creates a template description dictionary from the provided arguments.
 * This is useful to project type objects in the +projectTemplateDescriptions
 * method, where they can easily create the values of the returned
 * dictionary with this function.
 *
 * @param templateDescription The description of the template created.
 *      This argument may not be `nil'.
 * @param templateIcon The icon of the template. This argument may be `nil'.
 *
 * @return A dictionary formatted like the following:
 * {
 *   Description = "templateDescription";
 *   Icon = (NSImage *) templateIcon;
 * }
 */
static inline NSDictionary *
PMCreateTemplateDescription(NSString * templateDescription,
                            NSImage * templateIcon)
{
  return [NSDictionary dictionaryWithObjectsAndKeys:
    templateDescription, @"Description",
    // templateIcon can be `nil' here!
    templateIcon, @"Icon",
    nil];
}
