/*
    ProjectImageView.m

    Implementation of the ProjectImageView class for the
    ProjectManager application.

    Copyright (C) 2005  Saso Kiselkov

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#import "ProjectImageView.h"

#import <Foundation/NSArray.h>

#import <AppKit/NSEvent.h>
#import <AppKit/NSPasteboard.h>
#import <AppKit/NSWindow.h>
#import <AppKit/NSDragging.h>
#import <AppKit/NSImage.h>

#import "FileManager.h"
#import "FileManagerDelegate.h"
#import "../../ProjectDocument.h"
#import "../../NSImageAdditions.h"

@implementation ProjectImageView

- (void) drawRect: (NSRect) r
{
  [super drawRect: r];

  if (showsLinkIndicator)
    {
      static NSImage * linkIndicator = nil;

      if (linkIndicator == nil)
        {
          ASSIGN(linkIndicator, [NSImage imageNamed: @"LinkIndicator"
                                              owner: self]);
        }

      [linkIndicator compositeToPoint: NSZeroPoint
                            operation: NSCompositeSourceOver];
    }
}

- (void) mouseDown: (NSEvent *) ev
{
  if ([ev clickCount] > 1)
    {
      [fileManager openFile: self];
    }
  else
    {
      ev = [[self window] nextEventMatchingMask: NSAnyEventMask];
      if ([ev type] == NSLeftMouseDragged)
        {
          NSPasteboard * pb = [NSPasteboard pasteboardWithName: NSDragPboard];
          NSArray * selectedFiles = [fileManager selectedFiles];
          NSMutableArray * filePaths = [NSMutableArray arrayWithCapacity:
            [selectedFiles count]];
          NSEnumerator * e;
          NSSize imgSize = [[self image] size];
          NSString * filename;

          draggingMask = NSDragOperationCopy | NSDragOperationMove |
            NSDragOperationLink;

          if (selectedFiles == nil)
            {
              return;
            }

          e = [selectedFiles objectEnumerator];
          while ((filename = [e nextObject]) != nil)
            {
              BOOL isCategory = ([fileManager typeOfFileAtPath: filename] ==
                FMFileTypeCategory);
              NSString * path = [[fileManager delegate]
                pathToFile: filename isCategory: isCategory];

              // don't allow linking if the selection contains a category
              if (isCategory)
                {
                  draggingMask = draggingMask & (~NSDragOperationLink);
                }

              // don't allow moving if deleting the file isn't allowed
              if (![[fileManager delegate] canDeletePath: filename])
                {
                  draggingMask = draggingMask & (~NSDragOperationMove);
                }

              if (path != nil)
                {
                  [filePaths addObject: path];
                }
            }

          // we declare two types:
          //  ProjectsFilesPboardType - used when dragging files within
          //        the project. It declares the project and category
          //        where a file belongs.
          //  NSFilenamesPboardType - used in all other cases - when a file
          //        is dragged out of it's containing project, either to
          //        another project, or another file system location.
          [pb declareTypes: [NSArray arrayWithObjects:
            ProjectFilesPboardType, NSFilenamesPboardType, nil]
                     owner: nil];

          [pb setPropertyList: [NSDictionary dictionaryWithObjectsAndKeys:
            selectedFiles, @"Filenames",
            [[fileManager document] fileName], @"Project",
            nil]
                  forType: ProjectFilesPboardType];
          [pb setPropertyList: filePaths forType: NSFilenamesPboardType];

          [self dragImage: [self image]
                       at: [self convertPoint: [ev locationInWindow] fromView: nil]
                   offset: NSMakeSize(-(imgSize.width/2), -(imgSize.height/2))
                    event: ev
               pasteboard: pb
                   source: self
                slideBack: YES];
        }
    }
}

- (unsigned int)draggingSourceOperationMaskForLocal:(BOOL)isLocal
{
  return draggingMask;
}

- (void) setShowsLinkIndicator: (BOOL) flag
{
  showsLinkIndicator = flag;
  [self setNeedsDisplay: YES];
}

- (BOOL) showsLinkIndicator
{
  return showsLinkIndicator;
}

@end
