/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *            Copyright (C) 2006 SUSE Linux Products GmbH                  *
 *                                                                         *
 *             Author(s): Holger Macht <hmacht@suse.de>                    *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#include "config.h"

#include "powerlib_local.h"
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <string.h>
#include <liblazy.h>

#include <dbus/dbus.h>

#define ACPI_BATTERY_DIR  "/proc/acpi/battery"
#define ACPI_AC_DIR "/proc/acpi/ac_adapter"

int getBatteriesInfo(BatteryGeneral *battery_info)
{
	int last_full_capacity_sum = 0;
	int remaining_capacity_sum = 0;
	int error;

	if (battery_info == NULL) {
                errno = EINVAL;
                return -3;
        }
	
	battery_info->remaining_minutes = UNKNOWN;
	battery_info->remaining_percent = UNKNOWN;
	battery_info->charging_state = CHARG_STATE_UNKNOWN;

	char **batteries = NULL;
	error = liblazy_hal_find_device_by_capability("battery", &batteries);

	if (error || batteries == NULL) {
		pDebug(DBG_INFO, "Could not get list of batteries in system");
		return NO_DEVICE_ERROR;
	}

	int valid_batteries = 0;
	int rate = 0;
	for (int i = 0; batteries[i] != NULL; ++i) {
		char *type;
		int remaining_time;
		int charging;
		int discharging;
		int charging_state;
		int last_full_capacity;
		int remaining_capacity;

		liblazy_hal_get_property_string(batteries[i], "battery.type", &type);

		if (type == NULL || strcmp(type, "primary")) {
			liblazy_free_string(type);
			continue;
		}
		liblazy_free_string(type);

		liblazy_hal_get_property_int(batteries[i], "battery.remaining_time",
					     &remaining_time);
		liblazy_hal_get_property_bool(batteries[i], "battery.rechargeable.is_charging",
					      &charging);
		liblazy_hal_get_property_bool(batteries[i], "battery.rechargeable.is_discharging",
					      &discharging);

		/* calc remaining time from HAL */
		if (remaining_time >= 0) {
			valid_batteries++;
			if (battery_info->remaining_minutes == UNKNOWN)
				battery_info->remaining_minutes = 0;
			battery_info->remaining_minutes += remaining_time / 60;
		}

		charging_state = CHARG_STATE_UNKNOWN;
		/* calc charging state */
		if (charging)
			charging_state |= CHARG_STATE_CHARGING;
		if (discharging)
			charging_state |= CHARG_STATE_DISCHARGING;

		battery_info->charging_state |= charging_state;

		liblazy_hal_get_property_int(batteries[i], "battery.charge_level.current",
					     &remaining_capacity);

		liblazy_hal_get_property_int(batteries[i], "battery.charge_level.last_full",
					     &last_full_capacity);

		/* calculate the charging rate for the multiple battery case */
		if (remaining_time > 0 && remaining_capacity != UNKNOWN) {
			int to_go = 0;
			if (charging_state & CHARG_STATE_CHARGING) {
				if (last_full_capacity > 0)
					to_go = last_full_capacity - remaining_capacity;
			} else {
				to_go = remaining_capacity;
			}
			rate += (to_go / remaining_time);
		}

		/* FIXME: can we have one battery with valid last_full and a
			  second battery with last_full == UNKNOWN?
			  This would make those values totally bogus. */
		if (last_full_capacity > 0) {
			last_full_capacity_sum += last_full_capacity;

			/* see the FIXME above, same applies here... */
			if (remaining_capacity > 0) {
				remaining_capacity_sum += remaining_capacity;
			}
		}
	}

	if (rate > 0 && valid_batteries > 1) {	// with APM, this will never happen.
		if (battery_info->charging_state & CHARG_STATE_CHARGING) {
			battery_info->remaining_minutes =
				((last_full_capacity_sum - remaining_capacity_sum) / rate) / 60;
		} else {
			battery_info->remaining_minutes = (remaining_capacity_sum / rate) / 60;
		}
	}

	if (last_full_capacity_sum > 0 && remaining_capacity_sum >= 0)
		battery_info->remaining_percent =
			(remaining_capacity_sum * 100.0) / last_full_capacity_sum;

	return (valid_batteries == 0) ? 0 : 1;
}

int getACAdapterStatus(void)
{
	int test;
	int ret;

	char **ac_adapter=NULL;
	char *ac_device=NULL;
	int ac_state=-1;
	int dev_ac_state=-1;

	test = checkACPI();
	/* --------- ACPI ------------- // */

	if (test == ACPI) {
		/* TODO: Only check for module here, because device
		 * availability is checked for above */
		ret = check_ACPI_dir(ACPI_AC_DIR);
		if (ret < 0)
			/* returns either NO_DEVICE_ERROR or NO_MODULE_ERROR */
			return ret;
	} else if (test == NOT_SUPPORTED) {
		pDebug(DBG_DIAG, "Neither APM nor ACPI support found");
		return AC_UNKNOWN;
	}

	if (liblazy_hal_find_device_by_capability("ac_adapter", &ac_adapter)) {
		pDebug(DBG_ERR, "Could not get ac_adapter device");
		return NO_DEVICE_ERROR;
	}

	if (ac_adapter != NULL) {
		/* now could handle multiple ac-adapter (not just only taking the first):
		 * will give AC_ONLINE as soon as one of the adaperts is present
		 * will give AC_OFFLINE if at least one adapter is capable of giving status
		 * 	and those are all not-present */
		for (int i = 0; ac_adapter[i] != NULL; i++)
		{
			ac_device = strdup(ac_adapter[i]);
			liblazy_hal_get_property_bool(ac_device, "ac_adapter.present", &dev_ac_state);
			if (dev_ac_state == 1)
				ac_state = 1;
			else if (!dev_ac_state && ac_state == -1)
			{
				ac_state = 0;
			} 
			free(ac_device);
		}
		liblazy_free_strlist(ac_adapter);
	} else {
		liblazy_free_strlist(ac_adapter);
		return NO_DEVICE_ERROR;
	}
	
	if (ac_state == 1)
		return AC_ONLINE;
	else if (!ac_state)
		return AC_OFFLINE;
	else
		return AC_UNKNOWN;
}


/* only for ACPI!!!
return x < 0 on error
returns 0 if alarm has already sub-ceded(alarm will still be set)
returns positive value if alarm has been set successfully
*/

int setBatteryAlarm(int percent)
{
	int x;
	/* divide_factor: factor on which all batteries are above the wanted percent (statistically) */
	int remaining_capacity_sum, last_full_capacity_sum, rem_Perc_all;
	float divide_factor = 0.0;
	remaining_capacity_sum = 0;
	last_full_capacity_sum = 0;
	int alarm_value = 0;
	char file[MAX_FILE_PATH + 1] = "";
	char dev_name[MAX_FILE_PATH + 1] = "";
	char verify[MAX_LINE_SIZE + 1] = "";
	char temp[MAX_LINE_SIZE + 1] = "";
	int ret = 1;
	FILE *fp;

	if (checkACPI() != ACPI) {
		return NO_ACPI_ERROR;
	}

	/* check whether module has been loaded and device is available */
	ret = check_ACPI_dir(ACPI_BATTERY_DIR);
	if (ret < 0)
		/* returns either NO_DEVICE_ERROR or NO_MODULE_ERROR */
		return ret;

	if (percent <= 0 || percent > 100) {
		pDebug(DBG_DIAG, "illegal percent value for battery alarm: %d", percent);
		return -1; //UNKNOWN
	}

	char **batteries = NULL;
	liblazy_hal_find_device_by_capability("battery", &batteries);

	if (batteries == NULL) {
		pDebug(DBG_INFO, "Could not get list of batteries in system");
		return UNKNOWN;
	}

	/* iterate present battery devices. Don't iterate on existing devices: */
	/* Imagine BAT0 -> not present, BAT1 -> present */
	for (x = 0; batteries[x] != NULL; x++) {
		char *type;
		int present;

		liblazy_hal_get_property_string(batteries[x], "battery.type", &type);
		if (strcmp(type, "primary")) {
			liblazy_free_string(type);
			continue;
		}
		liblazy_free_string(type);

		liblazy_hal_get_property_bool(batteries[x], "battery.present", &present);
		if (present == 1) {
			int remaining_capacity;
			int last_full_capacity;

			liblazy_hal_get_property_int(batteries[x], "battery.charge_level.current",
						     &remaining_capacity);

			liblazy_hal_get_property_int(batteries[x], "battery.charge_level.last_full",
						     &last_full_capacity);
			if (remaining_capacity > 0) {
				if (last_full_capacity > 0) {
					last_full_capacity_sum += last_full_capacity;
					remaining_capacity_sum += remaining_capacity;
				}
			}
		}
	}

	BatteryGeneral bg;
	getBatteriesInfo(&bg);

	rem_Perc_all = bg.remaining_percent;
	if (rem_Perc_all > 140 || rem_Perc_all < 0) {
		pDebug(DBG_DIAG, "bogus value %d%% remaining, set to 100", rem_Perc_all);
		rem_Perc_all = 100;
	} else if (rem_Perc_all > 100)
			rem_Perc_all = 100;
	/*****************************************************************************/
	if (percent > 0)
		pDebug(DBG_INFO, "set alarms: rem_Perc_all: %d, want alarm at: %d, divisor: %f",
		       rem_Perc_all, percent, divide_factor);
	
	for (x = 0; batteries[x] != NULL; x++) {
		char *type;
		int present;
		int remaining_capacity;
		int last_full_capacity;
		char *path = NULL;
		
		liblazy_hal_get_property_string(batteries[x], "battery.type", &type);
		if (strcmp(type, "primary")) {
			liblazy_free_string(type);
			continue;
		}
		liblazy_free_string(type);
		
		divide_factor = (float)rem_Perc_all / percent;
			
		liblazy_hal_get_property_bool(batteries[x], "battery.present", &present);
		if (present != 1)
			continue;
		liblazy_hal_get_property_int(batteries[x], "battery.charge_level.current",
					     &remaining_capacity);
		liblazy_hal_get_property_int(batteries[x], "battery.charge_level.last_full",
					     &last_full_capacity);

		if (divide_factor < 1) {
			// we would set the alarm higher than the current charge level.
			pDebug(DBG_DIAG, "divisor: %f, smaller than 1", divide_factor);
			goto ERROR;
		}
		alarm_value = (int)roundf(remaining_capacity / divide_factor);

		pDebug(DBG_DIAG, "Battery %d, remaining: %d, last full: %d, set alarm to: %d",
		       x, remaining_capacity, last_full_capacity, alarm_value);
		
		/*******  write alarm for each battery ************************/
		if (getDirEntry(x, dev_name, sizeof(dev_name), ACPI_BATTERY_DIR) < 0) {
			pDebug(DBG_DIAG, "could not get acpi directory name for primary battery %d", x);
			goto ERROR;
		}
		
		liblazy_hal_get_property_string(batteries[x], "linux.acpi_path", &path);

		if (path == NULL) {
			goto ERROR;
		}

		if (strlen(path) + strlen("/alarm") + 1 > MAX_FILE_PATH) {
			liblazy_free_string(path);
			pDebug(DBG_DIAG, "Path %s/alarm too long", path);
			goto ERROR;
		} else {
			strcat(file, path);
			strcat(file, "/alarm");

			if (_write_line(file, "%d\n", alarm_value) < 0) {
				liblazy_free_string(path);
				pDebug(DBG_INFO, "Could not write alarm value");
				goto ERROR;
			} else {
				/* don't set return value has already been set to 0 or initialised with 1 */
/* ##################### please review me ################################################ */
// can this work? and what fp are we using if this fails?
				if ((fp = fopen(file, "r")) == NULL) {
					pDebug(DBG_DIAG, "Could not read alarm settings made (%s)", strerror(errno));
				}
/* ##################### please review me ################################################ */

				if (getColonValue(fp, verify, sizeof(verify), temp, sizeof(temp)) == 1) {
					if (atoi(verify) != alarm_value) {
						pDebug(DBG_DIAG, "failure. wrote: %d, read: %s", alarm_value, verify);
						fclose(fp);
						liblazy_free_string(path);
						goto ERROR;
					}
					pDebug(DBG_DEBUG, "Alarm value %d successfully written to %s", alarm_value, file);
				} else {
					pDebug(DBG_INFO, "Writing of alarm value couldn't be verified");
				}
				fclose(fp);
			}
			liblazy_free_string(path);
		}
		/*******  write alarm for each battery ************************/
	}

	liblazy_free_strlist(batteries);
	return ret;

ERROR:
	liblazy_free_strlist(batteries);
	return UNKNOWN;

}

