/*
 Copyright (C) 2011 Christian Dywan <christian@twotoasts.de>

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 See the file COPYING for the full license text.
*/

namespace Postler {
    internal class HelperProcess : Object {
        internal string? display_name;
        internal bool unverified;
        internal string? folder = null;
        string? command = null;
        IOChannel inputc;
        IOChannel errorc;

        internal HelperProcess (string? display_name=null, bool unverified=false) {
            this.display_name = display_name;
            this.unverified = unverified;
        }

        internal void execute (string command) {
            this.command = command;

            try {
                string[] argv;
                Pid pid;
                int out_fd, err_fd;
                Shell.parse_argv (command, out argv);
                /* Enforce untranslated output */
                Environment.set_variable ("LANG", "", true);
                Environment.set_variable ("LC_ALL", "", true);
                Environment.set_variable ("LANGUAGE", "", true);
                Process.spawn_async_with_pipes (null, argv, null,
                    SpawnFlags.SEARCH_PATH,
                    null, out pid, null, out out_fd, out err_fd);
                inputc = new IOChannel.unix_new (out_fd);
                inputc.add_watch_full (0, IOCondition.IN | IOCondition.HUP, input_callback);
                errorc = new IOChannel.unix_new (err_fd);
                errorc.add_watch_full (0, IOCondition.IN | IOCondition.HUP, input_callback);
            }
            catch (GLib.Error error) {
                GLib.debug ("Error, reading, %s: %s", command, error.message);
                done (error.message);
            }
        }

        bool input_callback (IOChannel channel, IOCondition condition) {
            if (command == null)
                return false;

            if ((condition & IOCondition.HUP) == IOCondition.HUP) {
                GLib.debug ("Done/ %s: %s",
                            channel == errorc ? "error" : "input", command);
                done ();
                return false;
            }

            try {
                StringBuilder msg = new StringBuilder ();
                size_t len;
                channel.read_line_string (msg, out len);
                GLib.debug ("Line/ %s: %s",
                            channel == errorc ? "error" : "input", msg.str);
                line_read (msg.str, channel == errorc);
                /* finnish may have been called in a line_read callback */
                if (command == null)
                    return false;
            }
            catch (Error error) {
                GLib.debug ("Error, reading, %s: %s", command, error.message);
                finnish (error.message);
                return false;
            }
            return true;
        }

        internal signal void line_read (string line, bool is_error);

        internal void finnish (string error_message="") {
            command = null;
            done (error_message);
        }

        internal signal void done (string error_message="");
    }

    [DBus (name = "org.elementary.Postler")]
    class PostlerService : Object {
        double total = 0;
        int unread = 0;
        Dock.Item dockitem;

#if HAVE_INDICATE
        Indicate.Server indicator;
        List<Indicate.Indicator> items;

        void update_inbox_indicator (Indicate.Indicator item) {
            string path = item.get_property ("url") + "/INBOX/new";
            uint new_messages = 0;
            try {
                var inbox = Dir.open (path, 0);
                while (inbox.read_name () != null)
                    new_messages++;
                var folder = File.new_for_path (path);
                var monitor = folder.monitor_directory (0, null);
                monitor.changed.connect ((monitor, file, other, event) => {
                    update_inbox_indicator (item);
                });
                indicator.set_data ("monitor", monitor);
                
            } catch (Error error) {
                GLib.warning ("Indicator: %s", error.message);
            }

            if (new_messages > 0) {
                item.set_property ("count", new_messages.to_string ());
                item.emit_show ();
            }
            else
                item.emit_hide ();
        }

        void add_inbox_indicator (AccountInfo info) {
            if (info.type != AccountType.IMAP)
                    return;

            var item = new Indicate.Indicator.with_server (indicator);
            item.set_property ("name", info.display_name);
            item.set_property ("url", info.path);
            item.set_property ("draw-attention", "true");
            item.user_display.connect ((item) => {
                string url = item.get_property ("url");
                Postler.App.spawn_module ("bureau", url + "/INBOX");
            });
            items.append (item);
            update_inbox_indicator (item);
        }
#endif

        bool new_message_timer () {
            receive ("");
            return true;
        }

        bool badge_timer () {
            var accounts = new Accounts ();
            uint new_messages = 0;
            foreach (var info in accounts.get_infos ()) {
                try {
                    var inbox = Dir.open (info.path + "/INBOX/new", 0);
                    while (inbox.read_name () != null)
                        new_messages++;
                }
                catch (GLib.Error error) { }
            }
            dockitem.set_badge (new_messages);
            return true;
        }

        public PostlerService () {
            GLib.Timeout.add_seconds (600, new_message_timer); /* 10 minutes */
            GLib.Timeout.add_seconds (30, badge_timer);

            dockitem = new Dock.Item.for_name (_("Postler"));

#if HAVE_INDICATE
            indicator = Indicate.Server.ref_default ();
            indicator.set_type ("message.email");
            indicator.set_desktop_file (
                Config.DATADIR + "/applications/postler.desktop");
            indicator.server_display.connect (() => {
                Postler.App.spawn_module ("bureau");
            });

            var item = new Indicate.Indicator.with_server (indicator);
            item.set_property ("name", _("Compose Message"));
            item.user_display.connect (() => {
                Postler.App.spawn_module ("compose");
            });
            item.emit_show ();
            items.append (item);

            if (Environment.find_program_in_path ("dexter") != null) {
                item = new Indicate.Indicator.with_server (indicator);
                item.set_property ("name", _("Contacts"));
                item.user_display.connect (() => {
                    new Dexter.Dexter ().show_window ();
                });
                item.emit_show ();
                items.append (item);
            }

            var accounts = new Accounts ();
            foreach (var info in accounts.get_infos ())
                add_inbox_indicator (info);
            indicator.show ();
#endif
        }

        void display_status (HelperProcess helper, ref string line) {
            string? error_id = null;
            if ("can't verify certificate" in line)
                error_id = "CERT";
            else if ("NO Login failed" in line)
                error_id = "LOGIN";
            else if ("UIDVALIDITY of slave changed" in line)
                error_id = "UIDVAL";
            else if ("Invalid UID messageset" in line)
                error_id = "BADUID";
            else if ("BAD Could not parse command" in line)
                error_id = "BADCMD";
            else if ("Failed to create pipe for communicating" in line)
                error_id = "NOPIPE";
            else if ("Error: channel" in line && " is locked" in line)
                error_id = "LOCK";
            if (error_id != null) {
                line = "PSTL/" + error_id + "/" + helper.folder + "/" + line;
                return;
            }

            string msg = line.split ("\n") [0];
            if (msg.contains ("master: ")) {
                total = (msg.split (" ") [1]).to_double ();
            }
            else if (msg.contains ("slave: ")) {
                string[] pieces = msg.split (" ");
                total = total - pieces[1].to_double ();
                if (helper.folder == "INBOX") {
                    unread += pieces[3].to_int ();
                    helper.folder = _("Inbox");
                }
            }
            else if (msg.contains ("S: ?")) {
                double count = (((msg.split ("/"))[2]).split (" ") [0]).to_double ();
                if (helper.folder.contains ("/"))
                    helper.folder = helper.folder.split ("/") [1];
                string state = _("Receiving %d of %d").printf ((int)count, (int)total);
                helper.folder = Postler.Folders.decode_foldername (helper.folder);
                progress (helper.display_name, helper.folder + "\n" + state,
                          total > count ? count / total : 0);
            }
            else if (!(msg.contains ("Synchronizing")) ) {
                if (msg.contains ("Selecting master ")) {
                    var tmp = msg.replace ("Selecting master ", "");
                    helper.folder = tmp.replace ("...", "");
                }
            }
        }

        public signal void progress (string account, string text, double fraction);

        public void receive (string account) {
            var infos = new GLib.List<AccountInfo> ();
            var accounts = new Accounts ();
            if (account == "") {
                foreach (var info in accounts.get_infos ()) {
                    if (info.type == AccountType.IMAP)
                        infos.prepend (info);
                }
            }
            else {
                foreach (var info in accounts.get_infos ()) {
                    if (info.name == account)
                        infos.prepend (info);
                }
            }

            if (infos.length () == 0) {
                received (account, _("The account doesn't exist"));
                return;
            }

            unread = 0;
            foreach (var info in infos) {
                try {
                    progress (info.display_name, _("Checking for mail..."), 0.0);
                    string command = accounts.get_receive_command (info);
                    var helper = new HelperProcess (info.display_name, info.unverified);
                    helper.line_read.connect ((current_line, is_error) => {
                        string line = current_line;
                        display_status (helper, ref line);

                        if (line.has_prefix ("PSTL/CERT/") && helper.unverified)
                            /* Do nothing, no verification desired. */ ;
                        else if (line.has_prefix ("PSTL/UIDVAL/"))
                            helper.done (line); /* Try to proceed. */
                        else if (line.has_prefix ("PSTL/LOCK/"))
                            helper.finnish (); /* Parallel access to one inbox */
                        else if (is_error)
                            helper.finnish (line);
                    });
                    helper.done.connect ((helper, error_message) => {
                        progress (helper.display_name, "", 0.0);
                        if (error_message != "") {
                            received (helper.display_name, error_message);
                            return;
                        }
                        GLib.debug ("Done: %d new messages", unread);
                        if (unread > 0) {
                            Postler.App.send_notification (
                                ngettext ("You have %d message",
                                "You have %d new messages", unread).printf (unread));
                            Postler.App.play_sound ("message-new-email");
                        }
                        received (helper.display_name, "");
                    });
                    helper.execute (command);
                } catch (Error error) {
                    received (info.display_name, error.message);
                    return;
                }
            }
        }

        public signal void received (string account, string error_message);

        public bool fetch (string account) {
            var infos = new GLib.List<AccountInfo> ();
            var accounts = new Accounts ();
            if (account == "") {
                foreach (var info in accounts.get_infos ()) {
                    if (info.type == AccountType.IMAP)
                        infos.prepend (info);
                }
            }
            else {
                foreach (var info in accounts.get_infos ()) {
                    if (info.name == account)
                        infos.prepend (info);
                }
            }

            if (infos.length () == 0)
                return false;

            unread = 0;
            foreach (var info in infos) {
                try {
                    string command = accounts.get_fetch_command (info);
                    string output;
                    Process.spawn_sync (null, command.split (" "), null,
                                        SpawnFlags.SEARCH_PATH, null, out output);
                    GLib.debug ("Fetching %s\n%s", info.name, output);
                    string[] listing = output.split ("INBOX\n"
                                                   + "Channel local-remote\n"
                                                   + "Opening slave local...\n"
                                                   + "local-remote:\n");
                    if (!(listing[0] != null && listing[1] != null))
                        throw new GLib.FileError.FAILED (_("Failed to list folders"));
                    string[] folders = listing[1].chomp ().split ("\n");
                    foreach (string current_folder in folders) {
                        string folder = info.path + "/" + current_folder.replace ("/", "~-");
                        DirUtils.create_with_parents (folder + "/new", 0700);
                        DirUtils.create (folder + "/cur", 0700);
                        DirUtils.create (folder + "/tmp", 0700);
                    }
                } catch (Error error) {
                    return false;
                }
            }

            return true;
        }

        public void send (string account, string filename) {
            var accounts = new Accounts ();
            try {
                foreach (var info in accounts.get_infos ())
                    if (info.name == account) {
                        string command = accounts.get_send_command (info, filename);
                        var helper = new HelperProcess (info.display_name, info.unverified);
                        helper.line_read.connect ((line, is_error) => {
                            if ("TLS certificate verification failed" in line)
                                helper.finnish (_("Can't verify mail server authenticity."));
                            else if ("Connection timed out" in line) {
                                helper.finnish (
                                    _("It is taking too long for the server to respond.") + "\n"
                                  + _("Try changing the port to 25, 587 or 465."));
                            }
                            else if ("Connection refused" in line) {
                                helper.finnish (
                                    _("The server rejected the connection.") + "\n"
                                  + _("Try changing the port to 25, 587 or 465."));
                            }
                            else if (is_error)
                                helper.finnish (line);
                        });
                        helper.done.connect ((error_message) => {
                            unowned string shell = Environment.get_variable ("SHELL");
                            if (error_message.has_prefix (shell + ": ")) {
                                /* i18n: A command line tool, eg. msmtp, is missing */
                                sent (account, filename,
                                      _("%s couldn't be executed.").printf ("msmtp"));
                            }
                            else {
                                sent (account, filename,
                                    (error_message ?? "").replace ("msmtp: ", ""));
                            }
                        });
                        helper.execute (command);
                        return;
                    }
            } catch (Error error) {
                sent (account, filename, error.message);
                return;
            }
            sent (account, filename, _("Account \"%s\" doesn't exist").printf (account));
        }

        public signal void sent (string account, string filename, string error_message);

        public void quit () {
            Gtk.main_quit ();
        }
    }

    public class Service {
        void bus_aquired (DBusConnection conn) {
            try {
                conn.register_object ("/org/elementary/postler",
                                      new PostlerService ());
            } catch (IOError e) {
                stderr.printf ("Could not register service\n");
                done (1);
            }
        }

        void name_aquired () {
        }

        void name_lost () {
            done (0);
        }

        public void run () {
            Bus.own_name (BusType.SESSION, "org.elementary.Postler",
                          BusNameOwnerFlags.NONE,
                          bus_aquired, name_aquired, name_lost);
            Gtk.main ();
            done (0);
        }

        public signal void done (int status);
    }
}

