/*
 Copyright (C) 2011 Christian Dywan <christian@twotoasts.de>

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 See the file COPYING for the full license text.
*/

public class Postler.Attachments : Gtk.IconView {
    public bool editable { get; set; default = false; }
    public int icon_size { get; set; default = 32; }
    int attachment_icon_size = 0;

    public Attachments () {
        model = new Gtk.ListStore (1, typeof (MessagePart));

        selection_mode = Gtk.SelectionMode.BROWSE;
        set ("item-orientation", Gtk.Orientation.HORIZONTAL);
        /* Allow the icon view to shrink, instead of only growing larger */
        set_size_request (128, -1);
        set_no_show_all (true);

        var renderer_pixbuf = new Gtk.CellRendererPixbuf ();
        pack_start (renderer_pixbuf, true);
        set_cell_data_func (renderer_pixbuf, render_pixbuf);
        var renderer_filename = new Gtk.CellRendererText ();
        pack_start (renderer_filename, false);
        set_cell_data_func (renderer_filename, render_filename);
        var renderer_delete = new Postler.CellRendererToggle ();
        pack_end (renderer_delete, false);
        set_cell_data_func (renderer_delete, render_delete);
        renderer_delete.toggled.connect ((renderer, path) => {
            remove_attachment (path);
        });

        key_press_event.connect ((event) => {
            if (!editable)
                return false;
            if (event.keyval == Gdk.keyval_from_name ("Delete")) {
                var path = get_selected_items ().nth_data (0);
                if (path != null)
                    remove_attachment (path.to_string ());
                return true;
            }
            return false;
        });
    }

    public void add_part (MessagePart part) {
        (model as Gtk.ListStore).insert_with_values (null, -1, 0, part);
        show ();
    }

    public void add_filename (string filename) {
        uchar[] data = {};
        bool uncertain;
        string mime_type = g_content_type_guess (filename, data, out uncertain);
        var part = new MessagePart (mime_type);
        part.filename = filename;
        add_part (part);
    }

    public void remove_attachment (string tree_path) {
        Gtk.TreeIter iter = Gtk.TreeIter ();
        if (!model.get_iter_from_string (out iter, tree_path))
            return;

        (model as Gtk.ListStore).remove (iter);
        if (model.iter_n_children (null) == 0)
            hide ();
    }

    public void remove_part (MessagePart part) {
        Gtk.TreeIter iter;
        if (!model.iter_children (out iter, null))
            return;
        do {
            MessagePart existing_part;
            model.get (iter, 0, out existing_part);
            if (existing_part == part) {
                (model as Gtk.ListStore).remove (iter);
                if (model.iter_n_children (null) == 0)
                    hide ();
                break;
            }
        } while (model.iter_next (ref iter));
    }

    public MessagePart? get_selected_part () {
        var paths = get_selected_items ();
        var path = paths != null ? paths.nth_data (0) : null;
        if (path != null) {
            Gtk.TreeIter iter;
            if (model.get_iter (out iter, path)) {
                MessagePart part;
                model.get (iter, 0, out part);
                return part;
            }
        }

        return null;
    }

    public void select (MessagePart part) {
        Gtk.TreeIter iter;
        if (!model.iter_children (out iter, null))
            return;
        do {
            MessagePart existing_part;
            model.get (iter, 0, out existing_part);
            if (existing_part == part)
                select_path (model.get_path (iter));
        } while (model.iter_next (ref iter));
    }

    void render_pixbuf (Gtk.CellLayout layout, Gtk.CellRenderer cell,
        Gtk.TreeModel model, Gtk.TreeIter iter) {

        if (attachment_icon_size == 0)
            attachment_icon_size = Gtk.icon_size_register ("attachment",
                                                           icon_size, icon_size);

        MessagePart part;
        model.get (iter, 0, out part);

        cell.set ("gicon", g_content_type_get_icon (part.mime_type),
                  "stock-size", attachment_icon_size);
    }

    void render_filename (Gtk.CellLayout layout, Gtk.CellRenderer cell,
        Gtk.TreeModel model, Gtk.TreeIter iter) {

        MessagePart part;
        model.get (iter, 0, out part);

        string text = part.filename != null
            ? Filename.display_basename (part.filename)
            : g_content_type_get_description (part.mime_type);
        cell.set ("text", text,
                  "height", icon_size + item_padding,
                  "wrap-mode", Pango.WrapMode.WORD_CHAR,
                  "wrap-width", 128);
    }

    void render_delete (Gtk.CellLayout layout, Gtk.CellRenderer cell,
        Gtk.TreeModel model, Gtk.TreeIter iter) {

        if (!editable)
            return;

        cell.set ("stock-id", Gtk.STOCK_REMOVE,
                  "width", 16, "height", 16);
    }
}

