/*
 * portreserve - reserve ports to prevent portmap mapping them
 * portrelease - allow the real service to bind to its port
 * Copyright (C) 2003 Tim Waugh <twaugh@redhat.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include "config.h"

#if HAVE_DIRENT_H
# include <dirent.h>
#endif

#include <errno.h>
#include <error.h>

#if HAVE_NETDB_H
# include <netdb.h>
#endif

#if HAVE_NETINET_IN_H
# include <netinet/in.h>
#endif

#if HAVE_STDLIB_H
# include <stdlib.h>
#endif

#include <stdio.h>

#if HAVE_STRING_H
# include <string.h>
#endif

#if HAVE_SYS_SELECT_H
# include <sys/select.h>
#endif

#if HAVE_SYS_SOCKET_H
# include <sys/socket.h>
#endif

#if HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif

#include <sys/un.h>

#if HAVE_UNISTD_H
# include <unistd.h>
#endif

#define UNIX_SOCKET "/var/run/portreserve/socket"

struct map {
	struct map *next;
	char *service;
	int socket;
};

static int debug = 0;

static int
portrelease (const char *service)
{
	struct sockaddr_un addr;
	int s = socket (PF_UNIX, SOCK_DGRAM, 0);
	if (s == -1)
		return -1;
	addr.sun_family = AF_UNIX;
	strcpy (addr.sun_path, UNIX_SOCKET);
	if (connect (s, (struct sockaddr *) &addr, sizeof (addr)) == -1)
		/* portreserve not running; nothing to do. */
		return 0;
	send (s, service, strlen (service), 0);
	close (s);
	return 0;
}

static int
real_file (const char *name)
{
	if (strchr (name, '~'))
		return 0;
	if (strchr (name, '.'))
		return 0;
	return 1;
}

static int
reserve (const char *file)
{
	struct sockaddr_in sin;
	struct servent *serv;
	char service[100];
	char *p;
	int sd;
	FILE *f = fopen (file, "r");
	if (!f)
		return -1;
	p = fgets (service, sizeof (service), f);
	fclose (f);
	if (!p)
		return -1;
	p = strchr (service, '\n');
	if (p)
		*p = '\0';
	serv = getservbyname (service, NULL);
	if (!serv) {
		char *endptr;
		int port = strtol (service, &endptr, 10);
		if (service == endptr)
			return -1;
		serv = getservbyport (port, NULL);
		if (!serv)
			return -1;
	}
	sd = socket (PF_INET, SOCK_STREAM, 0);
	if (sd == -1)
		return -1;
	sin.sin_port = (in_port_t) serv->s_port;
	sin.sin_family = AF_INET;
	sin.sin_addr.s_addr = INADDR_ANY;
	if (bind (sd, (struct sockaddr *) &sin, sizeof (sin)) == -1) {
		error (0, errno, "bind");
		return -1;
	}
	if (debug)
		fprintf (stderr, "Reserved port %d for %s\n",
			 ntohs (sin.sin_port), strrchr (file, '/') + 1);
	return sd;
}

static int
unix_socket (void)
{
	struct sockaddr_un addr;
	int s = socket (PF_UNIX, SOCK_DGRAM, 0);
	if (s == -1)
		return s;
	addr.sun_family = AF_UNIX;
	strcpy (addr.sun_path, UNIX_SOCKET);
	if (bind (s, (struct sockaddr *) &addr, sizeof (addr)) == -1) {
		if (connect (s, (struct sockaddr *) &addr,
			     sizeof (addr)) == 0) {
			close (s);
			error (EXIT_FAILURE, 0, "already running");
		}
		unlink (UNIX_SOCKET);
		if (bind (s, (struct sockaddr *) &addr,
			  sizeof (addr)) == -1) {
			close (s);
			error (EXIT_FAILURE, errno, "bind");
		}
	}
	return s;
}

static void
no_memory (void)
{
	exit (1);
}

static int
portreserve (void)
{
	struct map *maps = NULL;
	const char *dir = "/etc/portreserve/";
	char *cfgfile = malloc (strlen (dir) + NAME_MAX + 1);
	char *cfgf = cfgfile + strlen (dir);
	struct dirent *d;
	int unfd = unix_socket ();
	DIR *cfgdir;
	if (!cfgfile)
		return -1;
	strcpy (cfgfile, dir);
	cfgdir = opendir (dir);
	if (!cfgdir)
		return -1;
	while ((d = readdir (cfgdir)) != NULL) {
		struct map *map;
		int s;
		if (!real_file (d->d_name))
			continue;
		strcpy (cfgf, d->d_name);
		if ((s = reserve (cfgfile)) == -1)
			continue;
		map = malloc (sizeof (*map));
		if (!map)
			no_memory ();
		map->socket = s;
		map->next = maps;
		map->service = strdup (d->d_name);
		if (!map->service)
			no_memory ();
		maps = map;
	}
	closedir (cfgdir);

	while (maps) {
		fd_set fds;
		FD_ZERO (&fds);
		FD_SET (unfd, &fds);
		if (select (unfd + 1, &fds, NULL, NULL, NULL) &&
		    FD_ISSET (unfd, &fds)) {
			char service[100];
			ssize_t got;
			struct map *m, **prev;
			got = recv (unfd, service, sizeof (service) - 1, 0);
			if (got == -1)
				continue;
			service[got] = '\0';
			prev = &maps;
			m = maps;
			while (m) {
				if (!strcmp (service, "*")) {
					struct map *next = m->next;

					if (debug)
						fprintf (stderr,
							 "Released service "
							 "%s\n", m->service);
					close (m->socket);
					free (m->service);
					free (m);
					m = next;
					maps = NULL;
				} else if (!strcmp (m->service, service)) {
					if (debug)
						fprintf (stderr,
							 "Released service "
							 "%s\n", m->service);
					close (m->socket);
					*prev = m->next;
					free (m->service);
					free (m);
					break;
				} else {
					prev = &m->next;
					m = *prev;
				}
			}
		}
	}

	if (debug)
		fprintf (stderr, "All reservations taken.\n");

	return 0;
}

int
main (int argc, char **argv)
{
	const char *p = strrchr (argv[0], '/');
	if (!p++)
		p = argv[0];

	if (strstr (p, "portrelease")) {
		int i;
		int r = 0;
		for (i = 1; i < argc; i++)
			r += portrelease (argv[i]);
		return r;
	}

	if (argc > 1) {
		if (!strcmp (argv[1], "-d"))
			debug = 1;
		else
			error (EXIT_FAILURE, 0,
			       "See the man page to find out "
			       "what this program does.");
	}

	if (!debug) {
		switch (fork ()) {
		case 0:
			break;
		case -1:
			error (EXIT_FAILURE, errno, "fork");
		default:
			return 0;
		}

		close (STDIN_FILENO);
		close (STDOUT_FILENO);
		close (STDERR_FILENO);
		setsid ();
	}

	return portreserve ();
}
