/* A daemon that broadcasts all input from each connection. */

#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <stdlib.h>
#include <errno.h>
#include <limits.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>

#define BUFFER_SIZE (1<<13)

/* Routines for reporting and logging errors */

static void
info(const char *fmt, ...)
{
  va_list ap;			/* One could add a time stamp here */

  va_start(ap, fmt);
  vfprintf(stderr, fmt, ap);
  va_end(ap);
  fputc('\n', stderr);
}

static void
report(const char *fmt, va_list ap)
{
  int err = errno;
  vfprintf(stderr, fmt, ap);
  if (err != 0) {
    fputs(": ", stderr);
    perror("");
  }
  else
    fputc('\n', stderr);
}

/* Just report problem */
static void
warn(const char *fmt, ...)
{
  va_list ap;

  va_start(ap, fmt);
  report(fmt, ap);
  va_end(ap);
}

/* Report problem and exit */
static void
fatal(const char *fmt, ...)
{
  va_list ap;

  va_start(ap, fmt);
  report(fmt, ap);
  va_end(ap);
  exit(1);
}

static int
initserver(int port)
{
  int fd;
  struct sockaddr_in sain[1];
  struct linger sl[1];

  memset(sain, 0, sizeof(*sain));
  sain->sin_family = AF_INET;
  sain->sin_addr.s_addr = INADDR_ANY;
  sain->sin_port = htons((u_short)port);

  fd = socket(PF_INET, SOCK_STREAM, 0);

  sl->l_onoff = 0;
  sl->l_linger = 0;
  if (setsockopt(fd, SOL_SOCKET, SO_LINGER, (char *)sl, sizeof(*sl)) < 0)
    fatal("cannot remove linger");
  if (fd < 0)
    fatal("cannot create master socket");
  if (bind(fd, (struct sockaddr *)sain, sizeof(*sain)) < 0)
    fatal("cannot bind master socket");
  if (listen(fd, 10) < 0)
    fatal("cannot listen on master socket");
  return fd;
}

static int
start(int port)			/* Main loop in server */
{
  fd_set afds[1];		/* All file descriptors */
  fd_set rfds[1];		/* Readable file descriptors */
  struct timeval t[1];
  struct sockaddr_in fsin[1];
  int msock = initserver(port);
  int nfds = msock + 1;
  socklen_t alen;
  int n, fd, cc;
  char buf[BUFFER_SIZE];

  FD_ZERO(afds);
  FD_SET(msock, afds);
  t->tv_sec = 6 * 60 * 60;	/* A quarter of a day */
  t->tv_usec = 0;

  info("Broadcast %d listening on %d", getpid(), port);

  for (;;) {
    memcpy((char *)rfds, (char *)afds, sizeof(*rfds));
    if (select(nfds, rfds, (fd_set *)0, (fd_set *)0, t) < 0) {
      if (errno == EINTR)
	continue;
      else
	fatal("select error");
    }

    /* Check for time out */
    for(fd = 0;; fd++)
      if (fd >= nfds) {
	info("Server timed out");
	return 0;
      }
      else if (FD_ISSET(fd, rfds))
	break;

    /* Read data input. */
    n = 0;
    for (fd = 0; fd < nfds; fd++)
      if (fd != msock && FD_ISSET(fd, rfds)) {
	cc = read(fd, buf + n, BUFFER_SIZE - n);
	if (cc <= 0) {
	  if (cc < 0)
	    warn("error reading");
	  info("closing file descriptor %d", fd);
	  close(fd);
	  FD_CLR(fd, afds);
	  FD_CLR(fd, rfds);
	}
	else
	  n += cc;
      }

    if (n > 0)			/* There is data to broadcast */
      for (fd = 0; fd < nfds; fd++)
	if (fd != msock && FD_ISSET(fd, afds)) {
	  cc = write(fd, buf, n);
	  if (cc < 0) {
	    info("closing file descriptor %d", fd);
	    close(fd);
	    FD_CLR(fd, afds);
	  }
	}

    if (FD_ISSET(msock, rfds)) { /* Add connections here */
      alen = sizeof(*fsin);
      fd = accept(msock, (struct sockaddr *)fsin, &alen);
      if (fd < 0)
	fatal("accept error");
      if (nfds <= fd) nfds = fd + 1;
      info("accepting file descriptor %d", fd);
      FD_SET(fd, afds);
    }
  }
  return 0;
}

/*
 * daemonize runs the server as a daemon.  It opens the log file as
 * standard error, forks a process, makes the process into a session
 * leader, and changes the current directory to "/" so that the file
 * system from which the server was started can be unmounted while
 * the server is running.
 */
static void
daemonize(char *log)
{
  int pid, i, n;

  if (!log)
    fatal("No log file specified");

  if ((pid = fork()) < 0) {
    perror("fork");
    exit(1);
  }
  else if (pid != 0)
    exit(0);

  setsid();

  if (chdir("/") < 0) {
    perror("chdir");
    exit(1);
  }

  umask(037);

  n = (int)sysconf(_SC_OPEN_MAX);

  for (i = 0; i < n; i++)
    close(i);

  if (freopen(log, "w", stderr) == NULL) {
    perror("Cannot open log file");
    exit(1);
  }
  setvbuf(stderr, NULL, _IOLBF, BUFSIZ);
}

static int
usage(char *prog)
{
  fprintf(stderr,
	  "Usage: %s [-d log] port \n",
	  prog);
  return 1;
}

int
main(int argc, char *argv[])
{
  int c, errflg = 0;
  char *log = NULL;
  extern int optind;
  extern char *optarg;

  /* process program arguments */
  while ((c = getopt(argc, argv, "d:")) != -1)
    switch (c) {
    case 'd':			/* Run as a daemon */
      log = optarg;
      break;
    default:
      errflg++;
    }
  if (errflg)
    return usage(argv[0]);

  if (argc - optind != 1)
    return usage(argv[0]);

  c = atoi(argv[optind]);
  if (c <= 0)
    fatal("bad port %d", c);

  if (log)
    daemonize(log);

  return start(c);
}
