/* Copyright (c) 2001-2010, David A. Clunie DBA Pixelmed Publishing. All rights reserved. */

package com.pixelmed.test;

import com.pixelmed.dose.*;

import junit.framework.*;

import com.pixelmed.dicom.*;

public class TestCTDose extends TestCase {
	
	// constructor to support adding tests to suite ...
	
	public TestCTDose(String name) {
		super(name);
	}
	
	// add tests to suite manually, rather than depending on default of all test...() methods
	// in order to allow adding TestCTDose.suite() in AllTests.suite()
	// see Johannes Link. Unit Testing in Java pp36-47
	
	public static Test suite() {
		TestSuite suite = new TestSuite("TestCTDose");
		
		suite.addTest(new TestCTDose("testCTDoseConstructor_WithAllParameters"));
		suite.addTest(new TestCTDose("testCTDoseConstructor_WithAllParametersAndThreeAcquisitionsInTwoSeries"));
		suite.addTest(new TestCTDose("testCTDoseConstructor_WithOneAcquisitionsAndNoTotalDLP"));
		
		return suite;
	}
	
	private CommonDoseObserverContext observerContext;
	private CompositeInstanceContext compositeInstanceContext;
	
	private String deviceUID = "1.2.3.4";
	private String deviceName = "station1";
	private String manufacturer = "Acme";
	private String modelName = "Scanner";
	private String serialNumber = "72349236741";
	private String location = "Suite1";
	
	private String operatorName = "Smith^John";
	private String operatorID = "26354781234";
	private String physicianName = "Jones^Mary";
	private String physicianID = "23491234234";
	private String idIssuer = "99BLA";
	private String organization = "St. Elsewhere's";

	private String patientName = "Smith^Mary";
	private String patientID = "3764913624";
	private String patientBirthDate = "19600101";
	private String patientSex = "F";
	private String studyID = "612386812";
	private String seriesNumber = "12";
	private String instanceNumber = "38";
	private String referringPhysicianName = "Jones^Harriet";
	private String studyDate = "20010203";
	private String studyTime = "043000";
	
	private UIDGenerator u = new UIDGenerator("9999");
	private String sopInstanceUID;
	private String seriesInstanceUID;
	private String studyInstanceUID;
	
	protected void setUp() {
		observerContext = new CommonDoseObserverContext(deviceUID,deviceName,manufacturer,modelName,serialNumber,location,operatorName,operatorID,physicianName,physicianID,idIssuer,organization);

		try {
			UIDGenerator u = new UIDGenerator("9999");
			sopInstanceUID = u.getNewSOPInstanceUID(studyID,seriesNumber,instanceNumber);
			seriesInstanceUID = u.getNewSeriesInstanceUID(studyID,seriesNumber);
			studyInstanceUID = u.getNewStudyInstanceUID(studyID);
			AttributeList list = new AttributeList();
			{ Attribute a = new UniqueIdentifierAttribute(TagFromName.SOPInstanceUID); a.addValue(sopInstanceUID); list.put(a); }
			{ Attribute a = new UniqueIdentifierAttribute(TagFromName.SeriesInstanceUID); a.addValue(seriesInstanceUID); list.put(a); }
			{ Attribute a = new UniqueIdentifierAttribute(TagFromName.StudyInstanceUID); a.addValue(studyInstanceUID); list.put(a); }
			{ Attribute a = new PersonNameAttribute(TagFromName.PatientName); a.addValue(patientName); list.put(a); }
			{ Attribute a = new LongStringAttribute(TagFromName.PatientID); a.addValue(patientID); list.put(a); }
			{ Attribute a = new DateAttribute(TagFromName.PatientBirthDate); a.addValue(patientBirthDate); list.put(a); }
			{ Attribute a = new CodeStringAttribute(TagFromName.PatientSex); a.addValue(patientSex); list.put(a); }
			{ Attribute a = new ShortStringAttribute(TagFromName.StudyID); a.addValue(studyID); list.put(a); }
			{ Attribute a = new PersonNameAttribute(TagFromName.ReferringPhysicianName); a.addValue(referringPhysicianName); list.put(a); }
			{ Attribute a = new IntegerStringAttribute(TagFromName.SeriesNumber); a.addValue(seriesNumber); list.put(a); }
			{ Attribute a = new IntegerStringAttribute(TagFromName.InstanceNumber); a.addValue(instanceNumber); list.put(a); }
			{ Attribute a = new LongStringAttribute(TagFromName.Manufacturer); list.put(a); }
			{ Attribute a = new DateAttribute(TagFromName.StudyDate); a.addValue(studyDate); list.put(a); }
			{ Attribute a = new TimeAttribute(TagFromName.StudyTime); a.addValue(studyTime); list.put(a); }
//System.err.println("TestCTDose.setUp(): compositeInstanceContext.getAttributeList() =\n"+list);
			compositeInstanceContext = new CompositeInstanceContext(list);
		}
		catch (DicomException e) {
		}
	}
	
	protected void tearDown() {
	}
	
	public void testCTDoseConstructor_WithAllParameters() throws Exception {
		
		String dlpTotal = "1299.58";
		int totalNumberOfIrradiationEvents = 4;
		String scopeUID = "1.2.124.113532.172.16.11.4.20090807.144612.3424396";
		String startDateTime = "20010203043000+0000";
		String endDateTime   = "20010203043500+0000";
		String description   = "CAP";
		
		String expectToStringDetail = "Dose\tStart="+startDateTime+"\tEnd="+endDateTime+"\tModality=CT\tDescription="+description+"\tScope=Study\tUID="+scopeUID+"\tEvents="+Integer.toString(totalNumberOfIrradiationEvents)+"\tDLP Total="+dlpTotal+" mGycm\n";
		String expectToStringNoDetail = "Dose\tStart="+startDateTime+"\tModality=CT\tDescription="+description+"\tDLP Total="+dlpTotal+" mGycm\n";
		
		CTDose ctDose = new CTDose(dlpTotal,totalNumberOfIrradiationEvents,ScopeOfDoseAccummulation.STUDY,scopeUID,startDateTime,endDateTime,description);
		
		assertEquals("Checking DLP Total",dlpTotal,ctDose.getDLPTotal());
		assertEquals("Checking totalNumberOfIrradiationEvents",totalNumberOfIrradiationEvents,ctDose.getTotalNumberOfIrradiationEvents());
		assertEquals("Checking ScopeOfDoseAccummulation equality",ScopeOfDoseAccummulation.STUDY,ctDose.getScopeOfDoseAccummulation());
		assertEquals("Checking ScopeOfDoseAccummulation string equality","Study",ctDose.getScopeOfDoseAccummulation().toString());
		assertEquals("Checking scopeUID string equality",scopeUID,ctDose.getScopeUID());
		assertEquals("Checking startDateTime string equality",startDateTime,ctDose.getStartDateTime());
		assertEquals("Checking endDateTime string equality",endDateTime,ctDose.getEndDateTime());
		assertEquals("Checking description string equality",description,ctDose.getDescription());
		
		assertEquals("Checking toString default",expectToStringDetail,ctDose.toString());
		assertEquals("Checking toString detail",expectToStringDetail,ctDose.toString(true,false));
		assertEquals("Checking toString no detail",expectToStringNoDetail,ctDose.toString(false,false));

		assertFalse("Checking SR is not null",ctDose.getStructuredReport() == null);
	}
	
	public void testCTDoseConstructor_WithAllParametersAndThreeAcquisitionsInTwoSeries() throws Exception {
		
		CTDoseAcquisition acq0 = new CTDoseAcquisition("2",CTScanType.HELICAL,new ScanRange("S", "14.250","I","635.750"),"20.23","1362.24",CTPhantomType.selectFromDescription("BODY32"));
		CTDoseAcquisition acq1 = new CTDoseAcquisition("2",CTScanType.HELICAL,new ScanRange("I","635.250","I","665.250"),"20.23", "107.73",CTPhantomType.selectFromDescription("BODY32"));
		CTDoseAcquisition acq2 = new CTDoseAcquisition("3",CTScanType.HELICAL,new ScanRange("S", "14.250","S", "84.250"),"20.23", "172.99",CTPhantomType.selectFromDescription("BODY32"));

		String dlpTotal = "1642.96";
		int totalNumberOfIrradiationEvents = 3;
		String scopeUID = "1.2.124.113532.172.16.11.4.20090807.144612.3424396";
		String startDateTime = "20010203043000+0000";
		String startDateTimeFormatted = "2001/02/03 04:30:00";
		String endDateTime   = "20010203043500+0000";
		String endDateTimeFormatted   = "2001/02/03 04:35:00";
		String description   = "CAP";
	
		String expectToStringDetail =
			  "Dose\tStart="+startDateTime+"\tEnd="+endDateTime+"\tModality=CT\tDescription="+description+"\tScope=Study\tUID="+scopeUID+"\tEvents="+Integer.toString(totalNumberOfIrradiationEvents)+"\tDLP Total="+dlpTotal+" mGycm\n"
			+ "\tSeries=2\tHelical\tRange=S14.250-I635.750 mm\tCTDIvol=20.23 mGy\tDLP=1362.24 mGycm\tPhantom=BODY32\n"
			+ "\tSeries=2\tHelical\tRange=I635.250-I665.250 mm\tCTDIvol=20.23 mGy\tDLP=107.73 mGycm\tPhantom=BODY32\n"
			+ "\tSeries=3\tHelical\tRange=S14.250-S84.250 mm\tCTDIvol=20.23 mGy\tDLP=172.99 mGycm\tPhantom=BODY32\n"
			;
		
		String expectToStringNoDetail =
			  "Dose\tStart="+startDateTime+"\tModality=CT\tDescription="+description+"\tDLP Total="+dlpTotal+" mGycm\n";
		
		String expectToStringNoDetailPretty =
			  "Dose\t"+startDateTimeFormatted+"\tCT\t"+description+"\tDLP Total="+dlpTotal+" mGycm\n";
		
		CTDose ctDose = new CTDose(ScopeOfDoseAccummulation.STUDY,scopeUID,startDateTime,endDateTime,description);
		ctDose.addAcquisition(acq0);
		ctDose.addAcquisition(acq1);
		ctDose.addAcquisition(acq2);
		ctDose.setDLPTotal(dlpTotal);
		
		assertEquals("Checking DLP Total",dlpTotal,ctDose.getDLPTotal());
		assertTrue("Checking DLP Total as double",Double.parseDouble(dlpTotal) == Double.parseDouble(ctDose.getDLPTotal()));
		assertEquals("Checking totalNumberOfIrradiationEvents",totalNumberOfIrradiationEvents,ctDose.getTotalNumberOfIrradiationEvents());
		assertTrue("Checking ScopeOfDoseAccummulation equality",ScopeOfDoseAccummulation.STUDY.equals(ctDose.getScopeOfDoseAccummulation()));
		assertEquals("Checking ScopeOfDoseAccummulation string equality","Study",ctDose.getScopeOfDoseAccummulation().toString());
		assertEquals("Checking scopeUID string equality",scopeUID,ctDose.getScopeUID());
		assertEquals("Checking startDateTime string equality",startDateTime,ctDose.getStartDateTime());
		assertEquals("Checking endDateTime string equality",endDateTime,ctDose.getEndDateTime());
		assertEquals("Checking description string equality",description,ctDose.getDescription());

		assertEquals("Checking number of acquisitions",3,ctDose.getNumberOfAcquisitions());
		assertTrue("Checking CTDoseAcquisition 1 equality",acq0.equals(ctDose.getAcquisition(0)));
		assertTrue("Checking CTDoseAcquisition 2 equality",acq1.equals(ctDose.getAcquisition(1)));
		assertTrue("Checking CTDoseAcquisition 3 equality",acq2.equals(ctDose.getAcquisition(2)));

		assertTrue("Checking DLP Total as double with total from acquisitions",ctDose.specifiedDLPTotalMatchesDLPTotalFromAcquisitions());

		assertEquals("Checking toString default",expectToStringDetail,ctDose.toString());
		assertEquals("Checking toString detail",expectToStringDetail,ctDose.toString(true,false));
		assertEquals("Checking toString no detail",expectToStringNoDetail,ctDose.toString(false,false));
		assertEquals("Checking toString no detail pretty",expectToStringNoDetailPretty,ctDose.toString(false,true));
		
		assertFalse("Checking SR is not null",ctDose.getStructuredReport() == null);
	}
	
	public void testCTDoseConstructor_WithOneAcquisitionsAndNoTotalDLP() throws Exception {
		
		String dlp =  "1362.24";
		String ctdiVol = "20.23";
		CTDoseAcquisition acq0 = new CTDoseAcquisition("2",CTScanType.HELICAL,new ScanRange("S", "14.250","I","635.750"),ctdiVol,dlp,CTPhantomType.selectFromDescription("BODY32"));
		
		String irradiationEventUID = "1.2.3.4";
		CodedSequenceItem anatomy = new CodedSequenceItem("R-FAB56","SRT","Chest, Abdomen and Pelvis");
		String exposureTimeInSeconds = "1";
		String kvp = "120";
		String tubeCurrent = "397";
		String exposureTimePerRotation = "0.6";
		String nominalSingleCollimationWidthInMM = "0.625";
		String nominalTotalCollimationWidthInMM = "40";
		String pitchFactor = "0.984";

		CTAcquisitionParameters acqparam0 = new CTAcquisitionParameters(irradiationEventUID,anatomy,exposureTimeInSeconds,null,nominalSingleCollimationWidthInMM,nominalTotalCollimationWidthInMM,pitchFactor,kvp,tubeCurrent,exposureTimePerRotation);
		acqparam0.deriveScanningLengthFromDLPAndCTDIVol(dlp,ctdiVol);
		String scanningLength = com.pixelmed.utils.FloatFormatter.toString(1362.24/20.23*10);
		
		String dlpTotalFromEvents = dlp;

		String dlpTotal = null;
		int totalNumberOfIrradiationEvents = 1;
		String scopeUID = "1.2.124.113532.172.16.11.4.20090807.144612.3424396";
		String startDateTime = "20010203043000+0000";
		String endDateTime   = "20010203043500+0000";
		String description   = "CAP";
		
		CTDose ctDose = new CTDose(ScopeOfDoseAccummulation.STUDY,scopeUID,startDateTime,endDateTime,description);
		acq0.setAcquisitionParameters(acqparam0);
		ctDose.addAcquisition(acq0);
		ctDose.setDLPTotal(dlpTotal);
		
		ctDose.setObserverContext(observerContext);
		ctDose.setCompositeInstanceContext(compositeInstanceContext);
		
		assertEquals("Checking null DLP Total",dlpTotal,ctDose.getDLPTotal());
		assertFalse("Checking null DLP Total does not match total from acquisitions",ctDose.specifiedDLPTotalMatchesDLPTotalFromAcquisitions());
		assertEquals("Checking DLP total from acquisitions",dlpTotalFromEvents,ctDose.getDLPTotalFromAcquisitions());
		
		StructuredReport sr = ctDose.getStructuredReport();
		assertFalse("Checking SR is not null",sr == null);
		AttributeList list = ctDose.getAttributeList();
		assertFalse("Checking SR AttributeList is not null",list == null);
		org.w3c.dom.Document srDocument = new XMLRepresentationOfStructuredReportObjectFactory().getDocument(sr,list);
		assertFalse("Checking SR document is not null",srDocument == null);
//System.err.println("srDocument =\n");
//XMLRepresentationOfStructuredReportObjectFactory.write(System.err,srDocument);
		
		assertEquals("Checking document title","X-Ray Radiation Dose Report",javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/concept/@cm",srDocument));
		assertEquals("Checking procedure reported","Computed Tomography X-Ray",javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/code[concept/@cv='121058']/value/@cm",srDocument));
		assertEquals("Checking diagnostic intent","Diagnostic Intent",javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/code[concept/@cv='121058']/code[concept/@cv='G-C0E8']/value/@cm",srDocument));

		assertEquals("Checking observer type","Device",javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/code[concept/@cv='121005']/value/@cm",srDocument));
		assertEquals("Checking Device Observer UID",deviceUID,javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/uidref[concept/@cv='121012']/value",srDocument));
		assertEquals("Checking Device Observer Name",deviceName,javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/text[concept/@cv='121013']/value",srDocument));
		assertEquals("Checking Device Observer Manufacturer",manufacturer,javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/text[concept/@cv='121014']/value",srDocument));
		assertEquals("Checking Device Observer Model Name",modelName,javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/text[concept/@cv='121015']/value",srDocument));
		assertEquals("Checking Device Observer Serial Number",serialNumber,javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/text[concept/@cv='121016']/value",srDocument));
		assertEquals("Checking Device Observer Physical Location during observation",location,javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/text[concept/@cv='121017']/value",srDocument));

		assertEquals("Checking Start of X-Ray Irradiation",startDateTime,javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/datetime[concept/@cv='113809']/value",srDocument));
		assertEquals("Checking End of X-Ray Irradiation",endDateTime,javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/datetime[concept/@cv='113810']/value",srDocument));

		assertEquals("Checking scope of accumulation","Study",javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/code[concept/@cv='113705']/value/@cm",srDocument));
		assertEquals("Checking scope of accumulation UIDREF",scopeUID,javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/code[concept/@cv='113705']/uidref/value",srDocument));

		// CONTAINER: (113811,DCM,"CT Accumulated Dose Data")
		assertEquals("Checking Total Number of Irradiation Events","1",           javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/container[concept/@cv='113811']/num[concept/@cv='113812']/value",srDocument));
		assertEquals("Checking Total Number of Irradiation Events units","events",javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/container[concept/@cv='113811']/num[concept/@cv='113812']/units/@cm",srDocument));
		assertEquals("Checking CT Dose Length Product Total",dlpTotalFromEvents,javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/container[concept/@cv='113811']/num[concept/@cv='113813']/value",srDocument));
		assertEquals("Checking CT Dose Length Product Total units","mGycm",     javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/container[concept/@cv='113811']/num[concept/@cv='113813']/units/@cm",srDocument));

		// CONTAINER: (113819,DCM,"CT Acquisition")
		assertEquals("Checking Target Region","Chest, Abdomen and Pelvis",javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/container[concept/@cv='113819']/code[concept/@cv='123014']/value/@cm",srDocument));
		assertEquals("Checking CT Acquisition Type","Spiral Acquisition",javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/container[concept/@cv='113819']/code[concept/@cv='113820']/value/@cm",srDocument));
		assertEquals("Checking Irradiation Event UID",irradiationEventUID,javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/container[concept/@cv='113819']/uidref[concept/@cv='113769']/value",srDocument));

		assertEquals("Checking ExposureTime",exposureTimeInSeconds,javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/container[concept/@cv='113819']/container[concept/@cv='113822']/num[concept/@cv='113824']/value",srDocument));
		assertEquals("Checking ExposureTime units","s",            javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/container[concept/@cv='113819']/container[concept/@cv='113822']/num[concept/@cv='113824']/units/@cm",srDocument));
		assertEquals("Checking Scanning Length",scanningLength,javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/container[concept/@cv='113819']/container[concept/@cv='113822']/num[concept/@cv='113825']/value",srDocument));
		assertEquals("Checking Scanning Length units","mm"    ,javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/container[concept/@cv='113819']/container[concept/@cv='113822']/num[concept/@cv='113825']/units/@cm",srDocument));
		assertEquals("Checking Nominal Single Collimation Width",nominalSingleCollimationWidthInMM,javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/container[concept/@cv='113819']/container[concept/@cv='113822']/num[concept/@cv='113826']/value",srDocument));
		assertEquals("Checking Nominal Single Collimation Width units","mm"                       ,javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/container[concept/@cv='113819']/container[concept/@cv='113822']/num[concept/@cv='113826']/units/@cm",srDocument));
		assertEquals("Checking Nominal Total Collimation Width",nominalTotalCollimationWidthInMM,javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/container[concept/@cv='113819']/container[concept/@cv='113822']/num[concept/@cv='113827']/value",srDocument));
		assertEquals("Checking Nominal Total Collimation Width units","mm"                      ,javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/container[concept/@cv='113819']/container[concept/@cv='113822']/num[concept/@cv='113827']/units/@cm",srDocument));
		assertEquals("Checking Pitch Factor",pitchFactor  ,javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/container[concept/@cv='113819']/container[concept/@cv='113822']/num[concept/@cv='113828']/value",srDocument));
		assertEquals("Checking Pitch Factor units","ratio",javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/container[concept/@cv='113819']/container[concept/@cv='113822']/num[concept/@cv='113828']/units/@cm",srDocument));

		assertEquals("Checking Number of X-Ray Sources","1",                  javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/container[concept/@cv='113819']/container[concept/@cv='113822']/num[concept/@cv='113823']/value",srDocument));
		assertEquals("Checking Number of X-Ray Sources units","X-Ray sources",javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/container[concept/@cv='113819']/container[concept/@cv='113822']/num[concept/@cv='113823']/units/@cm",srDocument));
		assertEquals("Checking Identification Number of the X-Ray Source","1",javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/container[concept/@cv='113819']/container[concept/@cv='113822']/container[concept/@cv='113831']/text[concept/@cv='113832']/value",srDocument));
		assertEquals("Checking KVP",kvp,       javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/container[concept/@cv='113819']/container[concept/@cv='113822']/container[concept/@cv='113831']/num[concept/@cv='113733']/value",srDocument));
		assertEquals("Checking KVP units","kV",javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/container[concept/@cv='113819']/container[concept/@cv='113822']/container[concept/@cv='113831']/num[concept/@cv='113733']/units/@cm",srDocument));
		assertEquals("Checking Maximum X-Ray Tube Current",tubeCurrent,javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/container[concept/@cv='113819']/container[concept/@cv='113822']/container[concept/@cv='113831']/num[concept/@cv='113833']/value",srDocument));
		assertEquals("Checking Maximum X-Ray Tube Current units","mA", javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/container[concept/@cv='113819']/container[concept/@cv='113822']/container[concept/@cv='113831']/num[concept/@cv='113833']/units/@cm",srDocument));
		assertEquals("Checking X-Ray Tube Current",tubeCurrent,javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/container[concept/@cv='113819']/container[concept/@cv='113822']/container[concept/@cv='113831']/num[concept/@cv='113734']/value",srDocument));
		assertEquals("Checking X-Ray Tube Current units","mA", javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/container[concept/@cv='113819']/container[concept/@cv='113822']/container[concept/@cv='113831']/num[concept/@cv='113734']/units/@cm",srDocument));
		assertEquals("Checking Exposure Time per Rotation",exposureTimePerRotation,javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/container[concept/@cv='113819']/container[concept/@cv='113822']/container[concept/@cv='113831']/num[concept/@cv='113834']/value",srDocument));
		assertEquals("Checking Exposure Time per Rotation","s",                    javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/container[concept/@cv='113819']/container[concept/@cv='113822']/container[concept/@cv='113831']/num[concept/@cv='113834']/units/@cm",srDocument));
		
		assertEquals("Checking Mean CTDIvol",ctdiVol,    javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/container[concept/@cv='113819']/container[concept/@cv='113829']/num[concept/@cv='113830']/value",srDocument));
		assertEquals("Checking Mean CTDIvol units","mGy",javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/container[concept/@cv='113819']/container[concept/@cv='113829']/num[concept/@cv='113830']/units/@cm",srDocument));
		assertEquals("Checking CTDIw Phantom Type","IEC Body Dosimetry Phantom",javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/container[concept/@cv='113819']/container[concept/@cv='113829']/code[concept/@cv='113835']/value/@cm",srDocument));
		assertEquals("Checking DLP",dlp,          javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/container[concept/@cv='113819']/container[concept/@cv='113829']/num[concept/@cv='113838']/value",srDocument));
		assertEquals("Checking DLP units","mGycm",javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/container[concept/@cv='113819']/container[concept/@cv='113829']/num[concept/@cv='113838']/units/@cm",srDocument));

		assertEquals("Checking Device Role in Procedure","Irradiating Device",javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/container[concept/@cv='113819']/code[concept/@cv='113876']/value/@cm",srDocument));
		assertEquals("Checking Device Manufacturer",manufacturer,             javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/container[concept/@cv='113819']/code[concept/@cv='113876']/text[concept/@cv='113878']/value",srDocument));
		assertEquals("Checking Device Model Name",modelName,                  javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/container[concept/@cv='113819']/code[concept/@cv='113876']/text[concept/@cv='113879']/value",srDocument));
		assertEquals("Checking Device Serial Number",serialNumber,            javax.xml.xpath.XPathFactory.newInstance().newXPath().evaluate("/DicomStructuredReport/DicomStructuredReportContent/container[concept/@cv='113701']/container[concept/@cv='113819']/code[concept/@cv='113876']/text[concept/@cv='113880']/value",srDocument));
		
		String expectValidationString = 
		 "Found XRayRadiationDoseSR IOD\n"
		+"Parent content item (1.12.7: PNAME) has illegal relationship HAS PROPERTIES with child content item (1.12.7.1: CODE)\n"
		+"Parent content item (1.12.7: PNAME) has illegal relationship HAS PROPERTIES with child content item (1.12.7.2: TEXT)\n"
		+"Parent content item (1.12.7: PNAME) has illegal relationship HAS PROPERTIES with child content item (1.12.7.3: TEXT)\n"
		+"Parent content item (1.12.7: PNAME) has illegal relationship HAS PROPERTIES with child content item (1.12.7.4: TEXT)\n"
		+"Parent content item (1.12.7: PNAME) has illegal relationship HAS PROPERTIES with child content item (1.12.7.5: CODE)\n"
		+"Parent content item (1.15: PNAME) has illegal relationship HAS PROPERTIES with child content item (1.15.1: CODE)\n"
		+"Parent content item (1.15: PNAME) has illegal relationship HAS PROPERTIES with child content item (1.15.2: TEXT)\n"
		+"Parent content item (1.15: PNAME) has illegal relationship HAS PROPERTIES with child content item (1.15.3: TEXT)\n"
		+"Parent content item (1.15: PNAME) has illegal relationship HAS PROPERTIES with child content item (1.15.4: TEXT)\n"
		+"Parent content item (1.15: PNAME) has illegal relationship HAS PROPERTIES with child content item (1.15.5: CODE)\n"	
		+"Found Root Template TID_10011 (CTRadiationDose)\n"
		+"Root Template Validation Complete\n"
		+"IOD validation complete\n"
		;
		com.pixelmed.validate.DicomSRValidator validator = new com.pixelmed.validate.DicomSRValidator();
System.err.println("validate =\n"+validator.validate(list));
		assertEquals("Checking validation from StructuredReport Document",expectValidationString,validator.validate(list));
		
		assertEquals("Checking PatientName",patientName,Attribute.getSingleStringValueOrEmptyString(list,TagFromName.PatientName));
		assertEquals("Checking PatientID",patientID,Attribute.getSingleStringValueOrEmptyString(list,TagFromName.PatientID));
		assertEquals("Checking PatientBirthDate",patientBirthDate,Attribute.getSingleStringValueOrEmptyString(list,TagFromName.PatientBirthDate));
		assertEquals("Checking PatientSex",patientSex,Attribute.getSingleStringValueOrEmptyString(list,TagFromName.PatientSex));

		assertEquals("Checking StudyInstanceUID",studyInstanceUID,Attribute.getSingleStringValueOrEmptyString(list,TagFromName.StudyInstanceUID));
		assertEquals("Checking StudyID",studyID,Attribute.getSingleStringValueOrEmptyString(list,TagFromName.StudyID));
		assertEquals("Checking ReferringPhysicianName",referringPhysicianName,Attribute.getSingleStringValueOrEmptyString(list,TagFromName.ReferringPhysicianName));
		assertEquals("Checking AccessionNumber","",Attribute.getSingleStringValueOrEmptyString(list,TagFromName.AccessionNumber));
		assertEquals("Checking StudyDate",studyDate,Attribute.getSingleStringValueOrEmptyString(list,TagFromName.StudyDate));
		assertEquals("Checking StudyTime",studyTime,Attribute.getSingleStringValueOrEmptyString(list,TagFromName.StudyTime));
		
		assertEquals("Checking SeriesInstanceUID",seriesInstanceUID,Attribute.getSingleStringValueOrEmptyString(list,TagFromName.SeriesInstanceUID));
		assertEquals("Checking SeriesNumber",seriesNumber,Attribute.getSingleStringValueOrEmptyString(list,TagFromName.SeriesNumber));

		assertEquals("Checking SOPInstanceUID",sopInstanceUID,Attribute.getSingleStringValueOrEmptyString(list,TagFromName.SOPInstanceUID));
		assertEquals("Checking InstanceNumber",instanceNumber,Attribute.getSingleStringValueOrEmptyString(list,TagFromName.InstanceNumber));
		
	}
	
}
