/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment OpenGL ES-CM 1.1 plugin
 *
 * Copyright © 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Loïc Molinari <loic@fluendo.com>
 */

/*
 * The plugin duplicates all "generic" Pigment drawables in GLES drawables. The
 * GLES drawables are stored in layer (lists) inside the GLES viewport object.
 * The generic layers are duplicated when the user calls
 * pgm_viewport_set_canvas(). The duplicated GLES drawables are deleted when the
 * canvas is disposed, when another canvas is bound or when NULL is bound. The
 * GLES Viewport is connected to the Canvas 'added' and 'removed' signals so
 * that it can update the GLES drawable layers to reflect the "generic" one.
 *
 * The link between the "generic" and the GLES drawable is done through a
 * dedicated hash-table using the "generic" drawable address as the key and the
 * GL drawable address as the value. When a "generic" drawable property changes,
 * it emits the 'changed' signal to which the GLES Viewport is connected. The
 * handler stores the change as a task in an update queue which is flushed
 * during the automatic update done in the rendering thread in GLES Context.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include "pgmglesviewport.h"
#include "pgmglesimage.h"
#include "pgmglestext.h"

GST_DEBUG_CATEGORY_STATIC (pgm_gles_viewport_debug);
#define GST_CAT_DEFAULT pgm_gles_viewport_debug

/* Task types */
typedef enum {
  TASK_CHANGE,
  TASK_REORDER,
  TASK_ADD,
  TASK_REMOVE,
  TASK_REGENERATE,
  LAST_TASK
} TaskType;

/* Change task */
typedef struct {
  TaskType             type;
  PgmGlesDrawable     *glesdrawable;
  PgmDrawableProperty  property;
} TaskChange;

/* Reorder task */
typedef struct {
  TaskType          type;
  PgmGlesDrawable  *drawable;
  PgmDrawableLayer  layer;
  guint             order;
} TaskReorder;

/* Add task */
typedef struct {
  TaskType          type;
  PgmDrawable      *drawable;
  PgmDrawableLayer  layer;
  guint             order;
} TaskAdd;

/* Remove task */
typedef struct {
  TaskType          type;
  PgmDrawable      *drawable;
  PgmDrawableLayer  layer;
} TaskRemove;

/* Any task */
typedef struct {
  TaskType type;
} TaskAny;

/* Task */
typedef union {
  TaskType    type;
  TaskAny     any;
  TaskChange  change;
  TaskReorder reorder;
  TaskAdd     add;
  TaskRemove  remove;
} Task;

/* Task function type definition */
typedef void (*TaskFunc) (PgmGlesViewport *glesviewport, Task *task);

/* Task function prototypes */
static void do_task_change     (PgmGlesViewport *glesviewport, Task *task);
static void do_task_reorder    (PgmGlesViewport *glesviewport, Task *task);
static void do_task_add        (PgmGlesViewport *glesviewport, Task *task);
static void do_task_remove     (PgmGlesViewport *glesviewport, Task *task);
static void do_task_regenerate (PgmGlesViewport *glesviewport, Task *task);

/* Callback prototypes */
static void changed_cb (PgmDrawable *drawable,
                        PgmDrawableProperty property,
                        gpointer data);

/* Task function array */
static TaskFunc task_func[LAST_TASK] = {
  do_task_change,
  do_task_reorder,
  do_task_add,
  do_task_remove,
  do_task_regenerate
};

static PgmViewportClass *parent_class = NULL;

/* Private functions */

/* Create a change task */
static Task*
task_change_new (PgmGlesDrawable *glesdrawable,
                 PgmDrawableProperty property)
{
  Task *task;

  task = g_slice_new (Task);
  task->change.type = TASK_CHANGE;
  task->change.glesdrawable = gst_object_ref (glesdrawable);
  task->change.property = property;

  return task;
}

/* Free a change task */
static void
task_change_free (Task *task)
{
  g_return_if_fail (task != NULL);

  gst_object_unref (task->change.glesdrawable);

  g_slice_free (Task, task);
}

/* Create a reorder task */
static Task*
task_reorder_new (PgmDrawable *drawable,
                  PgmDrawableLayer layer,
                  guint order)
{
  Task *task;

  task = g_slice_new (Task);
  task->reorder.type = TASK_REORDER;
  task->reorder.drawable = gst_object_ref (drawable);
  task->reorder.layer = layer;
  task->reorder.order = order;

  return task;
}

/* Free a reorder task */
static void
task_reorder_free (Task *task)
{
  g_return_if_fail (task != NULL);

  gst_object_unref (task->reorder.drawable);

  g_slice_free (Task, task);
}

/* Create a add task */
static Task*
task_add_new (PgmDrawable *drawable,
              PgmDrawableLayer layer,
              guint order)
{
  Task *task;

  task = g_slice_new (Task);
  task->add.type = TASK_ADD;
  task->add.drawable = gst_object_ref (drawable);
  task->add.layer = layer;
  task->add.order = order;

  return task;
}

/* Free a add task */
static void
task_add_free (Task *task)
{
  g_return_if_fail (task != NULL);

  gst_object_unref (task->add.drawable);

  g_slice_free (Task, task);
}

/* Create a remove task */
static Task*
task_remove_new (PgmDrawable *drawable,
                 PgmDrawableLayer layer)
{
  Task *task;

  task = g_slice_new (Task);
  task->remove.type = TASK_REMOVE;
  task->remove.drawable = gst_object_ref (drawable);
  task->remove.layer = layer;

  return task;
}

/* Free a remove task */
static void
task_remove_free (Task *task)
{
  g_return_if_fail (task != NULL);

  gst_object_unref (task->remove.drawable);

  g_slice_free (Task, task);
}

/* Create a new generic task */
static Task*
task_any_new (TaskType type)
{
  Task *task;

  task = g_slice_new (Task);
  task->any.type = type;

  return task;
}

/* Create a new generic task */
static void
task_any_free (Task *task)
{
  g_slice_free (Task, task);
}

/* Free a generic task guessing the type */
static void
task_free (Task *task)
{
  switch (task->type)
    {
    case TASK_CHANGE:
      task_change_free (task);
      break;
    case TASK_REORDER:
      task_reorder_free (task);
      break;
    case TASK_ADD:
      task_add_free (task);
      break;
    case TASK_REMOVE:
      task_remove_free (task);
      break;
    default:
      task_any_free (task);
      break;
    }
}

/* Create a GLES drawable from a generic drawable */
static PgmGlesDrawable*
gles_drawable_new (PgmGlesViewport *glesviewport,
                   PgmDrawable *drawable)
{
  PgmGlesDrawable *glesdrawable = NULL;

  /* Create the Glesdrawable depending on the type */
  if (PGM_IS_IMAGE (drawable))
    {
      glesdrawable = pgm_gles_image_new (drawable, glesviewport);
      GST_DEBUG_OBJECT (glesviewport, "created %s",
                        GST_OBJECT_NAME (glesdrawable));
    }
  else if (PGM_IS_TEXT (drawable))
    {
      glesdrawable = pgm_gles_text_new (drawable, glesviewport);
      GST_DEBUG_OBJECT (glesviewport, "created %s",
                        GST_OBJECT_NAME (glesdrawable));
    }
  else
    {
      glesdrawable = NULL;
      GST_WARNING_OBJECT (glesviewport, "cannot create object from this type");
    }

  /* Make it easily retrievable */
  if (glesdrawable)
    {
      GST_OBJECT_LOCK (glesviewport);
      g_hash_table_insert (glesviewport->drawable_hash, drawable, glesdrawable);
      GST_OBJECT_UNLOCK (glesviewport);
    }

  return glesdrawable;
}

/* Free a GLES drawable */
static void
gles_drawable_free (PgmGlesViewport *glesviewport,
                    PgmGlesDrawable *glesdrawable)
{
  GSList *walk, *tmp;
  Task *task;

  /* Disconnect from the 'changed' signal and remove it from the hash */
  GST_OBJECT_LOCK (glesviewport);
  GST_OBJECT_LOCK (glesdrawable->drawable);

  g_signal_handler_disconnect (glesdrawable->drawable,
                               glesdrawable->change_handler);
  g_hash_table_remove (glesviewport->drawable_hash, glesdrawable->drawable);

  GST_OBJECT_UNLOCK (glesdrawable->drawable);
  GST_OBJECT_UNLOCK (glesviewport);

  /* Remove the glesdrawable from the update queue */
  g_mutex_lock (glesviewport->update_lock);
  walk = glesviewport->update_queue;
  while (walk)
    {
      tmp = walk->next;
      task = walk->data;
      /* Remove it only if it's a task dealing with glesdrawable */
      if (task->type == TASK_CHANGE)
        {
          if (task->change.glesdrawable == glesdrawable)
            {
              glesviewport->update_queue =
                g_slist_delete_link (glesviewport->update_queue, walk);
              task_change_free (task);
            }
        }
      walk = tmp;
    }
  g_mutex_unlock (glesviewport->update_lock);

  /* Then delete it */
  GST_DEBUG_OBJECT (glesviewport, "unreferencing %s",
                    GST_OBJECT_NAME (glesdrawable));
  gst_object_unref (glesdrawable);
}

/* Complete a change task and free it */
static void
do_task_change (PgmGlesViewport *glesviewport,
                Task *task)
{
  PgmGlesViewportClass *klass = PGM_GLES_VIEWPORT_GET_CLASS (glesviewport);

  klass->changed_func[task->change.property] (task->change.glesdrawable);

  task_change_free (task);
}

/* Complete a reorder task and free it */
static void
do_task_reorder (PgmGlesViewport *glesviewport,
                 Task *task)
{
  PgmGlesDrawable *glesdrawable;

  GST_OBJECT_LOCK (glesviewport);
  glesdrawable = g_hash_table_lookup (glesviewport->drawable_hash,
                                      task->reorder.drawable);
  GST_OBJECT_UNLOCK (glesviewport);

  if (!glesdrawable)
    goto removed;

  g_mutex_lock (glesviewport->layer_lock);

  switch (task->reorder.layer)
    {
    case PGM_DRAWABLE_NEAR:
      glesviewport->near_layer = g_list_remove (glesviewport->near_layer,
                                                glesdrawable);
      glesviewport->near_layer = g_list_insert (glesviewport->near_layer,
                                                glesdrawable,
                                                task->reorder.order);
      break;

    case PGM_DRAWABLE_MIDDLE:
      glesviewport->middle_layer = g_list_remove (glesviewport->middle_layer,
                                                  glesdrawable);
      glesviewport->middle_layer = g_list_insert (glesviewport->middle_layer,
                                                  glesdrawable,
                                                  task->reorder.order);
      break;

    case PGM_DRAWABLE_FAR:
      glesviewport->far_layer = g_list_remove (glesviewport->far_layer,
                                               glesdrawable);
      glesviewport->far_layer = g_list_insert (glesviewport->far_layer,
                                               glesdrawable,
                                               task->reorder.order);
      break;

    default:
      break;
    }

  g_mutex_unlock (glesviewport->layer_lock);

 removed:
  task_reorder_free (task);
}

/* Complete a add task and free it */
static void
do_task_add (PgmGlesViewport *glesviewport,
             Task *task)
{
  PgmGlesDrawable *glesdrawable;

  glesdrawable = gles_drawable_new (glesviewport, task->add.drawable);

  g_mutex_lock (glesviewport->layer_lock);
  switch (task->add.layer)
    {
    case PGM_DRAWABLE_NEAR:
      glesviewport->near_layer = g_list_insert (glesviewport->near_layer,
                                                glesdrawable,
                                                task->add.order);
      break;

    case PGM_DRAWABLE_MIDDLE:
      glesviewport->middle_layer = g_list_insert (glesviewport->middle_layer,
                                                  glesdrawable,
                                                  task->add.order);
      break;

    case PGM_DRAWABLE_FAR:
      glesviewport->far_layer = g_list_insert (glesviewport->far_layer,
                                               glesdrawable,
                                               task->add.order);
      break;

    default:
      break;
    }
  g_mutex_unlock (glesviewport->layer_lock);

  task_add_free (task);
}

/* Complete a remove task and free it */
static void
do_task_remove (PgmGlesViewport *glesviewport,
                Task *task)
{
  PgmGlesDrawable *glesdrawable;

  /* Get the Glesdrawable */
  GST_OBJECT_LOCK (glesviewport);
  glesdrawable = g_hash_table_lookup (glesviewport->drawable_hash,
                                      task->remove.drawable);
  GST_OBJECT_UNLOCK (glesviewport);

  /* Remove it from the list */
  g_mutex_lock (glesviewport->layer_lock);
  switch (task->remove.layer)
    {
    case PGM_DRAWABLE_NEAR:
      glesviewport->near_layer = g_list_remove (glesviewport->near_layer,
                                                glesdrawable);
      break;

    case PGM_DRAWABLE_MIDDLE:
      glesviewport->middle_layer = g_list_remove (glesviewport->middle_layer,
                                                  glesdrawable);
      break;

    case PGM_DRAWABLE_FAR:
      glesviewport->far_layer = g_list_remove (glesviewport->far_layer,
                                               glesdrawable);
      break;

    default:
      break;
    }
  g_mutex_unlock (glesviewport->layer_lock);

  /* And delete it */
  gles_drawable_free (glesviewport, glesdrawable);

  task_remove_free (task);
}

/* Complete a regenerate task and free it */
static void
do_task_regenerate (PgmGlesViewport *glesviewport,
                    Task *task)
{
  g_mutex_lock (glesviewport->layer_lock);

  g_list_foreach (glesviewport->near_layer,
                  (GFunc) pgm_gles_drawable_regenerate, NULL);
  g_list_foreach (glesviewport->middle_layer,
                  (GFunc) pgm_gles_drawable_regenerate, NULL);
  g_list_foreach (glesviewport->far_layer,
                  (GFunc) pgm_gles_drawable_regenerate, NULL);

  g_mutex_unlock (glesviewport->layer_lock);

  task_any_free (task);
}

/* Drawable 'changed' handler */
static void
changed_cb (PgmDrawable *drawable,
            PgmDrawableProperty property,
            gpointer data)
{
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (data);
  PgmGlesViewport *glesviewport = glesdrawable->glesviewport;

  GST_LOG_OBJECT (glesviewport, "drawable_changed_cb");

  if (glesviewport)
    {
      Task *task;

      /* Add the change task to the update queue */
      g_mutex_lock (glesviewport->update_lock);
      task = task_change_new (glesdrawable, property);
      glesviewport->update_queue = g_slist_prepend (glesviewport->update_queue,
                                                    task);
      g_mutex_unlock (glesviewport->update_lock);

      /* Request an update */
      pgm_gles_context_update (glesviewport->context);
    }
}

/* Canvas 'drawable-added' handler */
static void
drawable_added_cb (PgmCanvas *canvas,
                   PgmDrawable *drawable,
                   PgmDrawableLayer layer,
                   gint order,
                   gpointer data)
{
  PgmGlesViewport *glesviewport = PGM_GLES_VIEWPORT (data);
  Task *task;

  GST_DEBUG_OBJECT (glesviewport, "drawable_added_cb");

  /* Add the add task in the update queue */
  task = task_add_new (drawable, layer, order);
  g_mutex_lock (glesviewport->update_lock);
  glesviewport->update_queue = g_slist_prepend (glesviewport->update_queue,
                                                task);
  g_mutex_unlock (glesviewport->update_lock);

  /* Request an update */
  pgm_gles_context_update (glesviewport->context);
}

/* Canvas 'drawable-removed' handler */
static void
drawable_removed_cb (PgmCanvas *canvas,
                     PgmDrawable *drawable,
                     PgmDrawableLayer layer,
                     gpointer data)
{
  PgmGlesViewport *glesviewport = PGM_GLES_VIEWPORT (data);
  Task *task;

  GST_DEBUG_OBJECT (glesviewport, "drawable_removed_cb");

  /* Add the remove task in the update queue */
  task = task_remove_new (drawable, layer);
  g_mutex_lock (glesviewport->update_lock);
  glesviewport->update_queue = g_slist_prepend (glesviewport->update_queue,
                                                task);
  g_mutex_unlock (glesviewport->update_lock);

  /* Request an update */
  pgm_gles_context_update (glesviewport->context);
}

/* Canvas 'drawable-reordered' handler */
static void
drawable_reordered_cb (PgmCanvas *canvas,
                       PgmDrawable *drawable,
                       PgmDrawableLayer layer,
                       gint order,
                       gpointer data)
{
  PgmGlesViewport *glesviewport = PGM_GLES_VIEWPORT (data);
  Task *task;

  GST_DEBUG_OBJECT (glesviewport, "drawable_reordered_cb");

  /* Add the reorder task in the update queue */
  task = task_reorder_new (drawable, layer, order);
  g_mutex_lock (glesviewport->update_lock);
  glesviewport->update_queue = g_slist_prepend (glesviewport->update_queue,
                                                task);
  g_mutex_unlock (glesviewport->update_lock);

  /* Request an update */
  pgm_gles_context_update (glesviewport->context);
}

/* Canvas 'regenerated' handler */
static void
regenerated_cb (PgmCanvas *canvas,
                gpointer data)
{
  PgmGlesViewport *glesviewport = PGM_GLES_VIEWPORT (data);
  Task *task;

  GST_DEBUG_OBJECT (glesviewport, "drawable_regenerated_cb");

  /* Add the regenerate task in the update queue */
  task = task_any_new (TASK_REGENERATE);
  g_mutex_lock (glesviewport->update_lock);
  glesviewport->update_queue = g_slist_prepend (glesviewport->update_queue,
                                                task);
  g_mutex_unlock (glesviewport->update_lock);

  /* Request an update */
  pgm_gles_context_update (glesviewport->context);
}

/* Delete all GLES drawable from the given GLES viewport layer list and free it */
static void
delete_all_gles_drawable_from_layer (PgmGlesViewport *glesviewport,
                                     GList **layer)
{
  GList *walk = *layer;
  PgmGlesDrawable *glesdrawable;

  while (walk)
    {
      glesdrawable = (PgmGlesDrawable*) walk->data;
      gles_drawable_free (glesviewport, glesdrawable);
      walk = walk->next;
    }

  GST_OBJECT_LOCK (glesviewport);
  g_list_free (*layer);
  *layer = NULL;
  GST_OBJECT_UNLOCK (glesviewport);
}

/* Synchronize the drawables from the given canvas creating and adding the
 * the corresponding GLES drawables in the layers */
static void
sync_gles_drawable_from_canvas (PgmGlesViewport *glesviewport,
                                PgmCanvas *canvas)
{
  PgmGlesDrawable *glesdrawable;
  GList *walk;

  /* Sync near layer */
  walk = canvas->near_layer;
  while (walk)
    {
      glesdrawable = gles_drawable_new (glesviewport, (PgmDrawable*) walk->data);
      g_mutex_lock (glesviewport->layer_lock);
      glesviewport->near_layer = g_list_append (glesviewport->near_layer,
                                                glesdrawable);
      g_mutex_unlock (glesviewport->layer_lock);
      walk = walk->next;
    }

  /* Sync middle layer */
  walk = canvas->middle_layer;
  while (walk)
    {
      glesdrawable = gles_drawable_new (glesviewport, (PgmDrawable*) walk->data);
      g_mutex_lock (glesviewport->layer_lock);
      glesviewport->middle_layer = g_list_append (glesviewport->middle_layer,
                                                  glesdrawable);
      g_mutex_unlock (glesviewport->layer_lock);
      walk = walk->next;
    }

  /* Sync far layer */
  walk = canvas->far_layer;
  while (walk)
    {
      glesdrawable = gles_drawable_new (glesviewport, (PgmDrawable*) walk->data);
      g_mutex_lock (glesviewport->layer_lock);
      glesviewport->far_layer = g_list_append (glesviewport->far_layer,
                                               glesdrawable);
      g_mutex_unlock (glesviewport->layer_lock);
      walk = walk->next;
    }
}

/* PgmViewport methods */

static PgmError
pgm_gles_viewport_set_title (PgmViewport *viewport,
                             const gchar *title)
{
  PgmGlesViewport *glesviewport = PGM_GLES_VIEWPORT (viewport);

  GST_DEBUG_OBJECT (glesviewport, "set_title");

  return PGM_ERROR_X;
}

static PgmError
pgm_gles_viewport_show (PgmViewport *viewport)
{
  PgmGlesViewport *glesviewport = PGM_GLES_VIEWPORT (viewport);
  PgmGlesContextTask *task;

  GST_DEBUG_OBJECT (glesviewport, "show");

  task = pgm_gles_context_task_new (PGM_GLES_CONTEXT_VISIBILITY, NULL);
  pgm_gles_context_push_immediate_task (glesviewport->context, task);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gles_viewport_hide (PgmViewport *viewport)
{
  PgmGlesViewport *glesviewport = PGM_GLES_VIEWPORT (viewport);
  PgmGlesContextTask *task;

  GST_DEBUG_OBJECT (glesviewport, "hide");

  task = pgm_gles_context_task_new (PGM_GLES_CONTEXT_VISIBILITY, NULL);
  pgm_gles_context_push_immediate_task (glesviewport->context, task);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gles_viewport_set_decorated (PgmViewport *viewport,
                                 gboolean decorated)
{
  PgmGlesViewport *glesviewport = PGM_GLES_VIEWPORT (viewport);

  GST_DEBUG_OBJECT (glesviewport, "set_decorated");

  return PGM_ERROR_X;
}

static PgmError
pgm_gles_viewport_set_cursor (PgmViewport *viewport,
                              PgmViewportCursor cursor)
{
  PgmGlesViewport *glesviewport = PGM_GLES_VIEWPORT (viewport);

  GST_DEBUG_OBJECT (glesviewport, "set_cursor");

  return PGM_ERROR_X;
}

static PgmError
pgm_gles_viewport_set_icon (PgmViewport *viewport,
                            GdkPixbuf *icon)
{
  PgmGlesViewport *glesviewport = PGM_GLES_VIEWPORT (viewport);

  GST_DEBUG_OBJECT (glesviewport, "set_icon");

  return PGM_ERROR_X;
}

static PgmError
pgm_gles_viewport_set_drag_status (PgmViewport *viewport,
                                   gboolean accept)
{
  PgmGlesViewport *glesviewport = PGM_GLES_VIEWPORT (viewport);

  GST_DEBUG_OBJECT (glesviewport, "set_drag_status");

  return PGM_ERROR_X;
}

static PgmError
pgm_gles_viewport_set_size (PgmViewport *viewport,
                            gint width,
                            gint height)
{
  PgmGlesViewport *glesviewport = PGM_GLES_VIEWPORT (viewport);

  GST_DEBUG_OBJECT (glesviewport, "set_size");

  return PGM_ERROR_X;
}

static PgmError
pgm_gles_viewport_set_alpha_blending (PgmViewport *viewport,
                                      gboolean alpha_blending)
{
  PgmGlesViewport *glesviewport = PGM_GLES_VIEWPORT (viewport);
  PgmGlesContextTask *task;

  GST_DEBUG_OBJECT (glesviewport, "set_alpha_blending");

  task = pgm_gles_context_task_new (PGM_GLES_CONTEXT_ALPHA_BLENDING, NULL);
  pgm_gles_context_push_immediate_task (glesviewport->context, task);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gles_viewport_set_opacity (PgmViewport *viewport,
                               guchar opacity)
{
  PgmGlesViewport *glesviewport = PGM_GLES_VIEWPORT (viewport);

  GST_DEBUG_OBJECT (glesviewport, "set_opacity");

  return PGM_ERROR_X;
}

static PgmError
pgm_gles_viewport_set_fullscreen (PgmViewport *viewport,
                                  gboolean fullscreen)
{
  PgmGlesViewport *glesviewport = PGM_GLES_VIEWPORT (viewport);

  GST_DEBUG_OBJECT (glesviewport, "set_fullscreen");

  return PGM_ERROR_X;
}

static PgmError
pgm_gles_viewport_get_screen_resolution (PgmViewport *viewport,
                                         gint *width,
                                         gint *height)
{
  PgmGlesViewport *glesviewport = PGM_GLES_VIEWPORT (viewport);

  GST_LOG_OBJECT (glesviewport, "get_screen_resolution");

  pgm_gles_backend_get_screen_resolution (glesviewport->context->backend, width,
                                          height);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gles_viewport_get_screen_size_mm (PgmViewport *viewport,
                                      gint *width,
                                      gint *height)
{
  PgmGlesViewport *glesviewport = PGM_GLES_VIEWPORT (viewport);

  GST_LOG_OBJECT (glesviewport, "get_screen_size_mm");

  pgm_gles_backend_get_screen_size_mm (glesviewport->context->backend, width,
                                       height);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gles_viewport_set_canvas (PgmViewport *viewport,
                              PgmCanvas *canvas)
{
  PgmGlesViewport *glesviewport = PGM_GLES_VIEWPORT (viewport);

  GST_DEBUG_OBJECT (glesviewport, "set_canvas");

  /* There's a canvas bound, let's clear associated data */
  if (glesviewport->canvas)
    {
      GST_OBJECT_LOCK (glesviewport);
      GST_OBJECT_LOCK (glesviewport->canvas);

      /* Disconnect handlers */
      g_signal_handler_disconnect (glesviewport->canvas,
                                   glesviewport->add_handler);
      g_signal_handler_disconnect (glesviewport->canvas,
                                   glesviewport->remove_handler);
      g_signal_handler_disconnect (glesviewport->canvas,
                                   glesviewport->reorder_handler);
      g_signal_handler_disconnect (glesviewport->canvas,
                                   glesviewport->regenerated_handler);

      GST_OBJECT_UNLOCK (glesviewport->canvas);
      GST_OBJECT_UNLOCK (glesviewport);

      /* Delete our hierarchy */
      delete_all_gles_drawable_from_layer (glesviewport,
                                           &glesviewport->near_layer);
      delete_all_gles_drawable_from_layer (glesviewport,
                                           &glesviewport->middle_layer);
      delete_all_gles_drawable_from_layer (glesviewport,
                                           &glesviewport->far_layer);

      GST_OBJECT_LOCK (glesviewport);
      glesviewport->canvas = NULL;
      GST_OBJECT_UNLOCK (glesviewport);
    }

  /* Bind the new canvas if any */
  if (canvas)
    {
      sync_gles_drawable_from_canvas (glesviewport, canvas);

      GST_OBJECT_LOCK (glesviewport);
      GST_OBJECT_LOCK (canvas);

      /* Connect handlers */
      glesviewport->add_handler =
        g_signal_connect (G_OBJECT (canvas), "drawable-added",
                          G_CALLBACK (drawable_added_cb),
                          (gpointer) glesviewport);
      glesviewport->remove_handler =
        g_signal_connect (G_OBJECT (canvas), "drawable-removed",
                          G_CALLBACK (drawable_removed_cb),
                          (gpointer) glesviewport);
      glesviewport->reorder_handler =
        g_signal_connect (G_OBJECT (canvas), "drawable-reordered",
                          G_CALLBACK (drawable_reordered_cb),
                          (gpointer) glesviewport);
      glesviewport->regenerated_handler =
        g_signal_connect (G_OBJECT (canvas), "regenerated",
                          G_CALLBACK (regenerated_cb),
                          (gpointer) glesviewport);

      glesviewport->canvas = canvas;

      GST_OBJECT_UNLOCK (canvas);
      GST_OBJECT_UNLOCK (glesviewport);
    }

  return PGM_ERROR_OK;
}

static PgmError
pgm_gles_viewport_update_projection (PgmViewport *viewport)
{
  PgmGlesViewport *glesviewport = PGM_GLES_VIEWPORT (viewport);
  PgmGlesContextTask *task;

  GST_DEBUG_OBJECT (glesviewport, "update_projection");

  task = pgm_gles_context_task_new (PGM_GLES_CONTEXT_PROJECTION, NULL);
  pgm_gles_context_push_immediate_task (glesviewport->context, task);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gles_viewport_get_embedding_id (PgmViewport *viewport,
                                    gulong *embedding_id)
{
  return PGM_ERROR_X;
}

static PgmError
pgm_gles_viewport_get_pixel_formats (PgmViewport *viewport,
                                     gulong *formats_mask)
{
  PgmGlesViewport *glesviewport = PGM_GLES_VIEWPORT (viewport);

  GST_LOG_OBJECT (glesviewport, "get_pixel_formats");

  GST_OBJECT_LOCK (glesviewport);

  /* The classic formats supported by all hardware */
  *formats_mask = PGM_IMAGE_RGB | PGM_IMAGE_RGBA;

  if (glesviewport->context->feature_mask & PGM_GLES_FEAT_TEXTURE_FORMAT_BGRA)
    *formats_mask |= PGM_IMAGE_BGRA;

  /* BGR, YUYV and UYVY are not supported */

  GST_OBJECT_UNLOCK (glesviewport);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gles_viewport_get_caps_mask (PgmViewport *viewport,
                                 gulong *caps_mask)
{
  PgmGlesViewport *glesviewport = PGM_GLES_VIEWPORT (viewport);

  GST_LOG_OBJECT (glesviewport, "get_caps_mask");

  GST_OBJECT_LOCK (glesviewport);
  *caps_mask = glesviewport->capacities;
  GST_OBJECT_UNLOCK (glesviewport);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gles_viewport_get_frame_rate (PgmViewport *viewport,
                                  guint *frame_rate)
{
  PgmGlesViewport *glesviewport = PGM_GLES_VIEWPORT (viewport);

  GST_LOG_OBJECT (glesviewport, "get_frame_rate");

  GST_OBJECT_LOCK (glesviewport);
  *frame_rate = glesviewport->context->fps;
  GST_OBJECT_UNLOCK (glesviewport);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gles_viewport_read_pixels (PgmViewport *viewport,
                               guint x,
                               guint y,
                               guint width,
                               guint height,
                               guint8 *pixels)
{
  PgmGlesViewport *glesviewport = PGM_GLES_VIEWPORT (viewport);
  PgmGlesViewportPixelRectangle *rectangle = NULL;
  PgmGlesContextTask *task = NULL;

  GST_DEBUG_OBJECT (glesviewport, "read_pixels");

  rectangle = g_slice_new (PgmGlesViewportPixelRectangle);
  if (!rectangle)
    return PGM_ERROR_X;

  rectangle->x = x;
  rectangle->y = y;
  rectangle->width = width;
  rectangle->height = height;
  rectangle->pixels = pixels;

  task = pgm_gles_context_task_new (PGM_GLES_CONTEXT_READ_PIXELS, rectangle);
  pgm_gles_context_push_immediate_task (glesviewport->context, task);

  return PGM_ERROR_OK;
}

/* GObject stuff */

PGM_DEFINE_DYNAMIC_TYPE (PgmGlesViewport, pgm_gles_viewport, PGM_TYPE_VIEWPORT);

void
pgm_gles_viewport_register (GTypeModule *module)
{
  pgm_gles_viewport_register_type (module);
}

static void
pgm_gles_viewport_dispose (GObject *object)
{
  PgmGlesViewport *glesviewport = PGM_GLES_VIEWPORT (object);

  GST_DEBUG_OBJECT (glesviewport, "dispose");

  if (glesviewport->canvas)
    {
      /* Disconnect handlers */
      g_signal_handler_disconnect (glesviewport->canvas,
                                   glesviewport->add_handler);
      g_signal_handler_disconnect (glesviewport->canvas,
                                   glesviewport->remove_handler);
      g_signal_handler_disconnect (glesviewport->canvas,
                                   glesviewport->reorder_handler);
      g_signal_handler_disconnect (glesviewport->canvas,
                                   glesviewport->regenerated_handler);

      /* Delete our hierarchy */
      g_mutex_lock (glesviewport->layer_lock);
      delete_all_gles_drawable_from_layer (glesviewport,
                                           &glesviewport->near_layer);
      delete_all_gles_drawable_from_layer (glesviewport,
                                           &glesviewport->middle_layer);
      delete_all_gles_drawable_from_layer (glesviewport,
                                           &glesviewport->far_layer);
      g_mutex_unlock (glesviewport->layer_lock);
    }

  g_mutex_lock (glesviewport->update_lock);
  g_slist_foreach (glesviewport->update_queue, (GFunc) task_free, NULL);
  g_slist_free (glesviewport->update_queue);
  glesviewport->update_queue = NULL;
  g_mutex_unlock (glesviewport->update_lock);

  pgm_gles_context_free (glesviewport->context);

  g_mutex_free (glesviewport->layer_lock);
  g_mutex_free (glesviewport->update_lock);

  GST_CALL_PARENT (G_OBJECT_CLASS, dispose, (object));
}

static void
pgm_gles_viewport_class_init (PgmGlesViewportClass *klass)
{
  GObjectClass *gobject_class;
  PgmViewportClass *viewport_class;

  GST_DEBUG_CATEGORY_INIT (pgm_gles_viewport_debug, "pgm_gles_viewport", 0,
                           "OpenGL ES plugin: PgmGlesViewport");

  parent_class = g_type_class_peek_parent (klass);

  gobject_class = G_OBJECT_CLASS (klass);
  viewport_class = PGM_VIEWPORT_CLASS (klass);

  /* GObject virtual table */
  gobject_class->dispose = GST_DEBUG_FUNCPTR (pgm_gles_viewport_dispose);

  /* PgmViewport virtual table */
  viewport_class->set_title = GST_DEBUG_FUNCPTR (pgm_gles_viewport_set_title);
  viewport_class->show = GST_DEBUG_FUNCPTR (pgm_gles_viewport_show);
  viewport_class->hide = GST_DEBUG_FUNCPTR (pgm_gles_viewport_hide);
  viewport_class->set_decorated =
    GST_DEBUG_FUNCPTR (pgm_gles_viewport_set_decorated);
  viewport_class->set_cursor = GST_DEBUG_FUNCPTR (pgm_gles_viewport_set_cursor);
  viewport_class->set_icon = GST_DEBUG_FUNCPTR (pgm_gles_viewport_set_icon);
  viewport_class->set_drag_status =
    GST_DEBUG_FUNCPTR (pgm_gles_viewport_set_drag_status);
  viewport_class->set_size = GST_DEBUG_FUNCPTR (pgm_gles_viewport_set_size);
  viewport_class->set_alpha_blending =
    GST_DEBUG_FUNCPTR (pgm_gles_viewport_set_alpha_blending);
  viewport_class->set_opacity =
    GST_DEBUG_FUNCPTR (pgm_gles_viewport_set_opacity);
  viewport_class->set_fullscreen =
    GST_DEBUG_FUNCPTR (pgm_gles_viewport_set_fullscreen);
  viewport_class->get_screen_resolution =
    GST_DEBUG_FUNCPTR (pgm_gles_viewport_get_screen_resolution);
  viewport_class->get_screen_size_mm =
    GST_DEBUG_FUNCPTR (pgm_gles_viewport_get_screen_size_mm);
  viewport_class->set_canvas = GST_DEBUG_FUNCPTR (pgm_gles_viewport_set_canvas);
  viewport_class->update_projection =
    GST_DEBUG_FUNCPTR (pgm_gles_viewport_update_projection);
  viewport_class->get_embedding_id =
    GST_DEBUG_FUNCPTR (pgm_gles_viewport_get_embedding_id);
  viewport_class->get_pixel_formats =
    GST_DEBUG_FUNCPTR (pgm_gles_viewport_get_pixel_formats);
  viewport_class->get_caps_mask =
    GST_DEBUG_FUNCPTR (pgm_gles_viewport_get_caps_mask);
  viewport_class->get_frame_rate =
    GST_DEBUG_FUNCPTR (pgm_gles_viewport_get_frame_rate);
  viewport_class->read_pixels =
    GST_DEBUG_FUNCPTR (pgm_gles_viewport_read_pixels);

  /* PgmGlesDrawable changed table */
  klass->changed_func[PGM_DRAWABLE_VISIBILITY] =
    GST_DEBUG_FUNCPTR (pgm_gles_drawable_set_visibility);
  klass->changed_func[PGM_DRAWABLE_SIZE] =
    GST_DEBUG_FUNCPTR (pgm_gles_drawable_set_size);
  klass->changed_func[PGM_DRAWABLE_POSITION] =
    GST_DEBUG_FUNCPTR (pgm_gles_drawable_set_position);
  klass->changed_func[PGM_DRAWABLE_TRANSFORMATION_MATRIX] =
    GST_DEBUG_FUNCPTR (pgm_gles_drawable_set_transformation_matrix);
  klass->changed_func[PGM_DRAWABLE_BG_COLOR] =
    GST_DEBUG_FUNCPTR (pgm_gles_drawable_set_bg_color);
  klass->changed_func[PGM_DRAWABLE_FG_COLOR] =
    GST_DEBUG_FUNCPTR (pgm_gles_drawable_set_fg_color);
  klass->changed_func[PGM_DRAWABLE_OPACITY] =
    GST_DEBUG_FUNCPTR (pgm_gles_drawable_set_opacity);
  klass->changed_func[PGM_DRAWABLE_REGENERATE] =
    GST_DEBUG_FUNCPTR (pgm_gles_drawable_regenerate);

  /* PgmGlImage changed table */
  klass->changed_func[PGM_IMAGE_DATA_EMPTY] =
    GST_DEBUG_FUNCPTR ((PgmGlesDrawableChangedFunc) pgm_gles_image_clear);
  klass->changed_func[PGM_IMAGE_DATA_BUFFER] =
    GST_DEBUG_FUNCPTR ((PgmGlesDrawableChangedFunc) pgm_gles_image_set_from_buffer);
  klass->changed_func[PGM_IMAGE_DATA_GST_BUFFER] =
    GST_DEBUG_FUNCPTR ((PgmGlesDrawableChangedFunc) pgm_gles_image_set_from_gst_buffer);
  klass->changed_func[PGM_IMAGE_DATA_PIXBUF] =
    GST_DEBUG_FUNCPTR ((PgmGlesDrawableChangedFunc) pgm_gles_image_set_from_pixbuf);
  klass->changed_func[PGM_IMAGE_DATA_IMAGE] =
    GST_DEBUG_FUNCPTR ((PgmGlesDrawableChangedFunc) pgm_gles_image_set_from_image);
  klass->changed_func[PGM_IMAGE_MAPPING_MATRIX] =
    GST_DEBUG_FUNCPTR ((PgmGlesDrawableChangedFunc) pgm_gles_image_set_mapping_matrix);
  klass->changed_func[PGM_IMAGE_ALIGNMENT] =
    GST_DEBUG_FUNCPTR ((PgmGlesDrawableChangedFunc) pgm_gles_image_set_alignment);
  klass->changed_func[PGM_IMAGE_LAYOUT] =
    GST_DEBUG_FUNCPTR ((PgmGlesDrawableChangedFunc) pgm_gles_image_set_layout);
  klass->changed_func[PGM_IMAGE_INTERP] =
    GST_DEBUG_FUNCPTR ((PgmGlesDrawableChangedFunc) pgm_gles_image_set_interp);
  klass->changed_func[PGM_IMAGE_ASPECT_RATIO] =
    GST_DEBUG_FUNCPTR ((PgmGlesDrawableChangedFunc) pgm_gles_image_set_aspect_ratio);
  klass->changed_func[PGM_IMAGE_BORDER_WIDTH] =
    GST_DEBUG_FUNCPTR ((PgmGlesDrawableChangedFunc) pgm_gles_image_set_border_width);
  klass->changed_func[PGM_IMAGE_BORDER_INNER_COLOR] =
    GST_DEBUG_FUNCPTR ((PgmGlesDrawableChangedFunc) pgm_gles_image_set_border_inner_color);
  klass->changed_func[PGM_IMAGE_BORDER_OUTER_COLOR] =
    GST_DEBUG_FUNCPTR ((PgmGlesDrawableChangedFunc) pgm_gles_image_set_border_outer_color);

  /* PgmGlText changed table */
  klass->changed_func[PGM_TEXT_LABEL] =
    GST_DEBUG_FUNCPTR ((PgmGlesDrawableChangedFunc) pgm_gles_text_set_label);
  klass->changed_func[PGM_TEXT_MARKUP] =
    GST_DEBUG_FUNCPTR ((PgmGlesDrawableChangedFunc) pgm_gles_text_set_markup);
  klass->changed_func[PGM_TEXT_FONT_FAMILY] =
    GST_DEBUG_FUNCPTR ((PgmGlesDrawableChangedFunc) pgm_gles_text_set_font_family);
  klass->changed_func[PGM_TEXT_HEIGHT] =
    GST_DEBUG_FUNCPTR ((PgmGlesDrawableChangedFunc) pgm_gles_text_set_font_height);
  klass->changed_func[PGM_TEXT_ELLIPSIZE] =
    GST_DEBUG_FUNCPTR ((PgmGlesDrawableChangedFunc) pgm_gles_text_set_ellipsize);
  klass->changed_func[PGM_TEXT_JUSTIFY] =
    GST_DEBUG_FUNCPTR ((PgmGlesDrawableChangedFunc) pgm_gles_text_set_justify);
  klass->changed_func[PGM_TEXT_ALIGNMENT] =
    GST_DEBUG_FUNCPTR ((PgmGlesDrawableChangedFunc) pgm_gles_text_set_alignment);
  klass->changed_func[PGM_TEXT_WRAP] =
    GST_DEBUG_FUNCPTR ((PgmGlesDrawableChangedFunc) pgm_gles_text_set_wrap);
  klass->changed_func[PGM_TEXT_GRAVITY] =
    GST_DEBUG_FUNCPTR ((PgmGlesDrawableChangedFunc) pgm_gles_text_set_gravity);
  klass->changed_func[PGM_TEXT_STRETCH] =
    GST_DEBUG_FUNCPTR ((PgmGlesDrawableChangedFunc) pgm_gles_text_set_stretch);
  klass->changed_func[PGM_TEXT_STYLE] =
    GST_DEBUG_FUNCPTR ((PgmGlesDrawableChangedFunc) pgm_gles_text_set_style);
  klass->changed_func[PGM_TEXT_VARIANT] =
    GST_DEBUG_FUNCPTR ((PgmGlesDrawableChangedFunc) pgm_gles_text_set_variant);
  klass->changed_func[PGM_TEXT_WEIGHT] =
    GST_DEBUG_FUNCPTR ((PgmGlesDrawableChangedFunc) pgm_gles_text_set_weight);
  klass->changed_func[PGM_TEXT_LINE_SPACING] =
    GST_DEBUG_FUNCPTR ((PgmGlesDrawableChangedFunc) pgm_gles_text_set_line_spacing);
  klass->changed_func[PGM_TEXT_OUTLINE_COLOR] =
    GST_DEBUG_FUNCPTR ((PgmGlesDrawableChangedFunc) pgm_gles_text_set_outline_color);
  klass->changed_func[PGM_TEXT_OUTLINE_WIDTH] =
    GST_DEBUG_FUNCPTR ((PgmGlesDrawableChangedFunc) pgm_gles_text_set_outline_width);
}

static void
pgm_gles_viewport_class_finalize (PgmGlesViewportClass *klass)
{
  return;
}

static void
pgm_gles_viewport_init (PgmGlesViewport *glesviewport)
{
  GST_DEBUG_OBJECT (glesviewport, "init");

  /* Attached canvas */
  glesviewport->canvas = NULL;

  /* Empty lists */
  glesviewport->far_layer = NULL;
  glesviewport->middle_layer = NULL;
  glesviewport->near_layer = NULL;
  glesviewport->layer_lock = g_mutex_new ();

  /* Update task queue */
  glesviewport->update_queue = NULL;
  glesviewport->update_lock = g_mutex_new ();

  /* Create the drawable hash using pointers as keys */
  glesviewport->drawable_hash = g_hash_table_new (NULL, NULL);

  /* Create context */
  glesviewport->context = pgm_gles_context_new (glesviewport);

  /* Fill capacities */
  glesviewport->capacities = PGM_VIEWPORT_HARDWARE_ACCELERATION;
#ifdef TSLIB
  glesviewport->capacities &= PGM_VIEWPORT_TOUCHPAD;
#endif /* TSLIB */

  /* Drag status */
  glesviewport->drag_status = FALSE;
}

/* Public methods */

PgmViewport *
pgm_gles_viewport_new (void)
{
  PgmGlesViewport *glesviewport;

  glesviewport = g_object_new (PGM_TYPE_GLES_VIEWPORT, NULL);
  GST_DEBUG_OBJECT (glesviewport, "created new glesviewport");

  return PGM_VIEWPORT (glesviewport);
}

void
pgm_gles_viewport_flush_update_queue (PgmGlesViewport *glesviewport)
{
  GSList *copy, *walk;
  Task *task;

  /* Get a reverse copy of the update task list, and NULLify it */
  g_mutex_lock (glesviewport->update_lock);
  copy = g_slist_reverse (glesviewport->update_queue);
  glesviewport->update_queue = NULL;
  g_mutex_unlock (glesviewport->update_lock);

  /* Flush the update task list */
  walk = copy;
  while (walk)
    {
      task = walk->data;
      task_func[task->type] (glesviewport, task);
      walk = walk->next;
    }

  g_slist_free (copy);
  copy = NULL;
}

void
pgm_gles_viewport_update_drawable_projection (PgmGlesViewport *glesviewport)
{
  g_mutex_lock (glesviewport->layer_lock);

  /* Adapt position and size of all the drawables */
  g_list_foreach (glesviewport->near_layer,
                  (GFunc) pgm_gles_drawable_update_projection, NULL);
  g_list_foreach (glesviewport->middle_layer,
                  (GFunc) pgm_gles_drawable_update_projection, NULL);
  g_list_foreach (glesviewport->far_layer,
                  (GFunc) pgm_gles_drawable_update_projection, NULL);

  g_mutex_unlock (glesviewport->layer_lock);
}

/* Connects the GLES drawable to the "changed" signal. Should be called at the
 * creation of a new GLES drawable to be notified of any change. */
void
pgm_gles_viewport_connect_changed_callback (PgmGlesViewport *glesviewport,
                                            PgmGlesDrawable *glesdrawable)
{
  PgmDrawable *drawable = glesdrawable->drawable;

  GST_OBJECT_LOCK (drawable);
  glesdrawable->change_handler = g_signal_connect (drawable, "changed",
                                                   G_CALLBACK (changed_cb),
                                                   (gpointer) glesdrawable);
  GST_OBJECT_UNLOCK (drawable);
}
