# -*- mode: python; coding: utf-8 -*-
#
# Pigment Python binding video example
#
# Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the
# Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.
#
# Author: Loïc Molinari <loic@fluendo.com>

import sys, pgm, gobject, gst

fps_label = None

# Key press handler
def on_key_press(viewport, event, pipeline):
    # Terminate the mainloop on 'Escape' or 'q' key press event
    if event.keyval == pgm.keysyms.q or event.keyval == pgm.keysyms.Escape:
        pgm.main_quit()
    # Switch fullscreen state
    if event.keyval == pgm.keysyms.f:
        viewport.fullscreen = not viewport.fullscreen
    # Toggle fps display
    elif event.keyval == pgm.keysyms.p:
        global fps_label
        canvas = viewport.get_canvas()
        if fps_label == None:
            add_fps_display(canvas, viewport)
        else:
            canvas.remove(fps_label)
            fps_label = None

# Drag motion handler
def on_drag_motion(viewport, event, pipeline):
    # Accept the drag if the extension of the first uri is a media file
    if event.uri[0][-3:] in ("avi", "mpg", "ogg", "mkv", "mov", "wmv", "flv"):
        return True

# Drag drop handler
def on_drag_drop(viewport, event, pipeline):
    pipeline.set_state(gst.STATE_READY)
    pipeline.set_property('uri', event.uri[0])
    pipeline.set_state(gst.STATE_PLAYING)

# Terminate the mainloop on a delete event
def on_delete(viewport, event, pipeline):
    pipeline.set_state(gst.STATE_NULL)
    pgm.main_quit()

def add_fps_display(canvas, viewport):
    global fps_label
    fps_label = pgm.Text()
    fps_label.bg_a = 0
    fps_label.width = canvas.width/10.0
    fps_label.height = canvas.height/20.0
    fps_label.x = canvas.width - fps_label.width
    fps_label.y = canvas.height - fps_label.height
    fps_label.visible = True
    canvas.add(pgm.DRAWABLE_NEAR, fps_label)
    refresh_fps_display(viewport)
    gobject.timeout_add(1000, refresh_fps_display, viewport)

def refresh_fps_display(viewport):
    global fps_label
    if fps_label != None:
        fps_label.label = "%d fps" % viewport.frame_rate
        return True
    else:
        return False

def main(args):
    # OpenGL viewport creation
    gl = pgm.viewport_factory_make('opengl')
    gl.title = "Video drag 'n drop"
    gl.alpha_blending = False
    screen_ratio = float(gl.screen_size_mm[1])/gl.screen_size_mm[0]
    gl.size = (800, 800*screen_ratio)

    # Canvas and image drawable creation
    cvs = pgm.Canvas()
    cvs.size = (1.0, 1.0*screen_ratio)
    img = pgm.Image()

    # Bind the canvas to the OpenGL viewport
    gl.set_canvas(cvs)

    # Image properties
    img.position = (0.0, 0.0, 0.0)
    img.size = cvs.size
    img.bg_color = (0, 0, 0, 0)

    # A drawable needs to be shown
    img.show()

    # Add it to the middle layer of the canvas
    cvs.add(pgm.DRAWABLE_MIDDLE, img)

    # GStreamer pipeline setup
    sink = gst.element_factory_make('pgmimagesink')
    pipeline = gst.element_factory_make('playbin')
    pipeline.set_property('video-sink', sink)
    sink.set_property('image', img)
    if (len(sys.argv) > 1):
        pipeline.set_property('uri', sys.argv[1])
        pipeline.set_state(gst.STATE_PLAYING)

    # Let's start the mainloop
    gl.connect('delete-event', on_delete, pipeline)
    gl.connect('key-press-event', on_key_press, pipeline)
    gl.connect('drag-motion-event', on_drag_motion, pipeline)
    gl.connect('drag-drop-event', on_drag_drop, pipeline)
    gl.show()
    pgm.main()

if __name__ == '__main__':
    sys.exit(main(sys.argv[1:]))
