/*
 * Permafrost - Physical modelling framework
 *
 * Copyright (C) 2009, 2010 Stefano D'Angelo <zanga.mail@gmail.com>
 *
 * See the COPYING file for license conditions.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include "config.h"

#include "src/types.h"
#include "src/util.h"
#include "src/list.h"
#include "src/expr.h"
#include "src/parser.tab.h"
#include "src/parser.h"
#include "src/scanner.h"

struct parsed_files
  {
	list_t	consts;
	list_t	ext_funcs;
	list_t	blocks;
	list_t	macros;
	list_t	imported_modules;
  };

list_t parser_consts;
list_t parser_ext_funcs;
list_t parser_blocks;
list_t parser_macros;
list_t parser_systems;

static size_t parsed_files_count = 0;
static struct parsed_files *parsed_files = NULL;
static struct parsed_files *cur_file;

static void
parse(const char *filename, FILE *fp)
{
	yyscan_t scanner;
	int err;

	err = yylex_init(&scanner);
	if (err != 0)
	  {
		fprintf(stderr, "%s\n", strerror(errno));
		exit(EXIT_FAILURE);
	  }

	if (fp == NULL)
	  {
		fp = fopen(filename, "r");
		if (fp == NULL)
		  {
			fprintf(stderr, "%s: %s\n", filename, strerror(errno));
			exit(EXIT_FAILURE);
		  }
	  }

	yyset_in(fp, scanner);
	err = yyparse(scanner, filename);

	fclose(fp);

	yylex_destroy(scanner);

	if (err != 0)
		exit(EXIT_FAILURE);
}

void
parser_import(const char *module)
{
	char *filename;
	FILE *fp;

	if (list_find(cur_file->imported_modules,
		      (int (*)(void *, void *))strcmp, (void *)module) != NULL)
		return;

	list_append(cur_file->imported_modules, (void *)module);

	filename = xmalloc(strlen(module) + 5);
	sprintf(filename, "%s.pmf", module);

	fp = fopen(filename, "r");
	if (fp == NULL)
	  {
		filename = xrealloc(filename,
				    sizeof(PMF_LIB_DIR) + strlen(module) + 5);
		sprintf(filename, "%s%s.pmf", PMF_LIB_DIR, module);
		fp = fopen(filename, "r");
	  }

	if (fp == NULL)
	  {
		fprintf(stderr, "module `%s': %s: %s\n", module, filename,
			strerror(errno));
		exit(EXIT_FAILURE);
	  }

	parse(filename, fp);

	free(filename);
}

void
parser_parse(const char *filename)
{
	if (parser_systems == NULL)
		parser_systems = list_new();

	parsed_files = xrealloc(parsed_files, (parsed_files_count + 1)
					      * sizeof(struct parsed_files));
	cur_file = parsed_files + parsed_files_count;
	parsed_files_count++;

	cur_file->imported_modules = list_new();
	cur_file->consts = list_new();
	cur_file->ext_funcs = list_new();
	cur_file->blocks = list_new();
	cur_file->macros = list_new();

	parser_consts = cur_file->consts;
	parser_ext_funcs = cur_file->ext_funcs;
	parser_blocks = cur_file->blocks;
	parser_macros = cur_file->macros;

	parse(filename, NULL);
}

static void
free_data(void *data, void *context)
{
	free(data);
}

static void
free_port(void *data, void *context)
{
	struct port *p;

	p = (struct port *)data;

	free(p->id);
	free(p);
}

static void
free_stmt(void *data, void *context)
{
	struct stmt *s;

	s = (struct stmt *)data;

	expr_free(s->expr);
	free(s);
}

static void
free_const(void *data, void *context)
{
	struct const_v *v;

	v = (struct const_v *)data;

	free(v->id);
}

static void
free_ext_func(void *data, void *context)
{
	struct ext_func *f;

	f = (struct ext_func *)data;

	free(f->id);
	free(f->f_id);
	free(f->d_id);
	free(f->include);
	free(f->lib);
}

static void
free_block(void *data, void *context)
{
	struct block *b;

	b = (struct block *)data;

	list_for_each(b->ports, free_port, NULL);
	list_for_each(b->stmts, free_stmt, NULL);

	list_free(b->ports);
	list_free(b->stmts);

	free((void *)b->id);
	free(b);
}

static void
free_component(void *data, void *context)
{
	struct component *c;

	c = (struct component *)data;

	free(c->id);
	free(c);
}

static void
free_connection(void *data, void *context)
{
	free(data);
}

void
free_system(void *data, void *context)
{
	struct system *s;

	s = (struct system *)data;

	list_for_each(s->ports, free_port, NULL);
	list_for_each(s->components, free_component, NULL);
	list_for_each(s->connections, free_connection, NULL);

	list_free(s->ports);
	list_free(s->components);
	list_free(s->connections);

	free(s->id);
	free(s);
}

void
parser_free()
{
	size_t i;

	list_for_each(parser_systems, free_system, NULL);
	list_free(parser_systems);

	for (i = 0; i < parsed_files_count; i++)
	  {
		list_for_each(parsed_files[i].macros, free_system, NULL);
		list_free(parsed_files[i].macros);

		list_for_each(parsed_files[i].blocks, free_block, NULL);
		list_free(parsed_files[i].blocks);

		list_for_each(parsed_files[i].consts, free_const, NULL);
		list_free(parsed_files[i].consts);

		list_for_each(parsed_files[i].ext_funcs, free_ext_func, NULL);
		list_free(parsed_files[i].ext_funcs);

		list_for_each(parsed_files[i].imported_modules, free_data,
			      NULL);
		list_free(parsed_files[i].imported_modules);
	  }

	free(parsed_files);

	parsed_files = NULL;
	parsed_files_count = 0;
}
