
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: odk_private.h 2646 2007-08-17 16:50:50Z mschwerin $
 *
 */

#ifndef HAVE_ODK_PRIVATE_H
#define HAVE_ODK_PRIVATE_H

/**
 * @file odk_private.h
 *
 * This file contains private types and methods used inside the ODK system.
 * It should never be included outside the ODK system.
 *
 */

#include "odk.h"
#include "playlist.h"
#include "odk_plugin.h"

/**
 * The maximum number of colors allowed in a palette. This value is defined
 * by the xine engine.
 */
#define NUM_COLORS 256


/**
 * The maximum size of the image cache.
 */
#define NUM_IMAGE_CACHE_ENTRIES 10


/**
 * This represents a OSD palette. As the xine palettes can contain a maximum
 * of 256 colors, so can we.
 */
typedef struct {
    /// Number of colors currently contained in the palette.
    int num_colors;
    /// The color values (YUV format).
    uint32_t colors[NUM_COLORS];
    /// The transparency values.
    uint8_t transparency[NUM_COLORS];
} odk_palette_t;


/// This represents an event handler.
typedef struct {
    /// The event handler callback.
    event_handler_t event_handler;
    /// Data to be passed to the callback.
    void *event_handler_data;
    /// The priority of this event handler.
    odk_event_handler_priority_t priority;
} odk_event_handler_t;


/// This represents a OSD.
typedef struct {

    /**
     * The xine-lib OSD structure.
     * We use a double buffered system so we don't have an empty screen
     * between clearing the old OSD and drawing the new OSD that is seen as a
     * flash by the user.
     * */
    xine_osd_t *x_osd;
    xine_osd_t *x_osd_0;
    xine_osd_t *x_osd_1;
    /// Is the OSD currently visible.
    bool is_visible;
    /// Does the current output driver support unscaled OSD.
    bool is_unscaled_available;
    /// Are we to use unscaled OSD.
    bool use_unscaled;
    /// The palette.
    odk_palette_t palette;
    /// The X offset of this OSD.
    int x_off;
    /// The Y offset of this OSD.
    int y_off;
    /// The width of this OSD.
    int width;
    /// The height of this OSD.
    int height;
    /// The vertical scale factor.
    double vscale;
    /// The horizontal scale factor.
    double hscale;
    /// Is the current font a freetype font.
    bool is_current_font_freetype;
    /// The base resolution width.
    int base_width;
    /// The base resolution height.
    int base_height;
} odk_osd_t;


#ifdef HAVE_OSD_IMAGE

/**
 * The OSD image object.
 *
 * We support two methods to display an image. 
 *
 * 1) The image is stored in a pixel array (<code>data</code>). The colors are
 *    saved in the standard palette. The image is drawn onto the standard OSD
 *    drawing surface.
 *
 * @see odk_osd_draw_image
 *
 * 2) The image is drawn onto a separate OSD drawing surface
 *    (<code>x_osd</code>). The colors are stored in that OSD. 
 *
 * @see odk_osd_show_image
 *
 * No matter what method is used, the image is saved in the image cache.
 * Rescaling is automatically done when adapting the OSD size.
 */
struct odk_osd_image_s {
    /// The MRL of this image.
    char *mrl;
    /// The image pixel data.
    uint8_t *data;
    /// The OSD drawing surface containing the image.
    xine_osd_t *x_osd;
    /// Is the image currently visible?
    bool is_visible;
    /// The x-offset of the image.
    int x;
    /// The y-offset of the image.
    int y;
    /// The width of the image.
    int w;
    /// The height of the image.
    int h;
    /// The alignment of the image.
    int alignment;
    /// Should a border be drawn?
    bool border_draw;
    /// The color of the border.
    uint32_t border_color;
};


/// One entry of the image cache.
typedef struct {
    /// The time this entry was last used.
    time_t timestamp;
    /// The OSD image.
    odk_osd_image_t *image;
} image_cache_entry_t;


/// The image cache.
typedef struct {
    /// The number of entries in the cache.
    int fill;
    /// The cache entries.
    image_cache_entry_t entries[NUM_IMAGE_CACHE_ENTRIES];
} odk_image_cache_t;
#endif


/// The ODK data structure.
struct odk_s {
    /// The one and only xine instance.
    xine_t *xine;

    /// The audio ports.
    struct {
        int count;
        xine_audio_port_t **ports;
    } audio_ports;

    /// The main stream.
    xine_stream_t *main_stream;
    /// The subtitle stream.
    xine_stream_t *subtitle_stream;
    /// The current background stream for audio-only streams.
    xine_stream_t *background_stream;
    /// The current animation stream.
    xine_stream_t *animation_stream;

    /// The current post plugin for audio-only streams.
    xine_post_t *novideo_post;

    /// The onscreen display
    odk_osd_t osd;

#ifdef HAVE_OSD_IMAGE
    /// The image cache.
    odk_image_cache_t image_cache;
#endif

#ifdef HAVE_IMAGE_PLAYBACK
    /// The image slideshow job.
    int image_slideshow_job;
    /// The elapsed time of the slideshow job in secs.
    int image_slideshow_time;
    /// The length of the slideshow job in secs.
    int image_slideshow_length;
#endif

    /// The output window.
    odk_window_t *win;

    /// A list of event handlers.
    l_list_t *event_handlers;

    /// The HTTP input plugin.
    odk_input_t *http;
#ifdef HAVE_LIRC
    /// The LIRC input plugin.
    odk_input_t *lirc;
#endif
#ifdef HAVE_JOYSTICK
    /// The joystick input plugin.
    odk_input_t *joystick;
#endif

    /// The current aspect ratio of playback (NORMAL) mode.
    int aspect_ratio;

    /// Are we currently seeking in the stream.
    bool is_seeking_in_stream;
    /// Are we to forward events to xine-lib.
    bool forward_events_to_xine;

    /// The current mode.
    odk_mode_t current_mode;
    /// The current MRL.
    char *current_mrl;
    /// The current subtitle MRL.
    char *current_subtitle_mrl;
    /// The current title.
    char *current_title;
    /// The current alternative streams.
    playlist_t *current_alternatives;
    /// The current background MRL.
    char *current_background_mrl;
};


/**
 * Initialize post plugin support.
 *
 * @param odk                   The ODK data structure.
 */
void odk_post_init (odk_t * odk);


/**
 * Rewire the audio stream.
 *
 * @param odk                   The ODK data structure.
 */
void odk_post_audio_rewire (odk_t * odk);


/**
 * Unwire the audio stream.
 *
 * @param odk                   The ODK data structure.
 */
void odk_post_audio_unwire (odk_t * odk);


/**
 * Initialize the OSD.
 *
 * @param odk                   The ODK data structure.
 */
void odk_osd_init (odk_t * odk);


/**
 * Free the OSD.
 *
 * @param odk                   The ODK data structure.
 */
void odk_osd_free (odk_t * odk);


/**
 * Get the current frame size.
 *
 * @param odk                   The ODK data structure.
 * @param w                     A pointer to an integer in which to store the
 *                              current video frame width.
 * @param h                     A pointer to an integer in which to store the
 *                              current video frame height.
 */
void odk_get_frame_size (odk_t * odk, int *w, int *h);


/**
 * Adapt the OSD to the current frame size and output size.
 *
 * @param odk                   The ODK data structure.
 * @param fcd                   A data structure containing information about
 *                              the new video format.
 */
void odk_osd_adapt_size (odk_t * odk, xine_format_change_data_t * fcd);


/**
 * Initialize event handling.
 *
 * @param odk                   The ODK data structure.
 */
void odk_event_init (odk_t * odk);


/**
 * Frees the event handling stuff.
 *
 * @param odk                   The ODK data structure.
 */
void odk_event_free (odk_t * odk);


#endif /* HAVE_ODK_PRIVATE_H */
