
/*
 * Copyright (C) 2002-2003 Stefan Holst
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: odk_joystick.c 2402 2007-07-02 15:22:05Z mschwerin $
 *
 */
#include "config.h"

#include <errno.h>
#include <fcntl.h>
#include <regex.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#ifdef HAVE_LINUX_JOYSTICK_H
#include <linux/joystick.h>
#endif

#ifdef HAVE_LIBJSW
#include <jsw.h>
#endif

#include "environment.h"
#include "heap.h"
#include "i18n.h"
#include "logger.h"

#include "odk.h"
#include "odk_joystick.h"

#ifdef HAVE_JOYSTICK
typedef struct {
    odk_input_t class;
    odk_t *odk;
    void (*event_handler) (void *data, oxine_event_t * ev);
} js_input_t;

#if defined(__linux__)
static const char *devices[] =
    { "/dev/js0", "/dev/input/js0", "/dev/js1", "/dev/input/js1", NULL };
#elif defined(__FreeBSD__)
static const char *devices[] = { "/dev/joy0", "/dev/joy1", NULL };
#else
static const char *devices[] =
    { "/dev/js0", "/dev/input/js0", "/dev/js1", "/dev/input/js1", NULL };
#endif


#ifdef HAVE_LIBJSW
static int
open_jsw (void)
{
    /* 
     * Initialize (open) the joystick, passing it the pointer to
     * the jsd structure (it is safe to pass an uninitialized jsd
     * structure as seen here).
     */
    int di = 0;
    int status = 1;
    js_data_struct jsd;
    const char *calib = get_file_jsw_config ();
    while (devices[di] && status != JSSuccess) {
        status = JSInit (&jsd, devices[di], calib, JSFlagNonBlocking);
        di++;
    }
    if (status != JSSuccess) {
        JSClose (&jsd);
        error (_("Failed to open joystick using libjsw!"));
        return 0;
    }

    return 1;
}


static int
poll_jsw (js_input_t * js)
{
    /* 
     * Initialize (open) the joystick, passing it the pointer to
     * the jsd structure (it is safe to pass an uninitialized jsd
     * structure as seen here).
     */
    int di = 0;
    int status = 1;
    js_data_struct jsd;
    const char *calib = get_file_jsw_config ();
    while (devices[di] && status != JSSuccess) {
        status = JSInit (&jsd, devices[di], calib, JSFlagNonBlocking);
        di++;
    }
    if (status != JSSuccess) {
        JSClose (&jsd);
        error (_("Failed to open joystick using libjsw!"));
        return 0;
    }

    debug ("using libjsw to access joystick at %s", jsd.device_name);
    debug ("using joystick calibration file %s", jsd.calibration_file);

    if (jsd.last_calibrated == 0) {
        warn (_("Your joystick has never been calibrated!"));
        info (_("Please run 'jscalibrator -file %s -device "
                "%s' to calibrate your joystick."),
              jsd.calibration_file, jsd.device_name);
    }

    while (1) {
        pthread_testcancel ();

        oxine_event_t ev;
        ev.type = OXINE_EVENT_KEY;
        ev.source.key = 0;

        /* get new event */
        if (JSUpdate (&jsd) == JSGotEvent) {
            if (JSIsButtonAllocated (&jsd, 0)
                && jsd.button[0]->changed_state ==
                JSButtonChangedStateOnToOff) {
                ev.source.key = OXINE_KEY_ACTIVATE;
            }

            if (!ev.source.key && JSIsButtonAllocated (&jsd, 1)
                && jsd.button[1]->changed_state ==
                JSButtonChangedStateOnToOff) {
                ev.source.key = OXINE_KEY_SELECT;
            }

            if (!ev.source.key && JSIsButtonAllocated (&jsd, 2)
                && jsd.button[2]->changed_state ==
                JSButtonChangedStateOnToOff) {
                ev.source.key = OXINE_KEY_SHOW_OSD;
            }

            if (!ev.source.key && JSIsAxisAllocated (&jsd, 0)) {
                int value = jsd.axis[0]->cur - jsd.axis[0]->cen;
                int max = jsd.axis[0]->max - jsd.axis[0]->cen;
                int min = jsd.axis[0]->min - jsd.axis[0]->cen;
                if (value < (int) (min * 0.3))
                    ev.source.key = OXINE_KEY_LEFT;
                if (value > (int) (max * 0.3))
                    ev.source.key = OXINE_KEY_RIGHT;
            }

            if (!ev.source.key && JSIsAxisAllocated (&jsd, 1)) {
                int value = jsd.axis[1]->cur - jsd.axis[1]->cen;
                int max = jsd.axis[1]->max - jsd.axis[1]->cen;
                int min = jsd.axis[1]->min - jsd.axis[1]->cen;
                if (value < (int) (min * 0.3))
                    ev.source.key = OXINE_KEY_UP;
                if (value > (int) (max * 0.3))
                    ev.source.key = OXINE_KEY_DOWN;
            }

            if (ev.source.key)
                js->event_handler (js->odk, &ev);
        }

        usleep (300000);                                        /* Don't hog the cpu. */
    }
    JSClose (&jsd);
    return 1;
}
#endif /* HAVE_LIBJSW */


#ifdef HAVE_LINUX_JOYSTICK_H
static int
open_raw (void)
{
    int di = 0;
    int jsfd = -1;

    while (devices[di] && jsfd == -1) {
        jsfd = open (devices[di], O_RDONLY);
        di++;
    }

    if (jsfd == -1) {
        error (_("Failed to open joystick using raw interface!"));
        return 0;
    }

    close (jsfd);
    debug ("Opened joystick device %s", devices[di - 1]);

    return 1;
}


static int
poll_raw (js_input_t * js)
{
    int di = 0;
    int jsfd = -1;

    while (devices[di] && jsfd == -1) {
        jsfd = open (devices[di], O_RDONLY);
        di++;
    }
    if (jsfd == -1) {
        error (_("Failed to open joystick using raw interface!"));
        return 0;
    }
    else {
        debug ("Opened joystick device %s", devices[di - 1]);
    }
    fcntl (jsfd, F_SETFL, O_NONBLOCK);
    while (1) {
        pthread_testcancel ();

        struct js_event jsev;
        int size = sizeof (struct js_event);
        while (read (jsfd, &jsev, size) == size) {
            oxine_event_t ev;
            ev.type = OXINE_EVENT_KEY;
            ev.source.key = 0;
            switch (jsev.type) {
            case JS_EVENT_BUTTON:
                /* value -> press (0), release (1) */
                /* number -> button number */
                if (jsev.value == 1) {
                    if (jsev.number == 0) {
                        ev.source.key = OXINE_KEY_ACTIVATE;
                    }
                    else if (jsev.number == 1) {
                        ev.source.key = OXINE_KEY_SELECT;
                    }
                    else if (jsev.number == 2) {
                        ev.source.key = OXINE_KEY_SHOW_OSD;
                    }
                }
                break;
            case JS_EVENT_AXIS:
                /* number -> axis */
                if (jsev.number == 0) {
                    /* left  */
                    if (jsev.value < -16384) {
                        ev.source.key = OXINE_KEY_LEFT;
                    }
                    /* right */
                    else if (jsev.value > 16384) {
                        ev.source.key = OXINE_KEY_RIGHT;
                    }
                }
                else if (jsev.number == 1) {
                    /* up */
                    if (jsev.value < -16384) {
                        ev.source.key = OXINE_KEY_UP;
                    }
                    /* down */
                    else if (jsev.value > 16384) {
                        ev.source.key = OXINE_KEY_DOWN;
                    }
                }

                break;
            }

            pthread_testcancel ();

            if (ev.source.key)
                js->event_handler (js->odk, &ev);
        }
    }
    return 1;
}
#endif /* HAVE_LINUX_JOYSTICK_H */


static void *
js_thread (void *this)
{
    js_input_t *js = (js_input_t *) this;

    info (_("Successfully started the joystick input plugin."));
    debug ("   joystick thread: 0x%X", (int) pthread_self ());

#if defined (HAVE_LINUX_JOYSTICK_H) && defined (HAVE_LIBJSW)
    if (!poll_jsw (js) && !poll_raw (js)) {
#elif defined (HAVE_LINUX_JOYSTICK_H)
    if (!poll_raw (js)) {
#elif defined (HAVE_LIBJSW)
    if (!poll_jsw (js)) {
#else
    {
#endif
        info (_("Make sure you have the permissions "
                "to open the joystick devices."));
        info (_("Devices oxine tried to open are:"), devices[0]);
        int di = 0;
        while (devices[di]) {
            if (devices[di + 1]) {
                info ("%s, %s,", devices[di], devices[di + 1]);
                di += 2;
            }
            else {
                info ("%s", devices[di]);
                di += 1;
            }
        }
    }

    pthread_exit (NULL);
    return NULL;
}


odk_input_t *
start_joystick (odk_t * odk, event_handler_t event_handler)
{
    js_input_t *js = ho_new (js_input_t);
    js->odk = odk;
    js->event_handler = event_handler;

#if defined (HAVE_LINUX_JOYSTICK_H) && defined (HAVE_LIBJSW)
    if (!open_jsw () && !open_raw ()) {
#elif defined (HAVE_LINUX_JOYSTICK_H)
    if (!open_raw ()) {
#elif defined (HAVE_LIBJSW)
    if (!open_jsw ()) {
#else
    {
#endif
        info (_("Make sure you have the permissions "
                "to open the joystick devices."));
        info (_("Devices oxine tried to open are:"), devices[0]);
        int di = 0;
        while (devices[di]) {
            if (devices[di + 1]) {
                info ("%s, %s,", devices[di], devices[di + 1]);
                di += 2;
            }
            else {
                info ("%s", devices[di]);
                di += 1;
            }
        }

        ho_free (js);
        return NULL;
    }

    if (pthread_create (&js->class.thread, NULL, js_thread, js) != 0) {
        error (_("Could not create joystick thread: %s!"), strerror (errno));

        ho_free (js);
        return NULL;
    }

    return (odk_input_t *) js;
}


void
stop_joystick (odk_input_t * oi)
{
    js_input_t *js = (js_input_t *) oi;
    pthread_cancel (oi->thread);
    ho_free (js);

    info (_("Successfully stopped the joystick input plugin."));
}
#endif /* HAVE_JOYSTICK */
