///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/data/ObjectLoadStream.h>
#include <core/data/ObjectSaveStream.h>
#include <core/rendering/FrameBuffer.h>
#include <core/rendering/RenderSettings.h>
#include <core/utilities/ProgressIndicator.h>
#include <core/scene/animation/controller/StandardControllers.h>
#include <core/reference/CloneHelper.h>

#include "POVRayRenderer.h"
#include "POVRayRendererEditor.h"
#include "../export/POVRayExporter.h"

namespace POVRay {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(POVRayRenderer, PluginRenderer)
DEFINE_REFERENCE_FIELD(POVRayRenderer, IntegerController, "QualityLevel", _qualityLevel)
DEFINE_REFERENCE_FIELD(POVRayRenderer, BooleanController, "EnableAntialiasing", _enableAntialiasing)
DEFINE_REFERENCE_FIELD(POVRayRenderer, IntegerController, "SamplingMethod", _samplingMethod)
DEFINE_REFERENCE_FIELD(POVRayRenderer, FloatController, "AAThreshold", _AAThreshold)
DEFINE_REFERENCE_FIELD(POVRayRenderer, IntegerController, "AntialiasDepth", _antialiasDepth)
DEFINE_REFERENCE_FIELD(POVRayRenderer, BooleanController, "EnableJitter", _enableJitter)
DEFINE_REFERENCE_FIELD(POVRayRenderer, BooleanController, "HidePOVRayDisplay", _hidePOVRayDisplay)
DEFINE_REFERENCE_FIELD(POVRayRenderer, BooleanController, "EnableRadiosity", _enableRadiosity)
DEFINE_REFERENCE_FIELD(POVRayRenderer, IntegerController, "RadiosityRayCount", _radiosityRayCount)
DEFINE_REFERENCE_FIELD(POVRayRenderer, IntegerController, "RadiosityRecursionLimit", _radiosityRecursionLimit)
DEFINE_REFERENCE_FIELD(POVRayRenderer, FloatController, "RadiosityErrorBound", _radiosityErrorBound)
SET_PROPERTY_FIELD_LABEL(POVRayRenderer, _qualityLevel, "Quality level")
SET_PROPERTY_FIELD_LABEL(POVRayRenderer, _enableAntialiasing, "Enable anti-aliasing")
SET_PROPERTY_FIELD_LABEL(POVRayRenderer, _samplingMethod, "Sampling method")
SET_PROPERTY_FIELD_LABEL(POVRayRenderer, _AAThreshold, "Anti-aliasing threshold")
SET_PROPERTY_FIELD_LABEL(POVRayRenderer, _antialiasDepth, "Anti-aliasing depth")
SET_PROPERTY_FIELD_LABEL(POVRayRenderer, _enableJitter, "Enable jitter")
SET_PROPERTY_FIELD_LABEL(POVRayRenderer, _hidePOVRayDisplay, "Hide POV-Ray window during render")
SET_PROPERTY_FIELD_LABEL(POVRayRenderer, _enableRadiosity, "Enable radiosity")
SET_PROPERTY_FIELD_LABEL(POVRayRenderer, _radiosityRayCount, "Ray count")
SET_PROPERTY_FIELD_LABEL(POVRayRenderer, _radiosityRecursionLimit, "Recusion limit")
SET_PROPERTY_FIELD_LABEL(POVRayRenderer, _radiosityErrorBound, "Error bound")

/******************************************************************************
* Default constructor.
******************************************************************************/
POVRayRenderer::POVRayRenderer(bool isLoading)
	: PluginRenderer(isLoading)
{
	INIT_PROPERTY_FIELD(POVRayRenderer, _qualityLevel)
	INIT_PROPERTY_FIELD(POVRayRenderer, _enableAntialiasing)
	INIT_PROPERTY_FIELD(POVRayRenderer, _samplingMethod)
	INIT_PROPERTY_FIELD(POVRayRenderer, _AAThreshold)
	INIT_PROPERTY_FIELD(POVRayRenderer, _antialiasDepth)
	INIT_PROPERTY_FIELD(POVRayRenderer, _enableJitter)
	INIT_PROPERTY_FIELD(POVRayRenderer, _hidePOVRayDisplay)
	INIT_PROPERTY_FIELD(POVRayRenderer, _enableRadiosity)
	INIT_PROPERTY_FIELD(POVRayRenderer, _radiosityRayCount)
	INIT_PROPERTY_FIELD(POVRayRenderer, _radiosityRecursionLimit)
	INIT_PROPERTY_FIELD(POVRayRenderer, _radiosityErrorBound)
	if(!isLoading) {
		_qualityLevel = new ConstIntegerController();
		_enableAntialiasing = new ConstBooleanController();
		_samplingMethod = new ConstIntegerController();
		_AAThreshold = new ConstFloatController();
		_antialiasDepth = new ConstIntegerController();
		_enableJitter = new ConstBooleanController();
		_hidePOVRayDisplay = new ConstBooleanController();
		_enableRadiosity = new ConstBooleanController();
		_radiosityRayCount = new ConstIntegerController();
		_radiosityRecursionLimit = new ConstIntegerController();
		_radiosityErrorBound = new ConstFloatController();

		_qualityLevel->setCurrentValue(9);
		_enableAntialiasing->setCurrentValue(false);
		_samplingMethod->setCurrentValue(1);
		_AAThreshold->setCurrentValue(0.3f);
		_antialiasDepth->setCurrentValue(3);
		_enableJitter->setCurrentValue(true);
		_hidePOVRayDisplay->setCurrentValue(false);
		_enableRadiosity->setCurrentValue(false);
		_radiosityRayCount->setCurrentValue(35);
		_radiosityRecursionLimit->setCurrentValue(3);
		_radiosityErrorBound->setCurrentValue(1.8f);
	}
}

/******************************************************************************
* Prepares the renderer for rendering of the given scene.
******************************************************************************/
bool POVRayRenderer::startRender(DataSet* dataset)
{
	this->dataset = dataset;
	return true;
}

/******************************************************************************
* Renders a single animation frame into the given frame buffer.
******************************************************************************/
bool POVRayRenderer::renderFrame(TimeTicks time, CameraViewDescription view, FrameBuffer* frameBuffer)
{
	ProgressIndicator progress(tr("Writing scene to a temporary POV-Ray file."));
	if(progress.isCanceled()) return false;

	// Write scene to a temporary file.
	QTemporaryFile sceneFile(QDir::tempPath() + "/scene.XXXXXX.pov");
	if(!sceneFile.open())
		throw Exception(tr("Failed to open temporary POV-Ray scene file for writing."));
	VerboseLogger() << logdate << "Writing POV-Ray scene to the temporary file:" << sceneFile.fileName() << endl;
	sceneFile.setTextModeEnabled(true);

	// Let POV-Ray write the image to a temporary file.
	QTemporaryFile imageFile(QDir::tempPath() + "/povray.XXXXXX.png");
	if(!imageFile.open())
		throw Exception(tr("Failed to open temporary POV-Ray image file."));
	VerboseLogger() << logdate << "The rendered image will be written to the temporary file:" << imageFile.fileName() << endl;

	POVRayExporter exporter;
	exporter.ExportToPOVRay(&sceneFile, dataset, time, &view, this);

	// Specify POV-Ray options:
	QStringList parameters;
	parameters << QString("+W%1").arg(renderSettings()->imageWidth());		// Set image width
	parameters << QString("+H%1").arg(renderSettings()->imageHeight()); 	// Set image height
	parameters << "Pause_When_Done=off";	// No pause after rendering
	parameters << "Output_to_File=on";		// Output to file
	parameters << "-V";		// Verbose output
	parameters << "Output_File_Type=N";		// Output PNG image
	parameters << QString("Output_File_Name=%1").arg(QDir::toNativeSeparators(imageFile.fileName()));		// Output image to temporary file.
	parameters << QString("Input_File_Name=%1").arg(QDir::toNativeSeparators(sceneFile.fileName()));		// Read scene from temporary file.

	if(renderSettings()->generateAlphaChannel())
		parameters << "Output_Alpha=on";		// Output alpha channel
	else
		parameters << "Output_Alpha=off";		// No alpha channel

	if(_hidePOVRayDisplay->getValueAtTime(time))
		parameters << "Display=off";
	else
		parameters << "Display=on";

	// Pass quality settings to POV-Ray.
	if(_qualityLevel)
		parameters << QString("+Q%1").arg(_qualityLevel->getValueAtTime(time));
	if(_enableAntialiasing) {
		if(_enableAntialiasing->getValueAtTime(time) == true) {
			if(_AAThreshold)
				parameters << QString("+A%1").arg(_AAThreshold->getValueAtTime(time));
			else
				parameters << "+A";
		}
		else {
			parameters << "-A";
		}
	}
	if(_samplingMethod)
		parameters << QString("+AM%1").arg(_samplingMethod->getValueAtTime(time));
	if(_antialiasDepth)
		parameters << QString("+R%1").arg(_antialiasDepth->getValueAtTime(time));
	if(_enableJitter)
		parameters << (_enableJitter->getValueAtTime(time) ? "+J" : "-J");

	sceneFile.close();
	imageFile.close();

	// Start POV-Ray sub-process.
	progress.setLabelText(tr("Starting external POV-Ray program."));
	if(progress.isCanceled()) return false;

	QProcess povrayProcess;
	VerboseLogger() << logdate << "Starting POV-Ray sub-process (path =" << renderExecutable() << ")" << endl;
	VerboseLogger() << logdate << "POV-Ray parameters:" << parameters << endl;
	povrayProcess.setReadChannel(QProcess::StandardOutput);
	povrayProcess.start(renderExecutable(), parameters);
	if(!povrayProcess.waitForStarted()) {
		QString errorString = povrayProcess.errorString();
		if(povrayProcess.error() == QProcess::FailedToStart)
			errorString = tr("The process failed to start. Either the invoked program is missing, or you may have insufficient permissions to invoke the program.");
		throw Exception(tr("Could not run the POV-Ray executable: %1 (error code %2)\nPlease check your POV-Ray installation.\nExecutable path: %3").arg(errorString).arg(povrayProcess.error()).arg(renderExecutable()));
	}

	// Wait until POV-Ray has finished rendering.
	progress.setLabelText(tr("Waiting for external POV-Ray program..."));
	if(progress.isCanceled()) return false;

	povrayProcess.waitForFinished();
	OVITO_ASSERT(povrayProcess.exitStatus() == QProcess::NormalExit);
	VerboseLogger() << logdate << "POV-Ray console output:" << endl;
	VerboseLogger() << logdate << povrayProcess.readAllStandardError() << endl;
	VerboseLogger() << logdate << "POV-Ray program returned with exit code" << povrayProcess.exitCode() << endl;
	if(povrayProcess.exitCode() != 0)
		throw Exception(tr("POV-Ray program returned with error code %1.").arg(povrayProcess.exitCode()));

	// Get rendered image from POV-Ray process.
	progress.setLabelText(tr("Getting rendered image from POV-Ray."));
	if(progress.isCanceled()) return false;

	QImage povrayImage;
	if(!imageFile.open() || !povrayImage.load(&imageFile, "PNG")) {
		VerboseLogger() << logdate << "Temporary image filename was" << QDir::toNativeSeparators(imageFile.fileName()) << endl;
		throw Exception(tr("Failed to parse image data obtained from external POV-Ray program."));
	}

	// Fill frame buffer with background color.
	QPainter painter(&frameBuffer->image());

	if(!renderSettings()->generateAlphaChannel()) {
		Color backgroundColor(renderSettings()->backgroundColorController()->getValueAtTime(time));
		painter.fillRect(frameBuffer->image().rect(), QColor::fromRgbF(backgroundColor.r,backgroundColor.g,backgroundColor.b));
	}

	// Copy POV-Ray image to the internal frame buffer.
	painter.drawImage(0, 0, povrayImage);

	return true;
}

/******************************************************************************
* Finishes the rendering pass. This is called after all animation frames have been rendered
* or when the rendering operation has been aborted.
******************************************************************************/
void POVRayRenderer::endRender()
{
}

/******************************************************************************
* Returns the path to the POV-Ray executable.
******************************************************************************/
QString POVRayRenderer::renderExecutable()
{
	// Read path from the sttings file.
	QSettings settings;
	settings.beginGroup("povray");
	QString executablePath = settings.value("executable").toString();
	if(executablePath.isEmpty()) {
		// Use default value.
#ifdef Q_WS_WIN
		return "povray.exe";
#else
		return "povray";
#endif
	}
	else return executablePath;
}

/******************************************************************************
* Sets the path to the POV-Ray executable.
******************************************************************************/
void POVRayRenderer::setRenderExecutable(const QString& path)
{
	// Store path in the sttings file.
	QSettings settings;
	settings.beginGroup("povray");
	settings.setValue("executable", path);
}

};

