///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/utilities/PathManager.h>
#include <core/viewport/Viewport.h>
#include <core/viewport/ViewportManager.h>
#include <core/viewport/input/ViewportInputManager.h>
#include <core/viewport/input/NavigationModes.h>
#include <core/viewport/SceneRenderer.h>
#include <core/scene/animation/AnimManager.h>

namespace Core {

/******************************************************************************
* This is called by the system after the input handler is
* no longer the active handler.
******************************************************************************/
void NavigationMode::onDeactivated()
{
	if(viewport) {
		viewport->releaseMouse();
		// Retore old view transformation.
		viewport->setViewMatrix(oldViewMatrix);
		// Retore old projection.
		viewport->setFieldOfView(oldZoom);
		// Refresh viewport.
		viewport->updateViewport(true);
		viewport = NULL;
	}
	ViewportInputHandler::onDeactivated();
}

/******************************************************************************
* Handles the mouse down event for the given viewport.
******************************************************************************/
void NavigationMode::onMouseDown(Viewport& vp, QMouseEvent* event)
{
	if(event->button() == Qt::RightButton) {
		ViewportInputHandler::onMouseDown(vp, event);
	}
	else {
		viewport = &vp;
		vp.grabMouse();
		// Store starting point.
		startPoint = event->pos();
		// Store old matrices.
		oldViewMatrix = viewport->viewMatrix();
		oldZoom = viewport->fieldOfView();
	}
}

/******************************************************************************
* Handles the mouse up event for the given viewport.
******************************************************************************/
void NavigationMode::onMouseUp(Viewport& vp, QMouseEvent* event)
{
	if(viewport != NULL) {
		viewport->releaseMouse();
		viewport = NULL;
	}
}

/******************************************************************************
* Handles the mouse move event for the given viewport.
******************************************************************************/
void NavigationMode::onMouseMove(Viewport& vp, QMouseEvent* event)
{
	if(viewport) {
		modifyViewMatrix(*viewport, event->pos());
		modifyZoom(*viewport, event->pos());
		viewport->updateViewport(true);
		VIEWPORT_MANAGER.processViewportUpdates();
	}
}

///////////////////////////////////////////////////////////////////////////////
////////////////////////////////// Pan Mode ///////////////////////////////////

/******************************************************************************
* Computes the new view matrix based on the new mouse position.
******************************************************************************/
void PanMode::modifyViewMatrix(Viewport& vp, const QPoint& currentPos)
{
	if(vp.isPerspectiveProjection()) {
		FloatType scaling = 50.0 / (FloatType)vp.viewportRectangle().height();
		FloatType deltaX =  scaling * (FloatType)(currentPos.x() - startPoint.x());
		FloatType deltaY = -scaling * (FloatType)(currentPos.y() - startPoint.y());
		vp.settings()->setViewMatrix(AffineTransformation::translation(Vector3(deltaX, deltaY, 0)) * oldViewMatrix);
	}
	else {
		FloatType deltaX =  2.0 * (FloatType)(currentPos.x() - startPoint.x()) / (FloatType)vp.viewportRectangle().width();
		FloatType deltaY = -2.0 * (FloatType)(currentPos.y() - startPoint.y()) / (FloatType)vp.viewportRectangle().height();
		vp.settings()->setViewMatrix(AffineTransformation::translation(vp.inverseProjectionMatrix() * Vector3(deltaX, deltaY, 0)) * oldViewMatrix);
	}
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////// Zoom Mode ///////////////////////////////////

/******************************************************************************
* Computes the new view matrix based on the new mouse position.
******************************************************************************/
void ZoomMode::modifyViewMatrix(Viewport& vp, const QPoint& currentPos)
{
	if(vp.isPerspectiveProjection()) {
		FloatType amount = (FloatType)(startPoint.y() - currentPos.y()) * 0.5;
		vp.settings()->setViewMatrix(AffineTransformation::translation(Vector3(0, 0, amount)) * oldViewMatrix);
	}
}

/******************************************************************************
* Computes the new view zoom based on the new mouse position.
******************************************************************************/
void ZoomMode::modifyZoom(Viewport& vp, const QPoint& currentPos)
{
	if(!vp.isPerspectiveProjection()) {
		FloatType scaling = (FloatType)exp((FloatType)(currentPos.y() - startPoint.y()) * 0.006);
		vp.settings()->setFieldOfView(oldZoom * scaling);
	}
}

/******************************************************************************
* Zooms the viewport in or out.
******************************************************************************/
void ZoomMode::Zoom(Viewport& vp, FloatType steps)
{
	startPoint = QPoint(0, (int)(steps * 0.2));
	oldViewMatrix = vp.viewMatrix();
	oldZoom = vp.fieldOfView();

	modifyZoom(vp, QPoint(0, 0));
	modifyViewMatrix(vp, QPoint(0, 0));
	vp.updateViewport(true);
}

///////////////////////////////////////////////////////////////////////////////
//////////////////////////////// Orbit Mode ///////////////////////////////////

/******************************************************************************
* Handles the mouse down event for the given viewport.
******************************************************************************/
void OrbitMode::onMouseDown(Viewport& vp, QMouseEvent* event)
{
	NavigationMode::onMouseDown(vp, event);
	if(viewport) {
		// Compute orbiting center.
		if(!selectionCenter) {
			// Use click point on construction plane.
			orbitCenter = ORIGIN;
			viewport->grid().viewportComputePlaneIntersection(ORIGIN, orbitCenter);
			orbitCenter = viewport->grid().gridMatrix() * orbitCenter;
		}
		else {
			// Use center of selected objects.
			orbitCenter = ORIGIN;
			Box3 bb;
			// Compute bounding box of selection.
		    SceneRenderer* renderer = SceneRenderer::activeRenderer();
		    CHECK_POINTER(renderer);
			bb = renderer->sceneExtents(viewport->settings().get(), ANIM_MANAGER.time(), SceneRenderer::SELECTED_OBJECTS);
			if(bb.isEmpty())
	            bb = renderer->sceneExtents(viewport->settings().get(), ANIM_MANAGER.time(), SceneRenderer::ALL);
			if(!bb.isEmpty())
				orbitCenter = bb.center();
		}
	}
}

/******************************************************************************
* Computes the new view matrix based on the new mouse position.
******************************************************************************/
void OrbitMode::modifyViewMatrix(Viewport& vp, const QPoint& currentPos)
{
	if(!vp.isPerspectiveProjection())
		vp.settings()->setViewType(Viewport::VIEW_ORTHO);

	FloatType scaling = 4.0 / (FloatType)vp.viewportRectangle().height();
	FloatType deltaX = scaling * (FloatType)(currentPos.x() - startPoint.x());
	FloatType deltaY = scaling * (FloatType)(currentPos.y() - startPoint.y());
	Vector3 t = (oldViewMatrix * orbitCenter) - ORIGIN;
	vp.settings()->setViewMatrix(
		AffineTransformation::translation(t) *
		AffineTransformation::rotationX(deltaY) *
		AffineTransformation::translation(-t) *
		oldViewMatrix *
		AffineTransformation::translation(orbitCenter - ORIGIN) *
		AffineTransformation::rotationZ(deltaX) *
		AffineTransformation::translation(-(orbitCenter - ORIGIN)));
}

};
