///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/scene/SceneRoot.h>
#include <core/scene/ObjectNode.h>
#include <core/data/DataSetManager.h>
#include <core/data/DataSet.h>
#include <core/scene/animation/AnimManager.h>
#include <core/scene/objects/AbstractCameraObject.h>
#include "ViewportMenu.h"

namespace Core {

/******************************************************************************
* Initializes the menu.
******************************************************************************/
ViewportMenu::ViewportMenu(Viewport* vp) : viewport(vp)
{
	QAction* action;

	// Build menu.
	QActionGroup* shadingModeGroup = new QActionGroup(this);
	action = shadingModeGroup->addAction(tr("Wireframe"));
	connect(action, SIGNAL(triggered(bool)), this, SLOT(OnWireframeMode()));
	action->setCheckable(true);
	action->setChecked(viewport->settings()->shadingMode() == Viewport::SHADING_WIREFRAME);
	action = shadingModeGroup->addAction(tr("Shaded"));
	connect(action, SIGNAL(triggered(bool)), this, SLOT(OnShadedMode()));
	action->setCheckable(true);
	action->setChecked(viewport->settings()->shadingMode() == Viewport::SHADING_SHADED);
	action = shadingModeGroup->addAction(tr("Shaded+Edges"));
	connect(action, SIGNAL(triggered(bool)), this, SLOT(OnShadedWithEdgesMode()));
	action->setCheckable(true);
	action->setChecked(viewport->settings()->shadingMode() == Viewport::SHADING_SHADED_WITH_EDGES);
	addActions(shadingModeGroup->actions());
	addSeparator();

	action = addAction(tr("Show Grid"), this, SLOT(OnShowGrid(bool)));
	action->setCheckable(true);
	action->setChecked(viewport->grid().isVisible());
	addSeparator();

	action = addAction(tr("Show Render Frame"), this, SLOT(OnShowRenderFrame(bool)));
	action->setCheckable(true);
	action->setChecked(viewport->settings()->renderFrameShown());
	addSeparator();

	viewTypeMenu = addMenu(tr("View"));
	connect(viewTypeMenu, SIGNAL(aboutToShow()), this, SLOT(OnShowViewTypeMenu()));

	QActionGroup* viewTypeGroup = new QActionGroup(this);
	action = viewTypeGroup->addAction(tr("Top"));
	action->setCheckable(true);
	action->setChecked(viewport->settings()->viewType() == Viewport::VIEW_TOP);
	action->setData((int)Viewport::VIEW_TOP);
	action = viewTypeGroup->addAction(tr("Bottom"));
	action->setCheckable(true);
	action->setChecked(viewport->settings()->viewType() == Viewport::VIEW_BOTTOM);
	action->setData((int)Viewport::VIEW_BOTTOM);
	action = viewTypeGroup->addAction(tr("Front"));
	action->setCheckable(true);
	action->setChecked(viewport->settings()->viewType() == Viewport::VIEW_FRONT);
	action->setData((int)Viewport::VIEW_FRONT);
	action = viewTypeGroup->addAction(tr("Back"));
	action->setCheckable(true);
	action->setChecked(viewport->settings()->viewType() == Viewport::VIEW_BACK);
	action->setData((int)Viewport::VIEW_BACK);
	action = viewTypeGroup->addAction(tr("Left"));
	action->setCheckable(true);
	action->setChecked(viewport->settings()->viewType() == Viewport::VIEW_LEFT);
	action->setData((int)Viewport::VIEW_LEFT);
	action = viewTypeGroup->addAction(tr("Right"));
	action->setCheckable(true);
	action->setChecked(viewport->settings()->viewType() == Viewport::VIEW_RIGHT);
	action->setData((int)Viewport::VIEW_RIGHT);
	action = viewTypeGroup->addAction(tr("Ortho"));
	action->setCheckable(true);
	action->setChecked(viewport->settings()->viewType() == Viewport::VIEW_ORTHO);
	action->setData((int)Viewport::VIEW_ORTHO);
	action = viewTypeGroup->addAction(tr("Perspective"));
	action->setCheckable(true);
	action->setChecked(viewport->settings()->viewType() == Viewport::VIEW_PERSPECTIVE);
	action->setData((int)Viewport::VIEW_PERSPECTIVE);
	viewTypeMenu->addActions(viewTypeGroup->actions());
	viewTypeMenu->addSeparator();

	connect(viewTypeGroup, SIGNAL(triggered(QAction*)), this, SLOT(OnViewType(QAction*)));

	viewNodeGroup = new QActionGroup(this);

	connect(viewNodeGroup, SIGNAL(triggered(QAction*)), this, SLOT(OnViewNode(QAction*)));
}

/******************************************************************************
* Is called just before the "View Type" sub-menu is shown.
******************************************************************************/
void ViewportMenu::OnShowViewTypeMenu()
{
	Q_FOREACH(QAction* action, viewNodeGroup->actions()) {
		viewNodeGroup->removeAction(action);
		delete action;
	}

	// Scan the scene for suitable view nodes.
	for(SceneNodesIterator nodeIterator(DATASET_MANAGER.currentSet()->sceneRoot()); !nodeIterator.finished(); nodeIterator.next()) {

		ObjectNode* objNode = dynamic_object_cast<ObjectNode>(nodeIterator.current());

		TimeTicks time = ANIM_MANAGER.time();
		PipelineFlowState state = objNode->evalPipeline(time);
		AbstractCameraObject* camera = dynamic_object_cast<AbstractCameraObject>(state.result());
		if(camera) {

			// Add a menu entry for this camera node.
			QAction* action = viewNodeGroup->addAction(objNode->name());
			action->setCheckable(true);
			action->setChecked(viewport->settings()->viewNode() == objNode);
			action->setData(qVariantFromValue((void*)objNode));

		}
	}

	// Add menu entries to menu.
	viewTypeMenu->addActions(viewNodeGroup->actions());
}

/******************************************************************************
* Handles the menu item event.
******************************************************************************/
void ViewportMenu::OnShowGrid(bool checked)
{
	viewport->grid().setVisible(checked);
}

/******************************************************************************
* Handles the menu item event.
******************************************************************************/
void ViewportMenu::OnShowRenderFrame(bool checked)
{
	viewport->settings()->setRenderFrameShown(checked);
}

/******************************************************************************
* Handles the menu item event.
******************************************************************************/
void ViewportMenu::OnWireframeMode()
{
	viewport->settings()->setShadingMode(Viewport::SHADING_WIREFRAME);
}

/******************************************************************************
* Handles the menu item event.
******************************************************************************/
void ViewportMenu::OnShadedMode()
{
	viewport->settings()->setShadingMode(Viewport::SHADING_SHADED);
}

/******************************************************************************
* Handles the menu item event.
******************************************************************************/
void ViewportMenu::OnShadedWithEdgesMode()
{
	viewport->settings()->setShadingMode(Viewport::SHADING_SHADED_WITH_EDGES);
}

/******************************************************************************
* Handles the menu item event.
******************************************************************************/
void ViewportMenu::OnViewType(QAction* action)
{
	viewport->settings()->setViewType((Viewport::ViewportType)action->data().toInt());
}

/******************************************************************************
* Handles the menu item event.
******************************************************************************/
void ViewportMenu::OnViewNode(QAction* action)
{
	ObjectNode* viewNode = static_cast<ObjectNode*>(action->data().value<void*>());
	CHECK_OBJECT_POINTER(viewNode);

	viewport->settings()->setViewType(Viewport::VIEW_SCENENODE);
	viewport->settings()->setViewNode(viewNode);
}

};
