///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/gui/properties/NumericalPropertyUI.h>
#include <core/undo/UndoManager.h>
#include <core/viewport/ViewportManager.h>
#include <core/data/units/ParameterUnit.h>

namespace Core {

// Gives the class run-time type information.
IMPLEMENT_ABSTRACT_PLUGIN_CLASS(NumericalPropertyUI, PropertyParameterUI)

/******************************************************************************
* Constructor for a Qt property.
******************************************************************************/
NumericalPropertyUI::NumericalPropertyUI(PropertiesEditor* parentEditor, const char* propertyName, ParameterUnit* defaultParameterUnit, const QString& labelText) : 
	PropertyParameterUI(parentEditor, propertyName), _parameterUnit(defaultParameterUnit)
{
	initUIControls(labelText);	

	CHECK_OBJECT_POINTER(_parameterUnit);
	_spinner->setUnit(_parameterUnit);
}

/******************************************************************************
* Constructor for a PropertyField property.
******************************************************************************/
NumericalPropertyUI::NumericalPropertyUI(PropertiesEditor* parentEditor, const PropertyFieldDescriptor& propField, ParameterUnit* defaultParameterUnit) : 
	PropertyParameterUI(parentEditor, propField), _parameterUnit(defaultParameterUnit)
{
	CHECK_OBJECT_POINTER(_parameterUnit);
	
	// Look up the ParameterUnit for this parameter.
	try {
		ParameterUnit* customUnit = propField.parameterUnit();
		if(customUnit != NULL) _parameterUnit = customUnit;
	}
	catch(const Exception& ex) {		
		ex.showError();
	}
	
	initUIControls(propField.displayName() + ":");	
}

/******************************************************************************
* Creates the widgets for this property UI.
******************************************************************************/
void NumericalPropertyUI::initUIControls(const QString& labelText)
{
	// Create UI widgets.
	_label = new QLabel(labelText);
	_textBox = new QLineEdit();
	_spinner = new SpinnerWidget();
	connect(_spinner, SIGNAL(spinnerValueChanged()), this, SLOT(onSpinnerValueChanged()));	
	connect(_spinner, SIGNAL(spinnerDragStart()), this, SLOT(onSpinnerDragStart()));
	connect(_spinner, SIGNAL(spinnerDragStop()), this, SLOT(onSpinnerDragStop()));
	connect(_spinner, SIGNAL(spinnerDragAbort()), this, SLOT(onSpinnerDragAbort()));
	_spinner->setTextBox(_textBox);

	CHECK_OBJECT_POINTER(_parameterUnit);
	_spinner->setUnit(_parameterUnit);
}

/******************************************************************************
* Destructor, that releases all GUI controls.
******************************************************************************/
NumericalPropertyUI::~NumericalPropertyUI()
{
	// Release GUI controls. 
	delete label();
	delete spinner();
	delete textBox(); 
}

/******************************************************************************
* This method is called when a new editable object has been assigned to the properties owner this
* parameter UI belongs to. 
******************************************************************************/
void NumericalPropertyUI::resetUI()
{
	PropertyParameterUI::resetUI();	
	
	if(spinner()) 
		spinner()->setEnabled(editObject() != NULL && isEnabled());
}

/******************************************************************************
* Sets the enabled state of the UI.
******************************************************************************/
void NumericalPropertyUI::setEnabled(bool enabled)
{
	if(enabled == isEnabled()) return;
	PropertyParameterUI::setEnabled(enabled);
	if(spinner()) spinner()->setEnabled(editObject() != NULL && isEnabled());
}

/******************************************************************************
* Is called when the spinner value has changed.
******************************************************************************/
void NumericalPropertyUI::onSpinnerValueChanged()
{
	ViewportSuspender noVPUpdate;
	if(!UNDO_MANAGER.isRecording()) {
		UNDO_MANAGER.beginCompoundOperation(tr("Change Parameter"));		
		updatePropertyValue();
		UNDO_MANAGER.endCompoundOperation();
	}
	else {
		UNDO_MANAGER.currentCompoundOperation()->clear();
		updatePropertyValue();
	}
}

/******************************************************************************
* Is called when the user begins dragging the spinner interactively.
******************************************************************************/
void NumericalPropertyUI::onSpinnerDragStart()
{
	OVITO_ASSERT(!UNDO_MANAGER.isRecording());
	UNDO_MANAGER.beginCompoundOperation(tr("Change Parameter"));
}

/******************************************************************************
* Is called when the user stops dragging the spinner interactively.
******************************************************************************/
void NumericalPropertyUI::onSpinnerDragStop()
{
	OVITO_ASSERT(UNDO_MANAGER.isRecording());
	UNDO_MANAGER.endCompoundOperation();
}

/******************************************************************************
* Is called when the user aborts dragging the spinner interactively.
******************************************************************************/
void NumericalPropertyUI::onSpinnerDragAbort()
{
	OVITO_ASSERT(UNDO_MANAGER.isRecording());
	UNDO_MANAGER.currentCompoundOperation()->clear();
	UNDO_MANAGER.endCompoundOperation();
}

};

