///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/actions/ActionManager.h>
#include <core/gui/mainwnd/MainFrame.h>
#include <core/data/DataSetManager.h>
#include <core/data/importexport/ImportExportManager.h>
#include <core/plugins/branding/Branding.h>
#include <core/utilities/PathManager.h>
#include <core/undo/UndoManager.h>
#include "StdActions.h"

namespace Core {

/// Constructs the action objects and registers them with the ActionManager.
FileActionsHandler::FileActionsHandler()
{
	connect(addCommandAction(ACTION_EXIT, tr("Exit"), ":/core/main/file_quit.png", Qt::CTRL + Qt::Key_Q), SIGNAL(triggered(bool)), this, SLOT(OnExit()));
	connect(addCommandAction(ACTION_FILE_OPEN, tr("Open"), ":/core/main/file_open.png", QKeySequence::Open), SIGNAL(triggered(bool)), this, SLOT(OnFileOpen()));
	connect(addCommandAction(ACTION_FILE_SAVE, tr("Save"), ":/core/main/file_save.png", QKeySequence::Save), SIGNAL(triggered(bool)), this, SLOT(OnFileSave()));
	connect(addCommandAction(ACTION_FILE_SAVEAS, tr("Save As"), ":/core/main/file_save_as.png"), SIGNAL(triggered(bool)), this, SLOT(OnFileSaveAs()));
	connect(addCommandAction(ACTION_FILE_RESET, tr("Reset"), ":/core/main/file_reset.png"), SIGNAL(triggered(bool)), this, SLOT(OnFileReset()));
	connect(addCommandAction(ACTION_FILE_NEW, tr("New"), ":/core/main/file_new.png", QKeySequence::New), SIGNAL(triggered(bool)), this, SLOT(OnFileNew()));
	connect(addCommandAction(ACTION_FILE_IMPORT, tr("Import"), ":/core/main/file_import.png", Qt::CTRL + Qt::Key_I), SIGNAL(triggered(bool)), this, SLOT(OnFileImport()));
	connect(addCommandAction(ACTION_FILE_EXPORT, tr("Export"), ":/core/main/file_export.png", Qt::CTRL + Qt::Key_E), SIGNAL(triggered(bool)), this, SLOT(OnFileExport()));
	connect(addCommandAction(ACTION_HELP_ABOUT, tr("About")), SIGNAL(triggered(bool)), this, SLOT(OnHelpAbout()));
	connect(addCommandAction(ACTION_HELP_SHOW_ONLINE_HELP, tr("Manual")), SIGNAL(triggered(bool)), this, SLOT(OnHelpShowOnlineHelp()));
}

/// Creates and registers a single command action.
ActionProxy* FileActionsHandler::addCommandAction(const QString& id, const QString& title, const char* iconPath, const QKeySequence& shortcut)
{
	ActionProxy* proxy = ACTION_MANAGER.addAction(new Action(id), title, iconPath);
	if(!shortcut.isEmpty()) proxy->setShortcut(shortcut);
	return proxy;
}

/// Handles ACTION_EXIT command.
void FileActionsHandler::OnExit()
{
	if(APPLICATION_MANAGER.guiMode())
		MAIN_FRAME->close();
}

/// Handles ACTION_FILE_NEW command.
void FileActionsHandler::OnFileNew()
{
	if(DATASET_MANAGER.askForSaveChanges()) {
		DataSet::SmartPtr newSet(new DataSet());
		DATASET_MANAGER.setCurrentSet(newSet);
	}
}

/// Handles ACTION_FILE_RESET command.
void FileActionsHandler::OnFileReset()
{
	if(DATASET_MANAGER.askForSaveChanges())
		DATASET_MANAGER.fileReset();
}

/// Handles ACTION_FILE_OPEN command.
void FileActionsHandler::OnFileOpen()
{
	if(!DATASET_MANAGER.askForSaveChanges())
		return;

	QSettings settings;
	settings.beginGroup("file/scene");

	// Go the last directory used.
	QString defaultPath;
	DataSet* dataSet = DATASET_MANAGER.currentSet();
	if(dataSet == NULL || dataSet->filePath().isEmpty()) {
		defaultPath = settings.value("last_directory").toString();
		if(defaultPath.isEmpty())
			defaultPath = PATH_MANAGER.scenesDirectory();
	}
	else
		defaultPath = dataSet->filePath();

	QString filename = QFileDialog::getOpenFileName(0, tr("Load Scene"), defaultPath, BrandingManager::brandings().back()->sceneFileFilter());
    if(filename.isEmpty())
		return;

	// Remember directory for the next time...
	settings.setValue("last_directory", QFileInfo(filename).absolutePath());

	DATASET_MANAGER.fileLoad(filename);
}

/// Handles ACTION_FILE_SAVE command.
void FileActionsHandler::OnFileSave()
{
	DATASET_MANAGER.fileSave();
}

/// Handles ACTION_FILE_SAVEAS command.
void FileActionsHandler::OnFileSaveAs()
{
	DATASET_MANAGER.fileSaveAs();
}

/// Handles ACTION_HELP_ABOUT command.
void FileActionsHandler::OnHelpAbout()
{
	if(!APPLICATION_MANAGER.guiMode())
		return;

	OVITO_ASSERT(!BrandingManager::brandings().empty());
	Branding::SmartPtr branding = BrandingManager::brandings().back();

	QString text = QString("%1\n%2\n\n%3\n\n%4\n\n%5").arg(branding->productName(), branding->productVersion(),
		branding->productDescription(), branding->copyrightString(), branding->productWebSite());
	QMessageBox::about(MAIN_FRAME, branding->productName(), text);
}

/// Handles ACTION_HELP_SHOW_ONLINE_HELP command.
void FileActionsHandler::OnHelpShowOnlineHelp()
{
	if(!APPLICATION_MANAGER.guiMode())
		return;

	/*
	try {
		// Start the extern assistant program to show the help file.
		MAIN_FRAME->showQtAssistant();
	}
	catch(const Exception& ex) {
		ex.showError();
		*/

		try {
			// Use the web brwoser to show the online help instead.
			if(!QDesktopServices::openUrl(QUrl("http://www.ovito.org/manual/")))
				throw Exception(tr("Could not lauch the web browser to display the online manual. The URL is http://ovito.org/manual/"));
		}
		catch(const Exception& ex2) {
			ex2.showError();
		}
	/*
	}
	*/
}

/// Handles ACTION_FILE_IMPORT command.
void FileActionsHandler::OnFileImport()
{
	// Build filter string.
	QStringList filterStrings;
	Q_FOREACH(const ImportExportDescriptor& descr, IMPORTEXPORT_MANAGER.importers()) {
		filterStrings << QString("%1 (%2)").arg(descr.fileFilterDescription(), descr.fileFilter());
	}
	if(filterStrings.isEmpty()) {
		Exception(tr("This function is disabled. There are no import plugins installed.")).showError();
		return;
	}
	filterStrings.prepend(tr("<Auto-detect file format> (*)"));

	QSettings settings;
	settings.beginGroup("file/import");

	// Let the user select a file.
	QFileDialog dialog(0, tr("Import"));
	dialog.setFilters(filterStrings);
	dialog.setAcceptMode(QFileDialog::AcceptOpen);
	dialog.setFileMode(QFileDialog::ExistingFile);

	// Go the last directory used.
	QString lastImportDirectory = settings.value("last_import_dir").toString();
	if(!lastImportDirectory.isEmpty())
		dialog.setDirectory(lastImportDirectory);
	// Select the last import filter being used ...
	QString lastImportFilter = settings.value("last_import_filter").toString();
	if(!lastImportFilter.isEmpty())
		dialog.selectFilter(lastImportFilter);

	if(!dialog.exec())
		return;

	QStringList filesToImport = dialog.selectedFiles();
	if(filesToImport.isEmpty())
		return;
	QString importFile = filesToImport.front();

	// Remember directory for the next time...
	settings.setValue("last_import_dir", dialog.directory().absolutePath());
	// Remember import filter for the next time...
	settings.setValue("last_import_filter", dialog.selectedFilter());

	// Import selected file.
	try {
		int importFilterIndex = filterStrings.indexOf(dialog.selectedFilter()) - 1;
		OVITO_ASSERT(importFilterIndex >= -1 && importFilterIndex < filterStrings.size());

		ImporterExporter::SmartPtr importer;
		if(importFilterIndex >= 0)
			importer = IMPORTEXPORT_MANAGER.importers()[importFilterIndex].createService();
		else {
			importer = IMPORTEXPORT_MANAGER.autodetectFileFormat(importFile);
			if(!importer)
				throw Exception(tr("Could not auto-detect file format. Please specify the format explicitly."));
		}

		UNDO_MANAGER.beginCompoundOperation(tr("Import %1").arg(QFileInfo(importFile).baseName()));
		try {
			importer->importFile(importFile, DATASET_MANAGER.currentSet());
			UNDO_MANAGER.endCompoundOperation();
		}
		catch(...) {
			UNDO_MANAGER.currentCompoundOperation()->clear();
			UNDO_MANAGER.endCompoundOperation();
			throw;
		}
	}
	catch(const Exception& ex) {
		ex.showError();
	}
}


/// Handles ACTION_FILE_EXPORT command.
void FileActionsHandler::OnFileExport()
{
	// Build filter string.
	QStringList filterStrings;
	Q_FOREACH(const ImportExportDescriptor& descr, IMPORTEXPORT_MANAGER.exporters()) {
		filterStrings << QString("%1 (%2)").arg(descr.fileFilterDescription(), descr.fileFilter());
	}
	if(filterStrings.isEmpty()) {
		Exception(tr("This function is disabled. There are no export plugins installed.")).showError();
		return;
	}

	QSettings settings;
	settings.beginGroup("file/export");

	// Let the user select a destination file.
	QFileDialog dialog(0, tr("Export"));
	dialog.setFilters(filterStrings);
	dialog.setAcceptMode(QFileDialog::AcceptSave);
	dialog.setFileMode(QFileDialog::AnyFile);
	dialog.setConfirmOverwrite(true);

	// Go the last directory used.
	QString lastExportDirectory = settings.value("last_export_dir").toString();
	if(!lastExportDirectory.isEmpty())
		dialog.setDirectory(lastExportDirectory);
	// Select the last export filter being used ...
	QString lastExportFilter = settings.value("last_export_filter").toString();
	if(!lastExportFilter.isEmpty())
		dialog.selectFilter(lastExportFilter);

	if(!dialog.exec())
		return;

	QStringList files = dialog.selectedFiles();
	if(files.isEmpty())
		return;
	QString exportFile = files.front();

	// Remember directory for the next time...
	settings.setValue("last_export_dir", dialog.directory().absolutePath());
	// Remember export filter for the next time...
	settings.setValue("last_export_filter", dialog.selectedFilter());

	// Export to selected file.
	try {
		int exportFilterIndex = filterStrings.indexOf(dialog.selectedFilter());
		OVITO_ASSERT(exportFilterIndex >= 0 && exportFilterIndex < filterStrings.size());
		ImporterExporter::SmartPtr exporter = IMPORTEXPORT_MANAGER.exporters()[exportFilterIndex].createService();

		exporter->exportToFile(exportFile, DATASET_MANAGER.currentSet());
	}
	catch(const Exception& ex) {
		ex.showError();
	}
}

};
