///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/gui/RolloutContainer.h>
#include <core/scene/animation/AnimManager.h>
#include <core/viewport/Viewport.h>
#include <core/viewport/ViewportManager.h>
#include "AtomInformationUtility.h"
#include <atomviz/atoms/datachannels/AtomTypeDataChannel.h>

namespace AtomViz {

IMPLEMENT_PLUGIN_CLASS(AtomInformationUtility, UtilityPlugin)

/******************************************************************************
* Initializes the utility applet.
******************************************************************************/
AtomInformationUtility::AtomInformationUtility() : UtilityPlugin(), panel(NULL)
{
}

/******************************************************************************
* Shows the UI of the utility in the given RolloutContainer.
******************************************************************************/
void AtomInformationUtility::openUtility(RolloutContainer* container, const RolloutInsertionParameters& rolloutParams)
{
	// Create a rollout.
	panel = new QWidget();
	container->addRollout(panel, tr("Atom Information"), rolloutParams);

    // Create the rollout contents.
	QVBoxLayout* layout = new QVBoxLayout(panel);
	layout->setContentsMargins(4,4,4,4);
	layout->setSpacing(4);

	QLabel* resultLabel = new QLabel(tr("Pick an atom in the viewports."), panel);
	layout->addWidget(resultLabel);
	resultLabel->setTextInteractionFlags(Qt::TextSelectableByMouse|Qt::TextSelectableByKeyboard);

	inputMode = new AtomInformationInputMode(resultLabel);
	VIEWPORT_INPUT_MANAGER.pushInputHandler(inputMode);
}

/******************************************************************************
* Removes the UI of the utility from the rollout container.
******************************************************************************/
void AtomInformationUtility::closeUtility(RolloutContainer* container)
{
	VIEWPORT_INPUT_MANAGER.removeInputHandler(inputMode.get());
	inputMode = NULL;
	delete panel;
}

/******************************************************************************
* Handles the mouse down events for a Viewport.
******************************************************************************/
void AtomInformationInputMode::onMouseDown(Viewport& vp, QMouseEvent* event)
{
	ViewportInputHandler::onMouseDown(vp, event);

	if(event->button() == Qt::LeftButton) {
		if(pickAtom(vp, event->pos(), ANIM_MANAGER.time(), pickedAtom)) {
			QString text(tr("<p>Atom properties (index %1):<ul>").arg(pickedAtom.index));
			Q_FOREACH(DataChannel* channel, pickedAtom.atomsObject->dataChannels()) {
				if((int)channel->size() <= pickedAtom.index) continue;
				if(channel->type() != qMetaTypeId<int>() && channel->type() != qMetaTypeId<FloatType>()) continue;
				for(size_t component = 0; component < channel->componentCount(); component++) {
					text.append("<li>");
					text.append(channel->name());
					if(channel->componentNames().empty() == false) {
						text.append(".");
						text.append(channel->componentNames()[component]);
					}
					text.append(" = ");
					if(channel->type() == qMetaTypeId<int>()) {
						text.append(QString::number(channel->getIntComponent(pickedAtom.index, component)));
						AtomTypeDataChannel* atomTypeChannel = dynamic_object_cast<AtomTypeDataChannel>(channel);
						if(atomTypeChannel && atomTypeChannel->atomTypes().empty() == false) {
							AtomType* atype = atomTypeChannel->atomTypes()[channel->getIntComponent(pickedAtom.index, component) % atomTypeChannel->atomTypes().size()];
							if(atype) {
								text.append(" (" + atype->name() + ")");
							}
						}
					}
					else if(channel->type() == qMetaTypeId<FloatType>())
						text.append(QString::number(channel->getFloatComponent(pickedAtom.index, component)));
					text.append("</li>");
				}
			}
			text.append("</ul></p>");
			resultLabel->setText(text);
		}
		else {
			resultLabel->setText(tr("You didn't click on an atom."));
			pickedAtom.index = -1;
			pickedAtom.atomsObject = NULL;
		}
		VIEWPORT_MANAGER.updateViewports();
	}
}

/******************************************************************************
* Lets the input mode render its overlay content in a viewport.
******************************************************************************/
void AtomInformationInputMode::renderOverlay(Viewport* vp, bool isActive)
{
	ViewportInputHandler::renderOverlay(vp, isActive);
	renderSelectionMarker(vp, pickedAtom);
}

};
