///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/scene/animation/AnimManager.h>
#include <core/utilities/ProgressIndicator.h>

#include "POSCARParser.h"
#include <atomviz/atoms/AtomsObject.h>
#include <atomviz/atoms/datachannels/AtomTypeDataChannel.h>
#include <atomviz/atoms/datachannels/PositionDataChannel.h>

#include "../CompressedTextParserStream.h"

namespace AtomViz {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(POSCARParser, AtomsFileParser)

/******************************************************************************
* Reads an atomic data set from the input file.
******************************************************************************/
EvaluationStatus POSCARParser::loadAtomsFile(AtomsObject* destination, int movieFrame, bool suppressDialogs)
{
	CHECK_OBJECT_POINTER(destination);

	// Open the input file for reading.
	CompressedTextParserStream stream(inputFile());
	setlocale(LC_NUMERIC, "C");

	// Read comment line
	stream.readline();
	VerboseLogger() << "POSCAR file comment string:" << stream.line().c_str() << endl;

	// Read header
#ifdef USE_DOUBLE_PRECISION_FP
	#define FLOAT_SCANF_STRING_1   "%lg"
	#define FLOAT_SCANF_STRING_3   "%lg %lg %lg"
#else
	#define FLOAT_SCANF_STRING_1   "%g"
	#define FLOAT_SCANF_STRING_3   "%g %g %g"
#endif

	// Read global scaling factor
	FloatType scaling_factor = 0;
	stream.readline();
	if(sscanf(stream.line().c_str(), FLOAT_SCANF_STRING_1, &scaling_factor) != 1 || scaling_factor <= 0)
		throw Exception(tr("Invalid scaling factor (line 1): %1").arg(stream.line().c_str()));
	VerboseLogger() << "Global scaling factor:" << scaling_factor << endl;

	// Read cell matrix
	AffineTransformation cell(IDENTITY);
	for(size_t i=0; i<3; i++) {
		stream.readline();
		if(sscanf(stream.line().c_str(), FLOAT_SCANF_STRING_3, &cell(0,i), &cell(1,i), &cell(2,i)) != 3 || cell.column(i) == NULL_VECTOR)
			throw Exception(tr("Invalid cell vector (line %1): %2").arg(stream.lineNumber()).arg(stream.line().c_str()));
		VerboseLogger() << "Cell vector" << (i+1) << ":" << cell.column(i) << endl;
	}
	cell = cell * scaling_factor;
	destination->simulationCell()->setCellMatrix(cell);

	// This is used for string -> number conversion.
	QLocale locale = QLocale::c();

	// Parse number of atoms per type.
	QStringList tokens = QString(stream.readline().c_str()).split(QRegExp("\\s+"), QString::SkipEmptyParts);
	QVector<int> atomCounts;
	int totalAtomCount = 0;
	Q_FOREACH(const QString& token, tokens) {
		bool ok;
		int n = locale.toInt(token, &ok);
		if(!ok)
			throw Exception(tr("Invalid atom count (line %1): %2").arg(stream.lineNumber()).arg(stream.line().c_str()));
		atomCounts.push_back(n);
		totalAtomCount += n;
	}
	VerboseLogger() << "Found" << atomCounts.size() << "atom types in POSCAR file." << endl;

	destination->setAtomsCount((size_t)totalAtomCount);

	// Get the required data channels.
	AtomTypeDataChannel* typeChannel = static_object_cast<AtomTypeDataChannel>(destination->createStandardDataChannel(DataChannel::AtomTypeChannel));
	PositionDataChannel* posChannel = static_object_cast<PositionDataChannel>(destination->createStandardDataChannel(DataChannel::PositionChannel));

	OVITO_ASSERT(posChannel->size() == totalAtomCount);

	// Read in 'Selective dynamics' flag
	stream.readline();
	if(stream.line().at(0) == 'S' || stream.line().at(0) == 's') {
		stream.readline();
		VerboseLogger() << "Found 'Selective dynamics' flag" << endl;
	}

	// Parse coordinate system.
	bool isCartesian = false;
	if(stream.line().at(0) == 'C' || stream.line().at(0) == 'c' || stream.line().at(0) == 'K' || stream.line().at(0) == 'k') {
		isCartesian = true;
		VerboseLogger() << "Found 'Cartesian coordinates' flag" << endl;
	}

	// Read in atom coordinates.
	Point3* p = posChannel->dataPoint3();
	int* a = typeChannel->dataInt();
	for(int atype=1; atype<=atomCounts.size(); atype++) {
		typeChannel->createAtomType(atype);
		VerboseLogger() << "Loading" << atomCounts[atype-1] << "atoms of type" << atype << endl;
		for(int i = 0; i < atomCounts[atype-1]; i++, ++p, ++a) {
			stream.readline();
			*a = atype;
			if(sscanf(stream.line().c_str(), FLOAT_SCANF_STRING_3, &p->X, &p->Y, &p->Z) != 3)
				throw Exception(tr("Invalid atom coordinates (line %1): %2").arg(stream.lineNumber()).arg(stream.line().c_str()));
			if(!isCartesian)
				*p = cell * (*p);
			else
				*p = (*p) * scaling_factor;
		}
	}

	// Parse coordinate system for velocity vectors.
	stream.readline();
	if(!stream.eof()) {
		isCartesian = false;
		if(stream.line().at(0) == 'C' || stream.line().at(0) == 'c' || stream.line().at(0) == 'K' || stream.line().at(0) == 'k')
			isCartesian = true;

		// Read in atom velocities.
		DataChannel* velocityChannel = destination->createStandardDataChannel(DataChannel::VelocityChannel);
		Vector3* v = velocityChannel->dataVector3();
		for(int atype=1; atype<=atomCounts.size(); atype++) {
			for(int i = 0; i < atomCounts[atype-1]; i++, ++v) {
				stream.readline();
				if(sscanf(stream.line().c_str(), FLOAT_SCANF_STRING_3, &v->X, &v->Y, &v->Z) != 3)
					throw Exception(tr("Invalid atom velocity vector (line %1): %2").arg(stream.lineNumber()).arg(stream.line().c_str()));
				if(!isCartesian)
					*v = cell * (*v);
			}
		}
	}

	posChannel->setHighQualityRenderingInViewports(true);

	destination->invalidate();

	QString statusMessage = tr("Number of atoms: %1").arg(totalAtomCount);
	return EvaluationStatus(EvaluationStatus::EVALUATION_SUCCESS, statusMessage);
}

};	// End of namespace AtomViz
