///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include "ColumnChannelMappingEditor.h"

namespace AtomViz {

/******************************************************************************
 * Associates a column in the data file with a custom DataChannel in the AtomsObject.
 *****************************************************************************/
ColumnChannelMappingEditor::ColumnChannelMappingEditor(QWidget* parent)
	: QWidget(parent)
{
	// Create the table sub-widget.
	QVBoxLayout* layout = new QVBoxLayout(this);
	layout->setSpacing(0);
	layout->setContentsMargins(0,0,0,0);

	QGridLayout* tableWidgetLayout = new QGridLayout();
	tableWidget = new QTableWidget(this);
	tableWidgetLayout->addWidget(tableWidget, 0, 0);
	tableWidgetLayout->setRowMinimumHeight(0, 250);
	tableWidgetLayout->setRowStretch(0, 1);
	tableWidgetLayout->setColumnMinimumWidth(0, 450);
	tableWidgetLayout->setColumnStretch(0, 1);
	layout->addLayout(tableWidgetLayout);

	tableWidget->setColumnCount(3);
	QStringList horizontalHeaders;
	horizontalHeaders << tr("Data type");
	horizontalHeaders << tr("Data channel");
	horizontalHeaders << tr("Compnt.");
	tableWidget->setHorizontalHeaderLabels(horizontalHeaders);
	tableWidget->resizeColumnToContents(0);
	tableWidget->resizeColumnToContents(2);
	tableWidget->setEditTriggers(QAbstractItemView::AllEditTriggers);

	// Calculate the optimum with of the 2. column.
	QComboBox* box = new QComboBox();
	box->setSizeAdjustPolicy(QComboBox::AdjustToContents);
	QMapIterator<QString, DataChannel::DataChannelIdentifier> i(DataChannel::standardChannelList());
	while(i.hasNext()) {
		i.next();
		box->addItem(i.key(), i.value());
	}
	tableWidget->setColumnWidth(1, box->sizeHint().width());

	class ChannelDataTypeItemDelegate : public QItemDelegate {
	public:
		virtual QWidget* createEditor(QWidget* parent, const QStyleOptionViewItem& option, const QModelIndex& index ) const {
			QComboBox* box = new QComboBox(parent);
			box->addItem(dataChannelTypeToString(QMetaType::Void), QMetaType::Void);
			box->addItem(dataChannelTypeToString(qMetaTypeId<int>()), qMetaTypeId<int>());
			box->addItem(dataChannelTypeToString(qMetaTypeId<FloatType>()), qMetaTypeId<FloatType>());
			return box;
		}
		virtual void setEditorData(QWidget* editor, const QModelIndex& index) const {
			int value = index.model()->data(index, Qt::UserRole).toInt();
			QComboBox* box = static_cast<QComboBox*>(editor);
			box->setCurrentIndex(box->findData(value));
		}
		virtual void setModelData(QWidget* editor, QAbstractItemModel* model, const QModelIndex& index) const {
			QComboBox* box = static_cast<QComboBox*>(editor);
			int newType = box->itemData(box->currentIndex()).toInt();
			model->setData(index, newType, Qt::UserRole);
			model->setData(index, dataChannelTypeToString(newType));
			if(newType == QMetaType::Void) {
				model->setData(index.sibling(index.row(), 2), 0, Qt::UserRole);
				model->setData(index.sibling(index.row(), 2), "");
			}
			DataChannel::DataChannelIdentifier standardChannelId = (DataChannel::DataChannelIdentifier)index.sibling(index.row(), 1).data(Qt::UserRole).toInt();
			if(standardChannelId != DataChannel::UserDataChannel) {
				if(DataChannel::standardChannelType(standardChannelId) != newType) {
					model->setData(index.sibling(index.row(), 1), 0, Qt::UserRole);
					model->setData(index.sibling(index.row(), 1), QString());
					model->setData(index.sibling(index.row(), 2), 0, Qt::UserRole);
					model->setData(index.sibling(index.row(), 2), "");
				}
			}
		}
	};
	static ChannelDataTypeItemDelegate dataTypeItemDelegate;

	class ChannelNameItemDelegate : public QItemDelegate {
	public:
		virtual QWidget* createEditor(QWidget* parent, const QStyleOptionViewItem& option, const QModelIndex& index) const {
			QComboBox* box = new QComboBox(parent);
			box->setEditable(true);
			box->setDuplicatesEnabled(false);
			QMapIterator<QString, DataChannel::DataChannelIdentifier> i(DataChannel::standardChannelList());
			while(i.hasNext()) {
				i.next();
				box->addItem(i.key(), i.value());
			}
			return box;
		}
		virtual void setEditorData(QWidget* editor, const QModelIndex& index) const {
			int value = index.model()->data(index, Qt::UserRole).toInt();
			QComboBox* box = static_cast<QComboBox*>(editor);
			if(value < 0)
				box->setCurrentIndex(box->findData(value));
			else
				box->setEditText(index.model()->data(index).toString());
		}
		virtual void setModelData(QWidget* editor, QAbstractItemModel* model, const QModelIndex& index) const {
			QComboBox* box = static_cast<QComboBox*>(editor);
			QString newValue = box->currentText().trimmed();
			model->setData(index, newValue);
			DataChannel::DataChannelIdentifier id = DataChannel::standardChannelList().value(newValue);
			if(id != DataChannel::UserDataChannel) {
				model->setData(index, id, Qt::UserRole);
				int newType = DataChannel::standardChannelType(id);
				model->setData(index.sibling(index.row(), 0), dataChannelTypeToString(newType));
				model->setData(index.sibling(index.row(), 0), newType, Qt::UserRole);
				if(newType == QMetaType::Void) {
					model->setData(index.sibling(index.row(), 2), 0, Qt::UserRole);
					model->setData(index.sibling(index.row(), 2), "");
				}
				else {
					int vectorComponent = min(index.sibling(index.row(), 2).data(Qt::UserRole).toInt(), (int)DataChannel::standardChannelComponentCount(id)-1);
					QString componentName;
					if(DataChannel::standardChannelComponentNames(id).size() > vectorComponent) componentName = DataChannel::standardChannelComponentNames(id)[vectorComponent];
					model->setData(index.sibling(index.row(), 2), vectorComponent, Qt::UserRole);
					model->setData(index.sibling(index.row(), 2), componentName);
				}
			}
			else {
				model->setData(index, 0, Qt::UserRole);
				if(newValue.isEmpty()) {
					model->setData(index.sibling(index.row(), 0), dataChannelTypeToString(QMetaType::Void));
					model->setData(index.sibling(index.row(), 0), QMetaType::Void, Qt::UserRole);
					model->setData(index.sibling(index.row(), 2), 0, Qt::UserRole);
					model->setData(index.sibling(index.row(), 2), "");
				}
			}
		}
	};
	static ChannelNameItemDelegate nameItemDelegate;

	class VectorComponentItemDelegate : public QItemDelegate {
	public:
		virtual QWidget* createEditor(QWidget* parent, const QStyleOptionViewItem& option, const QModelIndex& index) const {
			QComboBox* box = new QComboBox(parent);
			return box;
		}
		virtual void setEditorData(QWidget* editor, const QModelIndex& index) const {
			int value = index.model()->data(index, Qt::UserRole).toInt();
			OVITO_ASSERT(value >= 0);
			QComboBox* box = static_cast<QComboBox*>(editor);
			int dataType = index.sibling(index.row(), 0).data(Qt::UserRole).toInt();
			DataChannel::DataChannelIdentifier dataChannelId = (DataChannel::DataChannelIdentifier)index.sibling(index.row(), 1).data(Qt::UserRole).toInt();
			box->clear();
			if(dataType != QMetaType::Void) {
				if(dataChannelId != DataChannel::UserDataChannel) {
					box->setEditable(false);
					Q_FOREACH(QString name, DataChannel::standardChannelComponentNames(dataChannelId))
						box->addItem(name);
				}
				else {
					box->setEditable(true);
					for(int i=1; i<10; i++)
						box->addItem(QString::number(i));
				}
				box->setCurrentIndex(value);
				box->setEnabled(box->count() > 0);
			}
			else box->setEnabled(false);
		}
		virtual void setModelData(QWidget* editor, QAbstractItemModel* model, const QModelIndex& index) const {
			QComboBox* box = static_cast<QComboBox*>(editor);
			if(box->currentIndex() >= 0) {
				model->setData(index, box->currentIndex(), Qt::UserRole);
				model->setData(index, box->currentText());
			}
			else {
				model->setData(index, 0, Qt::UserRole);
				model->setData(index, "");
			}
		}
	};
	static VectorComponentItemDelegate vectorComponentItemDelegate;

	tableWidget->setItemDelegateForColumn(0, &dataTypeItemDelegate);
	tableWidget->setItemDelegateForColumn(1, &nameItemDelegate);
	tableWidget->setItemDelegateForColumn(2, &vectorComponentItemDelegate);

	QHBoxLayout* layout2 = new QHBoxLayout();
	layout->addSpacing(6);
	layout->addLayout(layout2);

	QPushButton* presetMenuButton = new QPushButton(tr("Preset menu"), this);
	presetMenuButton->setMenu(&presetMenu);
	connect(&presetMenu, SIGNAL(aboutToShow()), this, SLOT(updatePresetMenu()));
	layout2->addWidget(presetMenuButton);
	layout2->addStretch(1);

	autoAssignButton = new QPushButton(tr("Auto-assign columns"), this);
	connect(autoAssignButton, SIGNAL(clicked(bool)), this, SLOT(onAutoAssignColumns()));
	layout2->addWidget(autoAssignButton);
}

/******************************************************************************
 * Fills the editor with the given mapping.
 *****************************************************************************/
void ColumnChannelMappingEditor::setMapping(const ColumnChannelMapping& mapping)
{
	// Update the number of table rows.
	// Create one row for each column in the input file.
	tableWidget->setRowCount(mapping.columnCount());
	bool hasColumnNames = false;
	for(int i=0; i<mapping.columnCount(); i++) {
		QTableWidgetItem* item;
		if(mapping.getColumnName(i).isEmpty())
			item = new QTableWidgetItem(tr("Col. %1").arg(i+1));
		else {
			item = new QTableWidgetItem(mapping.getColumnName(i));
			item->setData(Qt::UserRole, mapping.getColumnName(i));
			hasColumnNames = true;
		}
		tableWidget->setVerticalHeaderItem(i, item);
	}
	autoAssignButton->setEnabled(hasColumnNames);

	// Fill in the table cells.
	for(int i=0; i<mapping.columnCount(); i++) {

		QTableWidgetItem* dataTypeItem = new QTableWidgetItem(dataChannelTypeToString(mapping.getChannelType(i)));
		dataTypeItem->setData(Qt::UserRole, mapping.getChannelType(i));
		dataTypeItem->setFlags(Qt::ItemIsSelectable | Qt::ItemIsEditable | Qt::ItemIsEnabled);
		tableWidget->setItem(i, 0, dataTypeItem);

		QTableWidgetItem* nameItem = new QTableWidgetItem(mapping.getChannelName(i));
		nameItem->setData(Qt::UserRole, mapping.getChannelId(i));
		nameItem->setFlags(Qt::ItemIsSelectable | Qt::ItemIsEditable | Qt::ItemIsEnabled);
		tableWidget->setItem(i, 1, nameItem);

		OVITO_ASSERT(mapping.getVectorComponent(i) >= 0);
		int vectorComponent = mapping.getVectorComponent(i);
		QString componentName;
		if(mapping.getChannelId(i) != DataChannel::UserDataChannel) {
			DataChannel::DataChannelIdentifier stdId = mapping.getChannelId(i);
			vectorComponent = min(vectorComponent, (int)DataChannel::standardChannelComponentCount(stdId)-1);
			if(DataChannel::standardChannelComponentNames(stdId).size() > vectorComponent) componentName = DataChannel::standardChannelComponentNames(stdId)[vectorComponent];
		}
		else if(mapping.getChannelType(i) != QMetaType::Void) componentName = QString::number(vectorComponent+1);

		QTableWidgetItem* vectorComponentItem = new QTableWidgetItem(componentName);
		vectorComponentItem->setData(Qt::UserRole, vectorComponent);
		vectorComponentItem->setFlags(Qt::ItemIsSelectable | Qt::ItemIsEditable | Qt::ItemIsEnabled);
		tableWidget->setItem(i, 2, vectorComponentItem);
	}

	tableWidget->resizeRowsToContents();
}

/******************************************************************************
 * Returns the current contents of the editor.
 * Throws an Exception if the current mapping in the editor is not valid.
 *****************************************************************************/
ColumnChannelMapping ColumnChannelMappingEditor::mapping() const
{
	ColumnChannelMapping mapping;
	for(int row = 0; row < tableWidget->rowCount(); row++) {
		int dataType = tableWidget->item(row, 0)->data(Qt::UserRole).toInt();
		DataChannel::DataChannelIdentifier channelId = (DataChannel::DataChannelIdentifier)tableWidget->item(row, 1)->data(Qt::UserRole).toInt();
		QString channelName = tableWidget->item(row, 1)->data(Qt::DisplayRole).toString().trimmed();
		int vectorComponent = tableWidget->item(row, 2)->data(Qt::UserRole).toInt();
		OVITO_ASSERT(vectorComponent >= 0);

		QString columnName = tableWidget->verticalHeaderItem(row)->data(Qt::UserRole).toString();

		if(dataType == QMetaType::Void || channelName.isEmpty()) {
			mapping.ignoreColumn(row, columnName);
		}
		else {
			OVITO_ASSERT(channelId == DataChannel::UserDataChannel || channelName == DataChannel::standardChannelName(channelId));
			DataChannel::DataChannelIdentifier standardChannel = DataChannel::standardChannelList().value(channelName);
			if(standardChannel != DataChannel::UserDataChannel) {
				channelId = standardChannel;
				if(dataType != DataChannel::standardChannelType(standardChannel))
					throw Exception(tr("The standard data channel '%1' associated with column %2 has a wrong data type. It must be set to data type %3.")
						.arg(channelName).arg(row+1).arg(dataChannelTypeToString(DataChannel::standardChannelType(standardChannel))));
			}
			mapping.defineColumn(row, channelId, channelName, dataType, vectorComponent, columnName);
		}
	}
	return mapping;
}

/******************************************************************************
 * Updates the entries in the "Load Preset" menu.
 *****************************************************************************/
void ColumnChannelMappingEditor::updatePresetMenu()
{
	presetMenu.clear();

	QMenu* loadMenu = presetMenu.addMenu(QIcon(":/atomviz/mapping_preset_load.png"), tr("&Load Mapping"));
	Q_FOREACH(QString name, ColumnChannelMapping::listPresets()) {
		loadMenu->addAction(name, this, SLOT(onLoadPreset()));
	}
	loadMenu->setEnabled(!loadMenu->isEmpty());

	QMenu* saveMenu = presetMenu.addMenu(QIcon(":/atomviz/mapping_preset_save.png"), tr("&Save Mapping"));
	saveMenu->addAction(tr("&New Preset..."), this, SLOT(onSavePresetAs()));
	saveMenu->addSeparator();
	Q_FOREACH(QString name, ColumnChannelMapping::listPresets()) {
		saveMenu->addAction(name, this, SLOT(onSavePreset()));
	}

	QMenu* deleteMenu = presetMenu.addMenu(QIcon(":/atomviz/mapping_preset_delete.png"), tr("&Delete Mapping"));
	Q_FOREACH(QString name, ColumnChannelMapping::listPresets()) {
		deleteMenu->addAction(name, this, SLOT(onDeletePreset()));
	}
	deleteMenu->setEnabled(!deleteMenu->isEmpty());
}

/******************************************************************************
 * Loads a preset.
 *****************************************************************************/
void ColumnChannelMappingEditor::onLoadPreset()
{
	QAction* action = (QAction*)sender();
	CHECK_POINTER(action);

	try {
		QString name = action->text();
		ColumnChannelMapping m;
		m.loadPreset(name);

		m.shrink();
		if(m.columnCount() > tableWidget->rowCount()) {
			QMessageBox::warning(this, tr("Load Mapping"), tr("The column mapping '%1' contains more data columns than the atoms file. Additional columns will be removed.").arg(name));
		}
		m.setColumnCount(tableWidget->rowCount());

		setMapping(m);
	}
	catch(const Exception& ex) {
		ex.showError();
	}
}

/******************************************************************************
 * Saves a preset under a new name.
 *****************************************************************************/
void ColumnChannelMappingEditor::onSavePresetAs()
{
	try {
		ColumnChannelMapping m = ColumnChannelMappingEditor::mapping();

		QString name = QInputDialog::getText(this, tr("Save Mapping"),
			tr("Please enter a name for the column mapping:"));
		if(name.isEmpty()) return;

		if(ColumnChannelMapping::listPresets().contains(name)) {
			if(QMessageBox::question(this, tr("Save Mapping"),
					tr("There already a stored mapping with the same name. Do you want to overwrite it?"),
					QMessageBox::Yes|QMessageBox::No, QMessageBox::Yes) != QMessageBox::Yes)
				return;
		}

		m.savePreset(name);
	}
	catch(const Exception& ex) {
		ex.showError();
	}
}

/******************************************************************************
 * Saves a preset under an existing name.
 *****************************************************************************/
void ColumnChannelMappingEditor::onSavePreset()
{
	QAction* action = (QAction*)sender();
	CHECK_POINTER(action);

	try {
		ColumnChannelMapping m = ColumnChannelMappingEditor::mapping();

		QString name = action->text();
		if(name.isEmpty()) return;

		if(ColumnChannelMapping::listPresets().contains(name)) {
			if(QMessageBox::question(this, tr("Save Mapping"),
					tr("Do you want to overwrite the existing mapping '%1'?").arg(name),
					QMessageBox::Yes|QMessageBox::No, QMessageBox::Yes) != QMessageBox::Yes)
				return;
		}

		m.savePreset(name);
	}
	catch(const Exception& ex) {
		ex.showError();
	}
}

/******************************************************************************
 * Deletes a preset.
 *****************************************************************************/
void ColumnChannelMappingEditor::onDeletePreset()
{
	QAction* action = (QAction*)sender();
	CHECK_POINTER(action);

	try {
		QString name = action->text();

		if(QMessageBox::question(this, tr("Delete Mapping"),
				tr("Do you really want to delete the column mapping '%1'?").arg(name),
				QMessageBox::Yes|QMessageBox::No, QMessageBox::Yes) != QMessageBox::Yes)
			return;

		ColumnChannelMapping::deletePreset(name);
	}
	catch(const Exception& ex) {
		ex.showError();
	}
}

/******************************************************************************
 * Creates the default mapping if column names are available.
 *****************************************************************************/
void ColumnChannelMappingEditor::onAutoAssignColumns()
{
	ColumnChannelMapping m = mapping();
	for(int i=0; i<m.columnCount(); i++) {
		QString name = m.getColumnName(i);
		if(name.isEmpty()) continue;
		else if(name == "x" || name == "xu" || name == "xs") m.defineStandardColumn(i, DataChannel::PositionChannel, 0, name);
		else if(name == "y" || name == "yu" || name == "ys") m.defineStandardColumn(i, DataChannel::PositionChannel, 1, name);
		else if(name == "z" || name == "zu" || name == "zs") m.defineStandardColumn(i, DataChannel::PositionChannel, 2, name);
		else if(name == "vx") m.defineStandardColumn(i, DataChannel::VelocityChannel, 0, name);
		else if(name == "vy") m.defineStandardColumn(i, DataChannel::VelocityChannel, 1, name);
		else if(name == "vz") m.defineStandardColumn(i, DataChannel::VelocityChannel, 2, name);
		else if(name == "id") m.defineStandardColumn(i, DataChannel::AtomIndexChannel, 0, name);
		else if(name == "type") m.defineStandardColumn(i, DataChannel::AtomTypeChannel, 0, name);
		else if(name == "mass") m.defineStandardColumn(i, DataChannel::MassChannel, 0, name);
		else if(name == "ix") m.defineStandardColumn(i, DataChannel::PeriodicImageChannel, 0, name);
		else if(name == "iy") m.defineStandardColumn(i, DataChannel::PeriodicImageChannel, 1, name);
		else if(name == "iz") m.defineStandardColumn(i, DataChannel::PeriodicImageChannel, 2, name);
		else if(name == "fx") m.defineStandardColumn(i, DataChannel::ForceChannel, 0, name);
		else if(name == "fy") m.defineStandardColumn(i, DataChannel::ForceChannel, 1, name);
		else if(name == "fz") m.defineStandardColumn(i, DataChannel::ForceChannel, 2, name);
		else if(name == "c_cna") m.defineStandardColumn(i, DataChannel::CNATypeChannel, 0, name);
		else if(name == "c_epot") m.defineStandardColumn(i, DataChannel::PotentialEnergyChannel, 0, name);
		else if(name == "c_kpot") m.defineStandardColumn(i, DataChannel::KineticEnergyChannel, 0, name);
		else if(name == "c_stress[1]") m.defineStandardColumn(i, DataChannel::StressTensorChannel, 0, name);
		else if(name == "c_stress[2]") m.defineStandardColumn(i, DataChannel::StressTensorChannel, 1, name);
		else if(name == "c_stress[3]") m.defineStandardColumn(i, DataChannel::StressTensorChannel, 2, name);
		else if(name == "c_stress[4]") m.defineStandardColumn(i, DataChannel::StressTensorChannel, 3, name);
		else if(name == "c_stress[5]") m.defineStandardColumn(i, DataChannel::StressTensorChannel, 4, name);
		else if(name == "c_stress[6]") m.defineStandardColumn(i, DataChannel::StressTensorChannel, 5, name);
		else if(name == "selection") m.defineStandardColumn(i, DataChannel::SelectionChannel, 0, name);
		else {
			// First assume that the column cannot be mapped to one of the standard data channels.
			m.defineColumn(i, DataChannel::UserDataChannel, name, qMetaTypeId<FloatType>(), 0, name);

			// Check if the column name matches one of the standard channel names.
			QMap<QString, DataChannel::DataChannelIdentifier> stdChannels = DataChannel::standardChannelList();
			for(QMap<QString, DataChannel::DataChannelIdentifier>::const_iterator iter = stdChannels.constBegin(); iter != stdChannels.constEnd(); ++iter) {
				QString columnName = iter.key();
				columnName.remove(QRegExp("[^A-Za-z\\d_]"));
				QStringList componentNames = DataChannel::standardChannelComponentNames(iter.value());
				if(componentNames.empty()) {
					if(columnName == name) {
						m.defineStandardColumn(i, iter.value(), 0, name);
						break;
					}
				}
				else {
					bool match = false;
					for(int c = 0; c < componentNames.size(); c++) {
						QString fullColumnName = columnName + "." + componentNames[c];
						if(fullColumnName == name) {
							m.defineStandardColumn(i, iter.value(), c, name);
							match = true;
							break;
						}
					}
					if(match) break;
				}
			}
		}
	}
	setMapping(m);
}

/******************************************************************************
 * Returns the string representation of a data channel type.
 *****************************************************************************/
QString ColumnChannelMappingEditor::dataChannelTypeToString(int type)
{
	if(type == qMetaTypeId<int>()) return tr("Integer");
	else if(type == qMetaTypeId<FloatType>()) return tr("Float");
	else return tr("None");
}

};	// End of namespace AtomViz
