/* OpenVAS Manager - OMP Command Line Interface
 * $Id$
 * Description: Provides a command line client for the OpenVAS Manager
 *
 * Authors:
 * Matthew Mundell <matt@mundell.ukfsn.org>
 * Michael Wiegand <michael.wiegand@intevation.de>
 *
 * Copyright:
 * Copyright (C) 2009 Greenbone Networks GmbH
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2,
 * or, at your option, any later version as published by the Free
 * Software Foundation
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 */


/**
 * @file  omp-cli.c
 * @brief The OMP Command Line Interface
 *
 * TODO: Add documentation
 */

#include <assert.h>
#include <glib.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "corevers.h"

#include <openvas/openvas_server.h>
#include <openvas/openvas_logging.h>
#include <openvas/omp/omp.h>
#include <openvas/omp/xml.h>

/**
 * @brief The name of this program.
 */
#define OMPCLI_PROGNAME "omp-cli"

/**
 * @brief The version number of this program.
 */
#ifndef OMPCLI_VERSION
#define OMPCLI_VERSION "0.0.1"
#endif

/**
 * @brief Default Manager (openvasmd) address.
 */
#define OPENVASMD_ADDRESS "127.0.0.1"

/**
 * @brief Default Manager port.
 */
#define OPENVASMD_PORT 9390


/* Connection handling. */

/**
 * @brief Structure used to bundle the information needed to handle
 * @brief a connection to a server.
 */
typedef struct {
  gnutls_session_t session;   ///< GnuTLS Session to use.
  int socket;                 ///< Socket to manager.
  gchar* username;            ///< Username with which to connect.
  gchar* password;            ///< Password for user with which to connect.
  gchar* host_string;         ///< Server host string.
  gchar* port_string;         ///< Server port string.
  gint port;                  ///< Port of manager.
} server_connection_t;

/** @todo Return on fail. */
/**
 * @brief Connect to an openvas-manager, exit (\ref EXIT_FAILURE) if connection
 * @brief could not be established or authentication failed.
 *
 * Exits in the fail case and prints a message to stderr.
 *
 * @return TRUE. Does not return in fail case.
 */
static gboolean
manager_open (server_connection_t* connection)
{
  connection->socket = openvas_server_open (&connection->session,
                                            connection->host_string,
                                            connection->port);

  if (connection->socket == -1)
    {
      fprintf (stderr, "Failed to acquire socket.\n");
      exit (EXIT_FAILURE);
    }

  if (omp_authenticate (&connection->session,
                        connection->username,
                        connection->password))
    {
      openvas_server_close (connection->socket, connection->session);
      fprintf (stderr, "Failed to authenticate.\n");
      exit (EXIT_FAILURE);
    }

  return TRUE;
}

/**
 * @brief Closes the connection to a manager.
 *
 * @return 0 on success, -1 on failure.
 */
static int
manager_close (server_connection_t* server)
{
  return openvas_server_close (server->socket, server->session);
}

/**
 * @brief Print tasks.
 *
 * @param[in]  tasks  Tasks.
 *
 * @return 0 success, -1 error.
 */
static int
print_tasks (entities_t tasks)
{
  entity_t task;
  while ((task = first_entity (tasks)))
    {
      if (strcmp (entity_name (task), "task") == 0)
        {
          entity_t entity, report;
          entities_t reports;
          const char *id, *name, *status, *progress;

          id = entity_attribute (task, "id");
          if (id == NULL)
            {
              fprintf (stderr, "Failed to parse task ID.\n");
              return -1;
            }

          entity = entity_child (task, "name");
          if (entity == NULL)
            {
              fprintf (stderr, "Failed to parse task name.\n");
              return -1;
            }
          name = entity_text (entity);

          entity = entity_child (task, "status");
          if (entity == NULL)
            {
              fprintf (stderr, "Failed to parse task status.\n");
              return -1;
            }
          status = entity_text (entity);

          entity = entity_child (task, "progress");
          if (entity == NULL)
            {
              fprintf (stderr, "Failed to parse task progress.\n");
              return -1;
            }
          progress = entity_text (entity);

          printf ("%s  %-7s", id, status);
          if (strcmp (status, "Running") == 0)
            printf (" %2s%%  %s\n", progress, name);
          else
            printf ("      %s\n", name);

          /* Print any reports indented under the task. */

          reports = task->entities;
          while ((report = first_entity (reports)))
            {
              if (strcmp (entity_name (report), "report") == 0)
                {
                  entity_t messages;
                  const char *id, *status, *holes, *infos, *logs, *warnings;
                  const char *time_stamp;

                  id = entity_attribute (report, "id");
                  if (id == NULL)
                    {
                      fprintf (stderr, "Failed to parse report ID.\n");
                      return -1;
                    }

                  entity = entity_child (report, "scan_run_status");
                  if (entity == NULL)
                    {
                      fprintf (stderr, "Failed to parse report status.\n");
                      return -1;
                    }
                  status = entity_text (entity);

                  messages = entity_child (report, "messages");
                  if (entity == NULL)
                    {
                      fprintf (stderr, "Failed to parse report messages.\n");
                      return -1;
                    }

                  entity = entity_child (messages, "hole");
                  if (entity == NULL)
                    {
                      fprintf (stderr, "Failed to parse report hole.\n");
                      return -1;
                    }
                  holes = entity_text (entity);

                  entity = entity_child (messages, "info");
                  if (entity == NULL)
                    {
                      fprintf (stderr, "Failed to parse report info.\n");
                      return -1;
                    }
                  infos = entity_text (entity);

                  entity = entity_child (messages, "log");
                  if (entity == NULL)
                    {
                      fprintf (stderr, "Failed to parse report log.\n");
                      return -1;
                    }
                  logs = entity_text (entity);

                  entity = entity_child (messages, "warning");
                  if (entity == NULL)
                    {
                      fprintf (stderr, "Failed to parse report warning.\n");
                      return -1;
                    }
                  warnings = entity_text (entity);

                  entity = entity_child (report, "timestamp");
                  if (entity == NULL)
                    {
                      fprintf (stderr, "Failed to parse report timestamp.\n");
                      return -1;
                    }
                  time_stamp = entity_text (entity);

                  printf ("  %s  %-7s  %2s  %2s  %2s  %2s  %s\n",
                          id, status, holes, warnings, infos, logs, time_stamp);
                }
              reports = next_entities (reports);
            }
        }
      tasks = next_entities (tasks);
    }
  return 0;
}


/* Commands. */

/**
 * @brief Performs the omp get_report command.
 *
 * This function exit (\ref EXIT_SUCCES) in case of success and
 * exit (\ref EXIT_FAILURE) in case of failure.
 *
 * @param connection Connection to manager to use.
 * @param point      Pointer to task_uuid id.
 * @param format     Queried report format.
 */
static int
manager_get_reports (server_connection_t* connection,
                     gchar** report_ids,
                     gchar* format)
{
  if (format == NULL || strcasecmp (format, "xml") == 0)
    {
      entity_t entity, report_xml;

      if (omp_get_report (&(connection->session),
                          *report_ids,
                          "xml",
                          &entity))
        {
          fprintf (stderr, "Failed to get report.\n");
          manager_close (connection);
          return -1;
        }

      report_xml = entity_child (entity, "report");
      if (report_xml == NULL)
        {
          free_entity (entity);
          fprintf (stderr, "Failed to get report.\n");
          manager_close (connection);
          return -1;
        }

      print_entity (stdout, report_xml);
    }
  else
    {
      void *report;
      gsize report_size;

      if (omp_get_report_format (&(connection->session),
                                 *report_ids,
                                 format,
                                 &report,
                                 &report_size))
        {
          fprintf (stderr, "Failed to get report.\n");
          manager_close (connection);
          return -1;
        }

      if (fwrite (report, 1, report_size, stdout) < report_size)
        {
          fprintf (stderr, "Failed to write entire report.\n");
          manager_close (connection);
          return -1;
        }
    }

  return 0;
}


/* Entry point. */

int
main (int argc, char** argv)
{
  server_connection_t* connection = NULL;
  /* The return status of the command. */
  int exit_status = -1;

  /* Global options. */
  static gboolean prompt = FALSE;
  static gboolean print_version = FALSE;
  static gboolean be_verbose = FALSE;
  static gchar *manager_host_string = NULL;
  static gchar *manager_port_string = NULL;
  static gchar *omp_username = NULL;
  static gchar *omp_password = NULL;
  /* Shared command options. */
  static gchar *name = NULL;
  /* Command create-task. */
  static gboolean cmd_create_task = FALSE;
  static gchar *comment = NULL;
  static gchar *config = NULL;
  static gboolean rc = FALSE;
  static gchar *target = NULL;
  /* Command delete-report. */
  static gboolean cmd_delete_report = FALSE;
  /* Command delete-task. */
  static gboolean cmd_delete_task = FALSE;
  /* Command get-report. */
  static gboolean cmd_get_report = FALSE;
  static gchar *format = NULL;
  /* Command get-status. */
  static gboolean cmd_get_status = FALSE;
  /* Command modify-task. */
  static gboolean cmd_modify_task = FALSE;
  static gboolean file = FALSE;
  /* Command start-task. */
  static gboolean cmd_start_task = FALSE;
  /* Command given as XML. */
  static gchar *cmd_xml = NULL;
  /* The rest of the args. */
  static gchar **rest = NULL;

  GError *error = NULL;

  GOptionContext *option_context;
  static GOptionEntry option_entries[]
    = {
        /* Global options. */
        { "host", 'h', 0, G_OPTION_ARG_STRING, &manager_host_string,
          "Connect to manager on host <host>", "<host>" },
        { "port", 'p', 0, G_OPTION_ARG_STRING, &manager_port_string,
          "Use port number <number>", "<number>" },
        { "version", 'V', 0, G_OPTION_ARG_NONE, &print_version,
          "Print version.", NULL },
        { "verbose", 'v', 0, G_OPTION_ARG_NONE, &be_verbose,
          "Verbose messages.", NULL },
        { "username", 'u', 0, G_OPTION_ARG_STRING, &omp_username,
          "OMP username", "<username>" },
        { "password", 'w', 0, G_OPTION_ARG_STRING, &omp_password,
          "OMP password", "<password>" },
        { "prompt", 'P', 0, G_OPTION_ARG_NONE, &prompt,
          "Prompt to exit.", NULL },
        /* Shared command options. */
        { "name", 'n', 0, G_OPTION_ARG_STRING, &name,
          "Name for create-task.",
          "<name>" },
        /* Command create-task. */
        { "create-task", 'C', 0, G_OPTION_ARG_NONE, &cmd_create_task,
          "Create a task.", NULL },
        { "comment", 'm', 0, G_OPTION_ARG_STRING, &comment,
          "Comment for create-task.",
          "<name>" },
        { "config", 'c', 0, G_OPTION_ARG_STRING, &config,
          "Config for create-task.",
          "<config>" },
        { "rc", 'r', 0, G_OPTION_ARG_NONE, &rc,
          "Create task with RC read from stdin.", NULL },
        { "target", 't', 0, G_OPTION_ARG_STRING, &target,
          "Target for create-task.",
          "<target>" },
        /* Command delete-report. */
        { "delete-report", 'E', 0, G_OPTION_ARG_NONE, &cmd_delete_report,
          "Delete one or more reports.", NULL },
        /* Command delete-task. */
        { "delete-task", 'D', 0, G_OPTION_ARG_NONE, &cmd_delete_task,
          "Delete one or more tasks.", NULL },
        /* Command get-report. */
        { "get-report", 'R', 0, G_OPTION_ARG_NONE, &cmd_get_report,
          "Get report of one task.", NULL },
        { "format", 'f', 0, G_OPTION_ARG_STRING, &format,
          "Format for get-report.",
          "<format>" },
        /* Command get-status. */
        { "get-status", 'G', 0, G_OPTION_ARG_NONE, &cmd_get_status,
          "Get status of one, many or all tasks.", NULL },
        /* Command start-task. */
        { "start-task", 'S', 0, G_OPTION_ARG_NONE, &cmd_start_task,
          "Start one or more tasks.", NULL },
        /* Command modify-task. */
        { "modify-task", 'M', 0, G_OPTION_ARG_NONE, &cmd_modify_task,
          "Modify a task.", NULL },
        { "file", 0, 0, G_OPTION_ARG_NONE, &file,
          "Add text in stdin as file on task.", NULL },
        /* Command as XML. */
        { "xml", 'X', 0, G_OPTION_ARG_STRING, &cmd_xml,
          "XML command (e.g. \"<help/>\", \"<get_version/>\")",
          "<command>" },
        { G_OPTION_REMAINING, 0, 0, G_OPTION_ARG_STRING_ARRAY, &rest,
          NULL, NULL },
        { NULL }
      };

  option_context = g_option_context_new ("- OpenVAS OMP Command"
                                         " Line Interface");
  g_option_context_add_main_entries (option_context, option_entries, NULL);
  if (!g_option_context_parse (option_context, &argc, &argv, &error))
    {
      printf ("%s\n\n", error->message);
      exit (EXIT_FAILURE);
    }

  if (print_version)
    {
      printf ("OMP Command Line Interface (%s) %s for %s\n",
              OMPCLI_PROGNAME, OMPCLI_VERSION, OPENVAS_OS_NAME);
      printf ("Copyright (C) 2009 Greenbone Networks GmbH\n\n");
      exit (EXIT_SUCCESS);
    }

  /* Check that one and at most one command option is present. */
  {
    int commands;
    commands = (int) cmd_create_task
               + (int) cmd_delete_report
               + (int) cmd_delete_task
               + (int) cmd_get_report
               + (int) cmd_get_status
               + (int) cmd_modify_task
               + (int) cmd_start_task
               + (int) (cmd_xml != NULL);
    if (commands == 0)
      {
        fprintf (stderr, "One command option must be present.\n");
        exit (EXIT_FAILURE);
      }
    if (commands > 1)
      {
        fprintf (stderr, "Only one command option must be present.\n");
        exit (EXIT_FAILURE);
      }
  }

  /* Setup the connection structure from the arguments. */

  connection = g_malloc0 (sizeof (*connection));

  if (manager_host_string == NULL)
    manager_host_string = OPENVASMD_ADDRESS;
  connection->host_string = manager_host_string;

  if (manager_port_string != NULL)
    connection->port = atoi (manager_port_string);
  else
    connection->port = OPENVASMD_PORT;

  if (connection->port <= 0 || connection->port >= 65536)
    {
      fprintf (stderr, "Manager port must be a number between 0 and 65536.\n");
      exit (EXIT_FAILURE);
    }

  if (omp_username == NULL)
    omp_username = g_strdup (g_get_user_name ());
  connection->username = omp_username;

  if (omp_password == NULL)
    omp_password = g_strdup (omp_username);
  connection->password = omp_password;

  if (be_verbose)
    {
      printf ("Will try to connect to host %s, port %d...\n",
              connection->host_string,
              connection->port);
    }
  else
    {
      g_log_set_default_handler (openvas_log_silent, NULL);
    }

  /* Run the single command. */

  if (cmd_create_task)
    {
      char *id = NULL;

      if (rc && (config || target))
        {
          fprintf (stderr, "create-task rc given with config or target.\n");
          exit (EXIT_FAILURE);
        }

      manager_open (connection);

      if (rc)
        {
          gchar *content;
          gsize content_len;
          GIOChannel *stdin_channel;

          /* Mixing stream and file descriptor IO might lead to trouble. */
          error = NULL;
          stdin_channel = g_io_channel_unix_new (fileno (stdin));
          g_io_channel_read_to_end (stdin_channel, &content, &content_len, &error);
          g_io_channel_shutdown (stdin_channel, TRUE, NULL);
          g_io_channel_unref (stdin_channel);
          if (error)
            {
              fprintf (stderr, "failed to read from stdin: %s\n", error->message);
              g_error_free (error);
              exit (EXIT_FAILURE);
            }

          if (omp_create_task_rc (&(connection->session),
                                  content,
                                  content_len,
                                  name ? name : "unnamed task",
                                  comment ? comment : "",
                                  &id))
            {
              g_free (content);
              fprintf (stderr, "Failed to create task.\n");
              manager_close (connection);
              exit (EXIT_FAILURE);
            }
        }
      else
        {
          if (omp_create_task (&(connection->session),
                               name ? name : "unnamed task",
                               config ? config : "Full and fast",
                               target ? target : "Localhost",
                               comment ? comment : "",
                               &id))
            {
              fprintf (stderr, "Failed to create task.\n");
              manager_close (connection);
              exit (EXIT_FAILURE);
            }
        }

      printf (id);
      putchar ('\n');

      manager_close (connection);
      exit_status = 0;
    }
  else if (cmd_delete_report)
    {
      gchar **point = rest;

      if (point == NULL || *point == NULL)
        {
          fprintf (stderr, "delete-report requires at least one argument.\n");
          exit (EXIT_FAILURE);
        }

      manager_open (connection);

      while (*point)
        {
          if (omp_delete_report (&(connection->session), *point))
            {
              fprintf (stderr,
                       "Failed to delete report %s, exiting.\n",
                       *point);
              manager_close (connection);
              exit (EXIT_FAILURE);
            }
          point++;
        }

      manager_close (connection);
      exit_status = 0;
    }
  else if (cmd_delete_task)
    {
      gchar **point = rest;

      if (point == NULL || *point == NULL)
        {
          fprintf (stderr, "delete-task requires at least one argument.\n");
          exit (EXIT_FAILURE);
        }

      manager_open (connection);

      while (*point)
        {
          if (omp_delete_task (&(connection->session), *point))
            {
              fprintf (stderr, "Failed to delete task.\n");
              manager_close (connection);
              exit (EXIT_FAILURE);
            }
          point++;
        }

      manager_close (connection);
      exit_status = 0;
    }
  else if (cmd_get_status)
    {
      gchar **point = rest;
      entity_t status;

      manager_open (connection);

      if (point)
        while (*point)
          {
            if (omp_get_status (&(connection->session), *point, 0, &status))
              {
                fprintf (stderr, "Failed to get status of task %s.\n", *point);
                manager_close (connection);
                exit (EXIT_FAILURE);
              }
            else
              {
                if (print_tasks (status->entities))
                  {
                    manager_close (connection);
                    exit (EXIT_FAILURE);
                  }
              }

            point++;
          }
      else
        {
          if (omp_get_status (&(connection->session), NULL, 0, &status))
            {
              fprintf (stderr, "Failed to get status of all tasks.\n");
              exit (EXIT_FAILURE);
            }
          if (print_tasks (status->entities))
            {
              manager_close (connection);
              exit (EXIT_FAILURE);
            }
        }

      manager_close (connection);
      exit_status = 0;
    }
  else if (cmd_get_report)
    {
      gchar** report_ids = rest;

      if (report_ids == NULL || *report_ids == NULL)
        {
          fprintf (stderr, "get-report requires one argument.\n");
          exit (EXIT_FAILURE);
        }

      manager_open (connection);
      exit_status = manager_get_reports (connection, report_ids, format);
    }
  else if (cmd_modify_task)
    {
      gchar **point = rest;
      gchar *content;
      gsize content_len;
      GIOChannel *stdin_channel;

      if (point == NULL || *point == NULL)
        {
          fprintf (stderr, "modify-task requires one argument.\n");
          exit (EXIT_FAILURE);
        }

      if (name == NULL)
        {
          fprintf (stderr, "modify-task requires the name option (path to file).\n");
          exit (EXIT_FAILURE);
        }

      if (file == FALSE)
        {
          fprintf (stderr, "modify-task requires the file option.\n");
          exit (EXIT_FAILURE);
        }

      if (file)
        {
          manager_open (connection);

          /* Mixing stream and file descriptor IO might lead to trouble. */
          error = NULL;
          stdin_channel = g_io_channel_unix_new (fileno (stdin));
          g_io_channel_read_to_end (stdin_channel, &content, &content_len, &error);
          g_io_channel_shutdown (stdin_channel, TRUE, NULL);
          g_io_channel_unref (stdin_channel);
          if (error)
            {
              fprintf (stderr, "failed to read from stdin: %s\n", error->message);
              g_error_free (error);
              exit (EXIT_FAILURE);
            }

#if 0
          /** todo As in get-report, this is how the commands will work. */
          exit_status = manager_modify_task_file (connection,
                                                  *point,
                                                  name,
                                                  content,
                                                  content_len,
                                                  error);
#else
          if (omp_modify_task_file (&(connection->session), *point, name,
                                    content, content_len))
            {
              g_free (content);
              fprintf (stderr, "Failed to modify task.\n");
              manager_close (connection);
              exit (EXIT_FAILURE);
            }

          manager_close (connection);
          exit_status = 0;
#endif
        }
    }
  else if (cmd_start_task)
    {
      gchar **point = rest;

      if (point == NULL || *point == NULL)
        {
          fprintf (stderr, "start-task requires at least one argument.\n");
          exit (EXIT_FAILURE);
        }

      manager_open (connection);

      while (*point)
        {
          char *report_id;
          if (omp_start_task_report (&(connection->session),
                                     *point,
                                     &report_id))
            {
              fprintf (stderr, "Failed to start task.\n");
              manager_close (connection);
              exit (EXIT_FAILURE);
            }
          printf ("%s\n", report_id);
          free (report_id);
          point++;
        }
      exit_status = 0;

      manager_close (connection);
    }
  else if (cmd_xml)
    {
      manager_open (connection);

      /** @todo Move to connection_t and manager_open. */
      if (prompt)
        {
          fprintf (stderr, "Connected, press a key to continue.\n");
          getchar ();
        }

      if (be_verbose)
        printf ("Sending to manager: %s\n", cmd_xml);

      if (openvas_server_send (&(connection->session), cmd_xml) == -1)
        {
          manager_close (connection);
          fprintf (stderr, "Failed to send_to_manager.\n");
          exit (EXIT_FAILURE);
        }

      /* Read the response. */

      entity_t entity = NULL;
      if (read_entity (&(connection->session), &entity))
        {
          fprintf (stderr, "Failed to read response.\n");
          manager_close (connection);
          exit (EXIT_FAILURE);
        }

      if (be_verbose)
        printf ("Got response:\n");
      print_entity (stdout, entity);
      printf ("\n");

      /* Cleanup. */

      /** @todo Move to connection_t and manager_open. */
      if (prompt)
        {
          fprintf (stderr, "Press a key when done.\n");
          getchar ();
        }

      manager_close (connection);
      free_entity (entity);

      exit_status = 0;
    }
  else
    /* The option processing ensures that at least one command is present. */
    assert (0);

  /* Exit. */

  if (be_verbose)
    {
      if (exit_status)
        printf ("Command failed.\n");
      else
        printf ("Command completed successfully.\n");
    }

  exit (exit_status);
}
