#include "seqstandalone.h"
#include "seqplot_standalone.h"


#include <odinseq/seqsim.h>
#include <odinseq/seqmeth.h>

#include <tjutils/tjprofiler.h>

SeqTimecourseOpts::SeqTimecourseOpts()
 : JcampDxBlock("Timecourse Options") {

  JcampDxBlock::set_embedded(true);

  EddyCurrentAmpl=0.0;
  EddyCurrentAmpl.set_minmaxval(0.0,10.0);
  EddyCurrentAmpl.set_unit("%");
  EddyCurrentAmpl.set_description("Amplitude of eddy currents relative to the inducing gradient.");
  EddyCurrentAmpl.set_cmdline_option("ecamp");

  EddyCurrentTimeConst=2.0;
  EddyCurrentTimeConst.set_minmaxval(0.0,10.0);
  EddyCurrentTimeConst.set_unit(ODIN_TIME_UNIT);
  EddyCurrentTimeConst.set_description("Time constant of the exponentially decaying eddy currents.");
  EddyCurrentTimeConst.set_cmdline_option("ectime");

  append_member(EddyCurrentAmpl,"EddyCurrentAmpl");
  append_member(EddyCurrentTimeConst,"EddyCurrentTimeConst");
}


SeqSimulationOpts::SeqSimulationOpts()
 : JcampDxBlock("Simulation Options"), transm_coil(0), receiv_coil(0), coil_cache_up2date(false) {

  JcampDxBlock::set_embedded(true);

  SimThreads=numof_cores();
  SimThreads.set_minmaxval(1,16);
  SimThreads.set_description("Number of concurrent threads (parallel processing) during simulation");
  SimThreads.set_cmdline_option("nthreads");

  IntraVoxelMagnGrads=true;
  IntraVoxelMagnGrads.set_description("Consider intra-voxel magnetization gradients during simulation");
  IntraVoxelMagnGrads.set_cmdline_option("magsi");

  MagnMonitor=false;
  MagnMonitor.set_description("Monitor magnetization vector using vtk");
  MagnMonitor.set_cmdline_option("mon");

  ReceiverNoise=0.0;
  ReceiverNoise.set_minmaxval(0.0,10.0);
  ReceiverNoise.set_unit("%");
  ReceiverNoise.set_description("Noise generated by the receiver.");
  ReceiverNoise.set_cmdline_option("noise");

  TransmitterCoil.set_suffix("coi");
  TransmitterCoil.set_description("RF coil used for transmission. Leave blank for homogeneous coil.");
  TransmitterCoil.set_cmdline_option("tcoil");

  ReceiverCoil.set_suffix("coi");
  ReceiverCoil.set_description("RF coil used for acquisition. Leave blank for homogeneous coil.");
  ReceiverCoil.set_cmdline_option("rcoil");

  InitialMagnVector[0]=0.0;
  InitialMagnVector[1]=0.0;
  InitialMagnVector[2]=1.0;
  InitialMagnVector.set_description("Initial magnetization vector.");


#ifndef NO_THREADS
  append_member(SimThreads,"SimThreads");
#endif
  append_member(IntraVoxelMagnGrads,"IntraVoxelMagnGrads");
  append_member(MagnMonitor,"MagnMonitor");
  append_member(ReceiverNoise,"ReceiverNoise");
  append_member(TransmitterCoil,"TransmitterCoil");
  append_member(ReceiverCoil,"ReceiverCoil");
  append_member(InitialMagnVector,"InitialMagnVector");

}

void SeqSimulationOpts::update_coil_cache() const {
  if(coil_cache_up2date) return;
  outdate_coil_cache();
  if(filesize(TransmitterCoil.c_str())>0) {
    transm_coil=new CoilSensitivity("Transmitter Coil");
    if(transm_coil->load(TransmitterCoil)<=0) {
      delete transm_coil; transm_coil=0;
    } else SeqMethodProxy()->get_systemInfo().set_transmit_coil_name(TransmitterCoil.get_basename()); // update protocol
  }
  if(filesize(ReceiverCoil.c_str())>0) {
    receiv_coil=new CoilSensitivity("Receiver Coil");
    if(receiv_coil->load(ReceiverCoil)<=0) {
      delete receiv_coil; receiv_coil=0;
    }
  }
  coil_cache_up2date=true;
}


void SeqSimulationOpts::outdate_coil_cache() const {
  if(transm_coil) delete transm_coil; transm_coil=0;
  if(receiv_coil) delete receiv_coil; receiv_coil=0;
  coil_cache_up2date=false;
}


///////////////////////////////////////////////////////////////

STD_ostream& operator << (STD_ostream& os, const SeqPlotCurve& spc) {
  os << "---------------------------------------------" << STD_endl;
  os << "label=" << spc.label << "  ";
  os << "channel=" << spc.channel << "  ";
  os << "spikes=" << spc.spikes << "  ";
  os << STD_endl;
  
  for(unsigned int i=0; i<spc.x.size(); i++) {
    os << "y[" << i << "](" << spc.x[i] << ")=" << spc.y[i] << STD_endl;
  }

  if(spc.marklabel) os << "marker=" << spc.marklabel << "/" << spc.marker << "/" << spc.marker_x << STD_endl;

  return os;
}


///////////////////////////////////////////////////////////////

bool SeqPlotCurveRef::contains_timepoint(double timep) const {
  unsigned int curvesize=ptr->x.size();
  if(!curvesize) return false;

  double curvestart=start+ptr->x[0];
  double curveend  =start+ptr->x[curvesize-1];

  return( (curvestart<=timep) && (timep<=curveend) );
}


double SeqPlotCurveRef::interpolate_timepoint(double timep) const {
  double result=0.0;
  
  unsigned int curvesize=ptr->x.size();

  for(unsigned int i=0; i<(curvesize-1); i++) {

    double curr_x=start+ptr->x[i];
    double next_x=start+ptr->x[i+1];

    double curr_y=ptr->y[i];
    double next_y=ptr->y[i+1];

    if(timep==curr_x) {
      if(curr_x==next_x) result=STD_max(curr_y,next_y);
      else result=curr_y;
      break;
    }

    if(timep==next_x) {
      result=next_y;
      break;
    }

    if(!ptr->spikes && curr_x<timep && timep<next_x ) {
      double m= secureDivision( next_y-curr_y , next_x-curr_x );
      result= m * ( timep-curr_x ) + curr_y;
      break;
    }
  }

  return result;
}

void SeqPlotCurveRef::copy_to_syncpoint(SeqPlotSyncPoint& sp, double value) const {

  plotChannel src_channel=ptr->channel;
  bool src_is_gradchan= ( (src_channel>=Gread_plotchan) && (src_channel<=Gslice_plotchan) );

  if(src_is_gradchan && gradmatrix) {
    for(int target_channel=Gread_plotchan; target_channel<=Gslice_plotchan; target_channel++) {
      double rotfactor=(*gradmatrix)[target_channel-Gread_plotchan][src_channel-Gread_plotchan];
      sp.val[target_channel]+=rotfactor*value;
    }
  } else {
    sp.val[src_channel]+=value;
  }
    
  if(has_freq_phase) {
    sp.val[freq_plotchan]=freq;
    sp.val[phase_plotchan]=phase;
  }

}  



/////////////////////////////////////////////////////////////////////////////////////////////

struct CurvePointRef {
  CurvePointRef() : curveref(0), index(0) {}
  const SeqPlotCurveRef* curveref;
  int index;
};

/////////////////////////////////////////////////////////////////////////////////////////////

struct FrameTimepoint {
  FrameTimepoint(double timep, plotChannel curvechannel, const SeqPlotCurveRef* curve, int curveindex)
    : t(timep), markerref(0) {
    curve_point[curvechannel].curveref=curve;
    curve_point[curvechannel].index=curveindex;
  }

  FrameTimepoint(double timep, const SeqPlotCurveRef* marker) : t(timep), markerref(marker) {}
  

  double t;
  CurvePointRef curve_point[numof_plotchan];
  const SeqPlotCurveRef* markerref;

  bool contains_curve(const SeqPlotCurveRef* curve) const {
    for(int ichan=0; ichan<numof_plotchan; ichan++) {
      if(curve_point[ichan].curveref==curve) return true;
    }
    return false;
  }
        
  bool merge(const FrameTimepoint& tp2) {
    if(t!=tp2.t) return false;

    FrameTimepoint new_frame_timepoint=(*this);
    
    if(tp2.markerref) {
      if(new_frame_timepoint.markerref) return false;
      else new_frame_timepoint.markerref=tp2.markerref;
    }
    
    for(int ichan=0; ichan<numof_plotchan; ichan++) {
      if(tp2.curve_point[ichan].curveref) {
        if(new_frame_timepoint.curve_point[ichan].curveref) {
          return false;
        } else new_frame_timepoint.curve_point[ichan]=tp2.curve_point[ichan];
      }
    }

    (*this)=new_frame_timepoint;
    return true;
  }
            
  bool operator == (const FrameTimepoint& ft2) const {return (t==ft2.t);}
  bool operator <  (const FrameTimepoint& ft2) const {return (t<ft2.t);}
};


/////////////////////////////////////////////////////////////////////////////////////////////


void SeqPlotFrame::append_syncpoints(STD_list<SeqPlotSyncPoint>& synclist, double framestart) const {
  STD_list<SeqPlotCurveRef>::const_iterator refit;
  STD_list<FrameTimepoint>::const_iterator tpit;
  int ichan;
  
  STD_list<FrameTimepoint> timepoints_in_frame;


  // create a list of all timepoints and the corresponding curves
  for(refit=begin(); refit!=end(); ++refit) {
    const SeqPlotCurve& plotcurve=*(refit->ptr);

    unsigned int curvesize=plotcurve.x.size();
    for(unsigned int i=0; i<curvesize; i++) {
      double curve_timepoint_in_frame=refit->start+plotcurve.x[i];
      timepoints_in_frame.push_back(FrameTimepoint(curve_timepoint_in_frame, plotcurve.channel, &(*refit),i));
    }

    if(plotcurve.marker!=no_marker) {
      double marker_timepoint_in_frame=refit->start+plotcurve.marker_x;
      timepoints_in_frame.push_back(FrameTimepoint(marker_timepoint_in_frame, &(*refit)));
    }

  }

  timepoints_in_frame.sort();

  STD_list<FrameTimepoint> merged_timepoints_in_frame;


  // merge timepoints with the same time
  tpit=timepoints_in_frame.begin();
  while(tpit!=timepoints_in_frame.end()) {
    FrameTimepoint ft=(*tpit);
    ++tpit;
    while(tpit!=timepoints_in_frame.end() && ft.merge(*tpit)) ++tpit;
    merged_timepoints_in_frame.push_back(ft);
  }
  
  

  // create syncpoints from the list of timepoints
  for(tpit=merged_timepoints_in_frame.begin(); tpit!=merged_timepoints_in_frame.end(); ++tpit) {

    double syncpoint_timepoint_in_frame=tpit->t;
    SeqPlotSyncPoint sp(framestart+syncpoint_timepoint_in_frame);

    // deal with curves that are explicitely part of the TimePoint
    for(ichan=0; ichan<numof_plotchan; ichan++) {
      const SeqPlotCurveRef* curveref=tpit->curve_point[ichan].curveref;
      if(curveref) {
        int curveindex=tpit->curve_point[ichan].index;
        double value=curveref->ptr->y[curveindex];
        curveref->copy_to_syncpoint(sp,value);
      }
    }
    
    // interpolate remaining curves
    for(refit=begin(); refit!=end(); ++refit) {
      if(!tpit->contains_curve(&(*refit))) {
        if(refit->contains_timepoint(syncpoint_timepoint_in_frame)) {
          double value=refit->interpolate_timepoint(syncpoint_timepoint_in_frame);
          refit->copy_to_syncpoint(sp,value);
        }
      }
    }

    // deal with marker
    if(tpit->markerref) {
      sp.marker=tpit->markerref->ptr->marker;
      sp.marklabel=tpit->markerref->ptr->label;
    }

    synclist.push_back(sp);
    
  }
}


double SeqPlotFrame::get_latest_point() const {
  double result=0.0;
  for(STD_list<SeqPlotCurveRef>::const_iterator refit=begin(); refit!=end(); ++refit) {
    const SeqPlotCurve& plotcurve=*(refit->ptr);
    unsigned int curvesize=plotcurve.x.size();
    if(curvesize) {
      double latest=refit->start+plotcurve.x[curvesize-1];
      if(latest>result) result=latest;
    }
  }
  return result;
}

/////////////////////////////////////////////////////////////////

void SeqTimecourse::allocate(unsigned int allocsize) {
  size=allocsize;
  x=new double[allocsize];
  for(int ichan=0; ichan<numof_plotchan; ichan++) y[ichan]=new double[allocsize];
}



SeqTimecourse::SeqTimecourse(const STD_list<SeqPlotSyncPoint>& synclist, const SeqTimecourse* eddy_tcourse, ProgressMeter* progmeter)
 : signal_x(0), signal_y(0), noisegen(0) {
  Log<SeqStandAlone> odinlog("SeqTimecourse","");

  allocate(synclist.size());

  if(eddy_tcourse) ODINLOG(odinlog,normalDebug) << "eddy_tcourse->size=" << eddy_tcourse->size << STD_endl;
  else ODINLOG(odinlog,normalDebug) << "no eddy_tcourse" << STD_endl;
  
  unsigned int index=0;
  int ichan;
  for(STD_list<SeqPlotSyncPoint>::const_iterator syncit=synclist.begin(); syncit!=synclist.end(); ++syncit) {
    x[index]=syncit->timep;
    for(ichan=0; ichan<numof_plotchan; ichan++) {
      y[ichan][index]=syncit->val[ichan];
      if(eddy_tcourse && ichan>=Gread_plotchan && ichan<=Gslice_plotchan) {
        y[ichan][index]+=eddy_tcourse->y[ichan][index];
      }
    }
    if(syncit->val[rec_plotchan]>0.0) n_rec_points++;
    if(progmeter) progmeter->increase_counter();
    index++;
  }
  create_marker_values(synclist,progmeter);
}



void SeqTimecourse::create_marker_values(const STD_list<SeqPlotSyncPoint>& synclist, ProgressMeter* progmeter) {
  markers.clear();
  unsigned int index=0;
  for(STD_list<SeqPlotSyncPoint>::const_iterator syncit=synclist.begin(); syncit!=synclist.end(); ++syncit) {
    if(syncit->marker!=no_marker) {
      TimecourseMarker4Qwt tm4q;
      tm4q.x=x[index];
      for(int ichan=0; ichan<numof_plotchan; ichan++) tm4q.y[ichan]=y[ichan][index];
      tm4q.type=syncit->marker;
      markers.push_back(tm4q);
    }
    index++;
    if(progmeter) progmeter->refresh_display();
  }
  markers.reset_subcache();
}
  

SeqTimecourse::~SeqTimecourse() {
  if(x) delete x;
  for(int i=0; i<numof_plotchan; i++) if(y[i]) delete y[i];
  if(signal_x) delete[] signal_x;
  if(signal_y) delete[] signal_y;
  if(noisegen) delete noisegen;
}


unsigned int SeqTimecourse::get_index(double timepoint) const {
  unsigned int result=0;

  unsigned int n_hash_steps=size/TIMECOURSE_HASH_INTERVAL;
  for(unsigned int ihash=0; ihash<n_hash_steps; ihash++) {
    result=ihash*TIMECOURSE_HASH_INTERVAL;
    if(x[result]>timepoint) break;
  }

  if(x[result]>timepoint) {
    while(result!=0   && x[result]>timepoint) result--;
  } else {
    while(result<size && x[result]<timepoint) result++;
  }

  return result;
}


  
const SeqTimecourseData* SeqTimecourse::get_subtimecourse(double starttime, double endtime) const {
  Log<SeqStandAlone> odinlog("SeqTimecourse","get_subtimecourse");
  static SeqTimecourseData result;
  if(!size) return &result;

  unsigned int beginindex=get_index(starttime);
  unsigned int endindex=get_index(endtime);

  if(beginindex>EXTRA_TIMECOURSE_POINTS)      beginindex-=EXTRA_TIMECOURSE_POINTS; else beginindex=0;
  if(endindex<(size-EXTRA_TIMECOURSE_POINTS)) endindex  +=EXTRA_TIMECOURSE_POINTS; else endindex=size-1;

  result.size=endindex-beginindex;
  result.x=&(x[beginindex]);
  for(int i=0; i<numof_plotchan; i++) result.y[i]=&(y[i][beginindex]);

  ODINLOG(odinlog,normalDebug) << "result.size=" << result.size << STD_endl;

  return &result;
}


void SeqTimecourse::get_markers(STD_list<TimecourseMarker4Qwt>::const_iterator& result_begin, STD_list<TimecourseMarker4Qwt>::const_iterator& result_end, double starttime, double endtime  ) const {
  markers.get_sublist(result_begin, result_end, starttime, endtime);
}




bool SeqTimecourse::simulate(const STD_list<SeqPlotSyncPoint>& synclist, const STD_string& fidfile, const STD_string& samplefile, const SeqSimulationOpts& opts, ProgressMeter* progmeter, SeqSimFeedbackAbstract* feedback, const SeqPlotDataAbstract* plotdata) const {
  Log<SeqStandAlone> odinlog("SeqTimecourse","simulate");

  SeqMethodProxy method;
  unsigned int i;

  SeqSimMagsi mag("Magnetization");
  SeqSimMagsi magplot("Magnetization4Plot");

  mag.set_initial_vector(opts.InitialMagnVector[0], opts.InitialMagnVector[1], opts.InitialMagnVector[2]);

  Sample sample("Sample",false);
  if(sample.load(samplefile)<0) return false;

  Sample sampleplot("Sample4Plot",true,true);

  mag.set_numof_threads(opts.SimThreads);
  mag.set_intravoxel_simulation(opts.IntraVoxelMagnGrads);

  magplot.set_intravoxel_simulation(opts.IntraVoxelMagnGrads); // use only one thread by now

  mag.set_spat_rotmatrix(method->get_geometry().get_gradrotmatrix(true));

  Nuclei nuc;
  float gamma=nuc.get_gamma(method->get_main_nucleus());


  unsigned int ichan,nchan=1;
  CoilSensitivity* rec_coil=opts.get_receiv_coil();
  if(rec_coil) nchan=rec_coil->get_numof_channels();

  signal_label.resize(nchan);
  for(ichan=0; ichan<nchan; ichan++) signal_label[ichan]="signal_channel"+itos(ichan);

  cvector* acqresult=new cvector[nchan];
  for(ichan=0; ichan<nchan; ichan++) acqresult[ichan].resize(n_rec_points);
  unsigned int acqresult_counter=0;
  rmfile(fidfile.c_str());

  if(signal_x) delete[] signal_x; signal_x=new double[n_rec_points];

  if(signal_y) {
    for(ichan=0; ichan<signal_nchan; ichan++) delete[] signal_y[ichan];
    delete[] signal_y;
  }
  signal_nchan=nchan;
  signal_y=new double*[signal_nchan];
  for(ichan=0; ichan<signal_nchan; ichan++) signal_y[ichan]=new double[n_rec_points];

  unsigned int recindex=0;

  STD_list<unsigned int> endacqindices;

  // prepare noise generator
  double noisefactor=0.01*opts.ReceiverNoise;
  float maxnoise=0.0;
  if(noisefactor>0.0) {
    if(!noisegen) noisegen=new RandomDist();
    float maxsd=sample.get_spinDensity().maxvalue();
    float maxnoise=noisefactor*maxsd*mag.get_total_size();
    ODINLOG(odinlog,normalDebug) << "maxnoise=" << maxnoise << STD_endl;
  }

  // create cache for simulation
  mag.prepare_simulation(sample,opts.get_transm_coil(),opts.get_receiv_coil(),progmeter);
  magplot.prepare_simulation(sampleplot);


  progmeter->new_task(size,"Simulating NMR Signal");

  cvector signal;

  Profiler* prof=new Profiler("simulate");

  // iterate through list of syncpoints
  i=0;
  for(STD_list<SeqPlotSyncPoint>::const_iterator syncit=synclist.begin(); syncit!=synclist.end(); ++syncit) {
    y[signal_plotchan][i]=0.0;

    if(i) {
      SeqSimInterval simvals;
      simvals.dt=x[i]-x[i-1];
//      simvals.t=0.5*(x[i]+x[i-1]);
      double B1re= 0.5*(y[B1re_plotchan][i] +y[B1re_plotchan][i-1]);
      double B1im= 0.5*(y[B1im_plotchan][i] +y[B1im_plotchan][i-1]);
      simvals.B1=STD_complex(B1re,B1im);
      simvals.freq= 0.5*(y[freq_plotchan][i]  +y[freq_plotchan][i-1]);
      simvals.phase=0.5*(y[phase_plotchan][i] +y[phase_plotchan][i-1]);
      simvals.Gx=   0.5*(y[Gread_plotchan][i] +y[Gread_plotchan][i-1]);
      simvals.Gy=   0.5*(y[Gphase_plotchan][i]+y[Gphase_plotchan][i-1]);
      simvals.Gz=   0.5*(y[Gslice_plotchan][i]+y[Gslice_plotchan][i-1]);
      simvals.rec=y[rec_plotchan][i]; // acquire at end of simulation interval


      signal=mag.simulate(simvals,gamma);

      if(feedback) {

        float Mvec[3];
        float dM[3];

        double max_dt_fields=0.050;
        if( (cabs(simvals.B1) || simvals.Gx || simvals.Gy || simvals.Gz) &&  (simvals.dt>max_dt_fields) ) {
          unsigned int n_sub_dt=(unsigned int)(simvals.dt/max_dt_fields+0.5);
          simvals.dt=secureDivision(simvals.dt,n_sub_dt);
          ODINLOG(odinlog,normalDebug) << "splitting simulation into " << n_sub_dt << " sub intervals at t=" << x[i] << STD_endl;
          for(unsigned int isub=0; isub<n_sub_dt; isub++) {
            magplot.simulate(simvals,gamma);
            Mvec[0]=magplot.get_Mx()[0];
            Mvec[1]=magplot.get_My()[0];
            Mvec[2]=magplot.get_Mz()[0];
            if(opts.IntraVoxelMagnGrads) {
              dM[0]=magplot.dMx[0][0];
              dM[1]=magplot.dMy[0][0];
              dM[2]=magplot.dMz[0][0];
              feedback->plot_vector(x[i-1]+isub*simvals.dt,Mvec,dM);
            } else feedback->plot_vector(x[i-1]+isub*simvals.dt,Mvec,0);
            progmeter->refresh_display();
          }
        } else {
          magplot.simulate(simvals,gamma);
          Mvec[0]=magplot.get_Mx()[0];
          Mvec[1]=magplot.get_My()[0];
          Mvec[2]=magplot.get_Mz()[0];
          float timestamp=0.5*(x[i]+x[i-1]);
          if(opts.IntraVoxelMagnGrads) {
            dM[0]=magplot.dMx[0][0];
            dM[1]=magplot.dMy[0][0];
            dM[2]=magplot.dMz[0][0];
            feedback->plot_vector(timestamp,Mvec,dM);
          } feedback->plot_vector(timestamp,Mvec,0);
        }
      }


      if(simvals.rec>0.0) {

        // add noise to the signal
        if(maxnoise>0.0) {
         float stdev_noise=1.0e-3*maxnoise/simvals.dt; // arbitrary scale for noise
         for(ichan=0; ichan<nchan; ichan++) {
            float noise_re=noisegen->gaussian(stdev_noise);
            float noise_im=noisegen->gaussian(stdev_noise);
            signal[ichan]+=STD_complex(noise_re,noise_im);
          }
        }

        signal_x[recindex]=x[i];
        y[signal_plotchan][i]=0.0;
        for(ichan=0; ichan<nchan; ichan++) {
          y[signal_plotchan][i]+=cabs(signal[ichan]);
          acqresult[ichan][acqresult_counter]=signal[ichan];
          signal_y[ichan][recindex]=cabs(signal[ichan]);
        }
        acqresult_counter++;
        recindex++;
      }

      // memorize end-of-acq position and dump signal to disk
      if(syncit->marker==endacq_marker) {
        endacqindices.push_back(recindex);

        for(ichan=0; ichan<nchan; ichan++) acqresult[ichan].write(fidfile,appendMode,acqresult_counter);
        acqresult_counter=0;
      }


      // create snapshot of magnetization
      if(syncit->marker==snapshot_marker) {
        STD_string snap_fname(syncit->marklabel);
        if(snap_fname!="") {
          unsigned int totalsize=mag.get_total_size();
          ODINLOG(odinlog,normalDebug) << "dumping snapshot at " << x[i] << ODIN_TIME_UNIT << " of size " << totalsize << " to file " << syncit->marklabel << STD_endl;

          fvector dumpmagn(3*totalsize);
          fvector data;
          unsigned int offset=0;
          unsigned int i;
          data=mag.get_Mx();
          for(i=0; i<data.size(); i++) dumpmagn[offset+i]=data[i];
          offset+=data.size();
          data=mag.get_My();
          for(i=0; i<data.size(); i++) dumpmagn[offset+i]=data[i];
          offset+=data.size();
          data=mag.get_Mz();
          for(i=0; i<data.size(); i++) dumpmagn[offset+i]=data[i];
          dumpmagn.write(snap_fname,appendMode);
        }
      }
    }
    progmeter->increase_counter();
    i++;
  }
  mag.finalize_simulation();


  // transfer signal curves to plotting window
  unsigned int last_endof_acq=0;
  for(STD_list<unsigned int>::const_iterator it=endacqindices.begin(); it!=endacqindices.end(); ++it) {
    unsigned int acqwindow_size=(*it)-last_endof_acq;
    unsigned int acqwindow_start=last_endof_acq;
    ODINLOG(odinlog,normalDebug) << "endof_acq n_rec_points/start/size=" << n_rec_points << "/" << acqwindow_start << "/" << acqwindow_size << STD_endl;

    for(ichan=0; ichan<nchan; ichan++) {
      Curve4Qwt signal_curve;
      signal_curve.label=signal_label[ichan].c_str();
      signal_curve.channel=signal_plotchan;
      signal_curve.size=acqwindow_size;
      signal_curve.x=&(signal_x[acqwindow_start]);
      signal_curve.y=&(signal_y[ichan][acqwindow_start]);
      plotdata->add_signal_curve(signal_curve);
    }
    last_endof_acq=(*it);
  }

  delete prof;

  delete[] acqresult;

  return true;
}



/////////////////////////////////////////////////////////////////

template<int Nth_moment, bool ConstGrad>
SeqGradMomentTimecourse<Nth_moment,ConstGrad>::SeqGradMomentTimecourse(const STD_list<SeqPlotSyncPoint>& synclist, const SeqTimecourse& plain_tcourse, const STD_string& nucleus, ProgressMeter* progmeter)
 : SeqTimecourse(plain_tcourse) {
  allocate(size); // allocate new arrays

  int igrad,ichan;

  Nuclei n;
  double gamma=n.get_gamma(nucleus);

  double tau[3];
  for(igrad=0; igrad<3; igrad++) tau[igrad]=0.0;

  double moment_integral[3];
  for(igrad=0; igrad<3; igrad++) moment_integral[igrad]=0.0;

  unsigned int index=0;

  for(STD_list<SeqPlotSyncPoint>::const_iterator syncit=synclist.begin(); syncit!=synclist.end(); ++syncit) {
    x[index]=plain_tcourse.x[index];

    double x1=x[index];
    double x0=0.0;
    if(index) x0=x[index-1];
    double dx=x1-x0;

    bool active_transverse=true;

    for(ichan=0; ichan<numof_plotchan; ichan++) {
      y[ichan][index]=plain_tcourse.y[ichan][index];

      if(ichan>=Gread_plotchan && ichan<=Gslice_plotchan) {

        int gradchan=ichan-Gread_plotchan;

        if(active_transverse) {

          double y0=0.0;
          double y1=0.0;

          if(ConstGrad) {
            y0=1.0;
            y1=1.0;
          } else {
            y1=plain_tcourse.y[ichan][index];
            if(index) y0=plain_tcourse.y[ichan][index-1];
          }

          double dy=y1-y0;

          double m=secureDivision(dy,dx);

          double x0_tau=tau[gradchan];
          double x1_tau=tau[gradchan]+dx;


          double moment_increment=      m/double(Nth_moment+2)*(pow(x1_tau,Nth_moment+2)-pow(x0_tau,Nth_moment+2));
          moment_increment+=(y0-m*x0_tau)/double(Nth_moment+1)*(pow(x1_tau,Nth_moment+1)-pow(x0_tau,Nth_moment+1));

          moment_integral[gradchan]+=gamma*moment_increment;
        }


        if(syncit->marker==excitation_marker) { // reset
          moment_integral[gradchan]=0.0;
          tau[gradchan]=0.0;
          active_transverse=true;
        }

        if(syncit->marker==refocusing_marker || syncit->marker==recallMagn_marker) { // invert phase
          moment_integral[gradchan]=-moment_integral[gradchan];
          active_transverse=true;
        }

        if(syncit->marker==storeMagn_marker) { // disable accumulation of integral
          active_transverse=false;
        }

        y[ichan][index]=moment_integral[gradchan];

        tau[gradchan]+=dx;
      }
    }
    if(progmeter) progmeter->increase_counter();
    index++;
  }
  create_marker_values(synclist,progmeter);
}

////////////////////////////////////////////////////////////

SeqTwoFuncIntegralTimecourse::SeqTwoFuncIntegralTimecourse(const STD_list<SeqPlotSyncPoint>& synclist, const SeqTimecourse& ka_tcourse, const SeqTimecourse& kb_tcourse, ProgressMeter* progmeter)
 : SeqTimecourse(ka_tcourse) {
  allocate(size); // allocate new arrays

  int igrad,ichan;

  double integral[3];
  for(igrad=0; igrad<3; igrad++) integral[igrad]=0.0;


  unsigned int index=0;
  for(STD_list<SeqPlotSyncPoint>::const_iterator syncit=synclist.begin(); syncit!=synclist.end(); ++syncit) {
    x[index]=ka_tcourse.x[index];

    double x1=x[index];
    double x0=0.0;
    if(index) x0=x[index-1];
    double dx=x1-x0;

    for(ichan=0; ichan<numof_plotchan; ichan++) {
      y[ichan][index]=ka_tcourse.y[ichan][index]; // default: copy all channels

      if(ichan>=Gread_plotchan && ichan<=Gslice_plotchan) {

        int gradchan=ichan-Gread_plotchan;

        double ka1=ka_tcourse.y[ichan][index];
        double kb1=kb_tcourse.y[ichan][index];

        double ka0=0.0;
        double kb0=0.0;

        if(index) {
          ka0=ka_tcourse.y[ichan][index-1];
          kb0=kb_tcourse.y[ichan][index-1];
        }

        double dka=ka1-ka0;
        double dkb=kb1-kb0;

        double incr = ( (6.0*dx*ka0 + 3.0*dka*dx)*kb0 + 3.0*dkb*dx*ka0 + 2.0*dka*dkb*dx) / 6.0; // integrate((ka0+x/dx*dka)*(kb0+x/dx*dkb),x,0,dx);

//        double incr = dx * ( k0*k0 + k0*dk + dk*dk/3.0 ); // integrate((kval_last+t/dx*dk)^2,t,0,dx);

        integral[gradchan]+= incr;

        y[ichan][index]=integral[gradchan];

        if(syncit->marker==excitation_marker) {
          integral[gradchan]=0.0;
        }
      }
    }
    if(progmeter) progmeter->increase_counter();
    index++;
  }
  create_marker_values(synclist,progmeter);
}

////////////////////////////////////////////////////////////

SeqSlewRateTimecourse::SeqSlewRateTimecourse(const STD_list<SeqPlotSyncPoint>& synclist, const SeqTimecourse& plain_tcourse, ProgressMeter* progmeter)
 : SeqTimecourse(plain_tcourse) {
  allocate(size); // allocate new arrays

  int ichan;

  double maxslewrate=SeqMethodProxy()->get_systemInfo().get_max_slew_rate();

  unsigned int index=0;
  for(STD_list<SeqPlotSyncPoint>::const_iterator syncit=synclist.begin(); syncit!=synclist.end(); ++syncit) {
    x[index]=plain_tcourse.x[index];

    double x1=x[index];
    double x0=0.0;
    if(index) x0=x[index-1];
    double dx=x1-x0;

    for(ichan=0; ichan<numof_plotchan; ichan++) {
      y[ichan][index]=plain_tcourse.y[ichan][index];

      if(ichan>=Gread_plotchan && ichan<=Gslice_plotchan) {

        double y1=plain_tcourse.y[ichan][index];
        double y0=0.0;
        if(index) y0=plain_tcourse.y[ichan][index-1];
        double dy=y1-y0;

        double sr=secureDivision(dy,dx);
        if(fabs(sr)>maxslewrate) {
          double sign=secureDivision(sr,fabs(sr));
          sr=sign*maxslewrate;
        }
        y[ichan][index]=sr;

      }
    }
    if(progmeter) progmeter->increase_counter();
    index++;
  }
  create_marker_values(synclist,progmeter);
}

////////////////////////////////////////////////////////////

SeqEddyCurrentTimecourse::SeqEddyCurrentTimecourse(const STD_list<SeqPlotSyncPoint>& synclist, const SeqTimecourse& slew_rate_tcourse, const SeqTimecourseOpts& opts, ProgressMeter* progmeter)
 : SeqTimecourse(slew_rate_tcourse) {
  Log<SeqStandAlone> odinlog("SeqEddyCurrentTimecourse","");
  allocate(size); // allocate new arrays

  double ec_ampl=opts.EddyCurrentAmpl/100.0;
  double ec_time=opts.EddyCurrentTimeConst;
  ODINLOG(odinlog,normalDebug) <<  "ec_time/ec_ampl=" << ec_time << "/" << ec_ampl << STD_endl;

  int ichan;

  for(unsigned int index=0; index<size; index++) {
    x[index]=slew_rate_tcourse.x[index];

    for(ichan=0; ichan<numof_plotchan; ichan++) {
      y[ichan][index]=slew_rate_tcourse.y[ichan][index];
      if(ichan>=Gread_plotchan && ichan<=Gslice_plotchan) {

        double x1=x[index];
        double x0=0.0;
        if(index) x0=x[index-1];
        double dx=x1-x0;

        // eddy currents induced by the previous interval
        double additional_eddy_currents = - ec_ampl * slew_rate_tcourse.y[ichan][index] * dx;
        double decay_factor = exp(-dx/ec_time);

        y[ichan][index]=0.0;
        if(index) y[ichan][index]=decay_factor * y[ichan][index-1] + additional_eddy_currents;
      }
    }
    if(progmeter) progmeter->increase_counter();
  }
  create_marker_values(synclist,progmeter);
}

////////////////////////////////////////////////////////////

SeqPlotData::SeqPlotData(const STD_string& objlabel) : Labeled(objlabel.c_str()),
  curves4qwt_cache_done(false), markers4qwt_cache_done(false), synclist_cache_done(false) {
  for(int itc=0; itc<numof_tcmodes; itc++) timecourse_cache[itc]=0;
}


void SeqPlotData::flush_frame(double framedur) {
  Log<SeqStandAlone> odinlog("SeqPlotData","flush_frame");

  double latest=current_frame.get_latest_point();
  double framedur_total=framestart_offset+framedur;
  double diff=latest-framedur_total;
  ODINLOG(odinlog,normalDebug) << "latest/framedur_total/diff=" << latest << "/" << framedur_total << "/" << diff << STD_endl;

  if(diff>1e-6) { // check whether there are remaining points leaping into next frame
    framestart_offset+=framedur;
  } else {
    unsigned int framesize=current_frame.size();
    ODINLOG(odinlog,normalDebug) << "appending frame with size=" << framesize << STD_endl;
    if(framesize || framedur_total>0.0) {
      current_frame.frameduration=framedur_total;
      push_back(current_frame);
    }
    current_frame.clear();
    framestart_offset=0.0;
  }
}

void SeqPlotData::reset() {
  Log<SeqStandAlone> odinlog("SeqPlotData","reset");
  ODINLOG(odinlog,normalDebug) << "mem(pre )=" << Profiler::get_memory_usage() << STD_endl;
  STD_list<SeqPlotFrame>::clear();
  ODINLOG(odinlog,normalDebug) << "size()=" << size() << STD_endl;
  current_frame.clear();
  signal_curves.reset();
  framestart_offset=0.0;
  clear_curves4qwt_cache();
  curves4qwt_cache_done=false;
  clear_markers4qwt_cache();
  markers4qwt_cache_done=false;
  clear_synclist_cache();
  for(int itc=0; itc<numof_tcmodes; itc++) clear_timecourse_cache(timecourseMode(itc));
  ODINLOG(odinlog,normalDebug) << "mem(post)=" << Profiler::get_memory_usage() << STD_endl;
}

void SeqPlotData::add_signal_curve(const Curve4Qwt& signal_curve) const {
  Log<SeqStandAlone> odinlog("SeqPlotData","add_signal_curve");
  ODINLOG(odinlog,normalDebug) << "signal_curves.size()/signal_curve.size=" << signal_curves.size() << "/" << signal_curve.size << STD_endl;
  signal_curves.push_back(signal_curve);
}


void SeqPlotData::clear_curves4qwt_cache() const {
  STD_list<Curve4Qwt>::iterator qwtit;
  for(qwtit=curves4qwt_cache.begin(); qwtit!=curves4qwt_cache.end(); ++qwtit) {
    if(qwtit->x) delete[] qwtit->x;
    if(qwtit->y) delete[] qwtit->y;
  }
  curves4qwt_cache.clear();
  for(qwtit=curves4qwt_cache_lowres.begin(); qwtit!=curves4qwt_cache_lowres.end(); ++qwtit) {
    if(qwtit->x) delete[] qwtit->x;
    if(qwtit->y) delete[] qwtit->y;
  }
  curves4qwt_cache_lowres.clear();
}

void SeqPlotData::clear_markers4qwt_cache() const {
  markers4qwt_cache.clear();
}

void SeqPlotData::clear_synclist_cache() const {
  synclist_cache.clear();
  synclist_cache_done=false;
}

void SeqPlotData::clear_timecourse_cache(timecourseMode type) const {
  Log<SeqStandAlone> odinlog("SeqPlotData","clear_timecourse_cache");
  ODINLOG(odinlog,normalDebug) << "timecourse_cache[" << type << "]=" << timecourse_cache[type] << STD_endl;
  if(timecourse_cache[type]) delete timecourse_cache[type];
  timecourse_cache[type]=0;
}


void SeqPlotData::create_curves4qwt_cache() const {
  if(curves4qwt_cache_done) return;
  clear_curves4qwt_cache();

  double framestart=0.0;
  int i;

  for(int ichan=0; ichan<numof_plotchan; ichan++) has_curves_on_channel_cache[ichan]=false; // reset

  for(STD_list<SeqPlotFrame>::const_iterator frameit=begin(); frameit!=end(); ++frameit) {

    for(STD_list<SeqPlotCurveRef>::const_iterator refit=frameit->begin(); refit!=frameit->end(); ++refit) {

      int curvesize=refit->ptr->x.size();
      if(curvesize) {

        Curve4Qwt c4q;
        c4q.label=refit->ptr->label;
        c4q.channel=refit->ptr->channel;
        c4q.size=curvesize+2; // create points with y=0 at beginning/end of curve
        c4q.spikes=refit->ptr->spikes;
        c4q.x=new double[c4q.size];
        c4q.y=new double[c4q.size];

        double min_y=0.0;
        double max_y=0.0;
        for(i=0; i<curvesize; i++) {
          c4q.x[i+1]=framestart+refit->start+refit->ptr->x[i];
          double yval=refit->ptr->y[i];
          c4q.y[i+1]=yval;
          if(yval<min_y) min_y=yval;
          if(yval>max_y) max_y=yval;
        }

        // points at beginning/end of curve
        c4q.x[0]=c4q.x[1];
        c4q.y[0]=0.0;
        c4q.x[c4q.size-1]=c4q.x[c4q.size-2];
        c4q.y[c4q.size-1]=0.0;

        c4q.has_freq_phase=refit->has_freq_phase;
        c4q.freq=refit->freq;
        c4q.phase=refit->phase;
        c4q.gradmatrix=refit->gradmatrix;
        curves4qwt_cache.push_back(c4q);

        STD_list<double> lowres_x;
        STD_list<double> lowres_y;
        lowres_x.push_back(c4q.x[0]);
        lowres_y.push_back(c4q.y[0]);
        for(i=1; i<(c4q.size-1); i++) {
          double last_y=c4q.y[i-1];
          double curr_y=c4q.y[i];
          double next_y=c4q.y[i+1];

          bool significant_point=false;
          if(last_y<=curr_y && next_y<curr_y) significant_point=true;
          if(last_y>=curr_y && next_y>curr_y) significant_point=true;
          if(last_y<curr_y && next_y<=curr_y) significant_point=true;
          if(last_y>curr_y && next_y>=curr_y) significant_point=true;

          if(significant_point) {
            lowres_x.push_back(c4q.x[i]);
            lowres_y.push_back(c4q.y[i]);
          }
        }
        lowres_x.push_back(c4q.x[c4q.size-1]);
        lowres_y.push_back(c4q.y[c4q.size-1]);

        Curve4Qwt c4q_lowres;
        c4q_lowres=c4q;
        c4q_lowres.size=lowres_x.size();
        c4q_lowres.x=new double[c4q_lowres.size];
        c4q_lowres.y=new double[c4q_lowres.size];
        i=0;
        STD_list<double>::const_iterator yit=lowres_y.begin();
        for(STD_list<double>::const_iterator xit=lowres_x.begin(); xit!=lowres_x.end(); ++xit) {
          c4q_lowres.x[i]=(*xit);
          c4q_lowres.y[i]=(*yit);
          i++; ++yit;
        }
        curves4qwt_cache_lowres.push_back(c4q_lowres);

        has_curves_on_channel_cache[c4q.channel]=true;
      }
    }
    framestart+=frameit->frameduration;
  }
  curves4qwt_cache.reset_subcache();
  curves4qwt_cache_lowres.reset_subcache();
  curves4qwt_cache_done=true;
}


void SeqPlotData::get_curves(STD_list<Curve4Qwt>::const_iterator& result_begin, STD_list<Curve4Qwt>::const_iterator& result_end, double starttime, double endtime, double max_highres_interval  ) const {
  Log<SeqStandAlone> odinlog("SeqPlotData","get_curves");
  create_curves4qwt_cache();
  double interval=endtime-starttime;
  if(interval>max_highres_interval) {
    curves4qwt_cache_lowres.get_sublist(result_begin, result_end, starttime, endtime);
  } else {
    curves4qwt_cache.get_sublist(result_begin, result_end, starttime, endtime);
  }
}

void SeqPlotData::get_signal_curves(STD_list<Curve4Qwt>::const_iterator& result_begin, STD_list<Curve4Qwt>::const_iterator& result_end, double starttime, double endtime  ) const {
  Log<SeqStandAlone> odinlog("SeqPlotData","get_signal_curves");
  ODINLOG(odinlog,normalDebug) << "size()=" << signal_curves.size() << STD_endl;
  signal_curves.get_sublist(result_begin, result_end, starttime, endtime);
}

void SeqPlotData::create_markers4qwt_cache() const {
  clear_markers4qwt_cache();
  double framestart=0.0;
  for(STD_list<SeqPlotFrame>::const_iterator frameit=begin(); frameit!=end(); ++frameit) {
    for(STD_list<SeqPlotCurveRef>::const_iterator refit=frameit->begin(); refit!=frameit->end(); ++refit) {
      if((refit->ptr->marker)!=no_marker) {
        Marker4Qwt m4q;
        m4q.label=refit->ptr->marklabel;
        m4q.x=framestart+refit->start+refit->ptr->marker_x;
        m4q.type=refit->ptr->marker;
        markers4qwt_cache.push_back(m4q);
      }
    }
    framestart+=frameit->frameduration;
  }
  markers4qwt_cache.reset_subcache();
  markers4qwt_cache_done=true;
}


void SeqPlotData::get_markers(STD_list<Marker4Qwt>::const_iterator& result_begin, STD_list<Marker4Qwt>::const_iterator& result_end, double starttime, double endtime  ) const {
  Log<SeqStandAlone> odinlog("SeqPlotData","get_markers");

  if(!markers4qwt_cache_done) create_markers4qwt_cache();
  markers4qwt_cache.get_sublist(result_begin, result_end, starttime, endtime);
}


void SeqPlotData::create_synclist_cache(ProgressMeter* progmeter) const {
  Log<SeqStandAlone> odinlog("SeqPlotData","create_synclist_cache");
  clear_synclist_cache();

  SeqPlotSyncPoint sp_start(0.0);
  synclist_cache.push_back(sp_start);

  SeqPlotSyncPoint sp_frameend(0.0);
  double framestart=0.0;
  for(STD_list<SeqPlotFrame>::const_iterator frameit=begin(); frameit!=end(); ++frameit) {
    frameit->append_syncpoints(synclist_cache,framestart);
    framestart+=frameit->frameduration;
    sp_frameend.timep=framestart;
    synclist_cache.push_back(sp_frameend);
    if(progmeter) progmeter->increase_counter();
  }

  SeqPlotSyncPoint sp_end(framestart);
  synclist_cache.push_back(sp_end);

  synclist_cache_done=true;
}

void SeqPlotData::create_timecourse_cache(timecourseMode type, const STD_string& nucleus, ProgressMeter* progmeter) const {
  Log<SeqStandAlone> odinlog("SeqPlotData","create_timecourse_cache");
  clear_timecourse_cache(type);

  SeqTimecourse* result=0; // exception safe

  unsigned int tcourse_size=synclist_cache.size();

  STD_string progtxt=STD_string("Creating ")+timecourseLabel[type]+" Timecourse";

  if(type==tcmode_plain) {
    create_timecourses(tcmode_eddy_currents,nucleus,progmeter);
    if(progmeter) progmeter->new_task(tcourse_size,progtxt.c_str());
    result=new SeqTimecourse(synclist_cache,timecourse_cache[tcmode_eddy_currents],progmeter);
  }

  if(type==tcmode_kspace) {
    create_timecourses(tcmode_plain,nucleus,progmeter);
    if(progmeter) progmeter->new_task(tcourse_size,progtxt.c_str());
    result=new SeqGradMomentTimecourse<0>(synclist_cache,*(timecourse_cache[tcmode_plain]),nucleus,progmeter);
  }

  if(type==tcmode_M1) {
    create_timecourses(tcmode_plain,nucleus,progmeter);
    if(progmeter) progmeter->new_task(tcourse_size,progtxt.c_str());
    result=new SeqGradMomentTimecourse<1>(synclist_cache,*(timecourse_cache[tcmode_plain]),nucleus,progmeter);
  }

  if(type==tcmode_M2) {
    create_timecourses(tcmode_plain,nucleus,progmeter);
    if(progmeter) progmeter->new_task(tcourse_size,progtxt.c_str());
    result=new SeqGradMomentTimecourse<2>(synclist_cache,*(timecourse_cache[tcmode_plain]),nucleus,progmeter);
  }

  if(type==tcmode_b_trace) {
    create_timecourses(tcmode_kspace,nucleus,progmeter);
    if(progmeter) progmeter->new_task(tcourse_size,progtxt.c_str());
    result=new SeqTwoFuncIntegralTimecourse(synclist_cache,*(timecourse_cache[tcmode_kspace]),*(timecourse_cache[tcmode_kspace]),progmeter);
  }

  if(type==tcmode_backgr_kspace) {
    create_timecourses(tcmode_plain,nucleus,progmeter);
    if(progmeter) progmeter->new_task(tcourse_size,progtxt.c_str());
    result=new SeqGradMomentTimecourse<0,true>(synclist_cache,*(timecourse_cache[tcmode_plain]),nucleus,progmeter);
  }

  if(type==tcmode_backgr_crossterm) {
    create_timecourses(tcmode_kspace,nucleus,progmeter);
    create_timecourses(tcmode_backgr_kspace,nucleus,progmeter);
    if(progmeter) progmeter->new_task(tcourse_size,progtxt.c_str());
    result=new SeqTwoFuncIntegralTimecourse(synclist_cache,*(timecourse_cache[tcmode_kspace]),*(timecourse_cache[tcmode_backgr_kspace]),progmeter);
  }

  if(type==tcmode_slew_rate) {
    SeqTimecourse* plain_tc_without_eddy_currents=new SeqTimecourse(synclist_cache,0,progmeter);
    if(progmeter) progmeter->new_task(tcourse_size,progtxt.c_str());
    result=new SeqSlewRateTimecourse(synclist_cache,*plain_tc_without_eddy_currents,progmeter);
    delete plain_tc_without_eddy_currents;
  }

  if(type==tcmode_eddy_currents) {
    if(timecourse_opts.EddyCurrentAmpl>0.0 && timecourse_opts.EddyCurrentTimeConst>0.0) {
      create_timecourses(tcmode_slew_rate,nucleus,progmeter);
      if(progmeter) progmeter->new_task(tcourse_size,progtxt.c_str());
      result=new SeqEddyCurrentTimecourse(synclist_cache,*(timecourse_cache[tcmode_slew_rate]),timecourse_opts,progmeter);
    }
  }


  ODINLOG(odinlog,normalDebug) << "result[" << type << "]=" << result << STD_endl;
  timecourse_cache[type]=result;
}


bool SeqPlotData::create_timecourses(timecourseMode type, const STD_string& nucleus, ProgressMeter* progmeter) const {
  Log<SeqStandAlone> odinlog("SeqPlotData","create_timecourses");

  if(!synclist_cache_done) create_synclist_cache(progmeter);
  if(!timecourse_cache[type]) create_timecourse_cache(type, nucleus, progmeter);

  return bool(timecourse_cache[type]);
}

const SeqTimecourseData* SeqPlotData::get_timecourse(timecourseMode type) const {
  return timecourse_cache[type];
}

const SeqTimecourseData* SeqPlotData::get_subtimecourse(timecourseMode type, double starttime, double endtime) const {
  if(timecourse_cache[type]) return timecourse_cache[type]->get_subtimecourse(starttime, endtime);
  return 0;
}


void SeqPlotData::get_timecourse_markers(timecourseMode type, STD_list<TimecourseMarker4Qwt>::const_iterator& result_begin, STD_list<TimecourseMarker4Qwt>::const_iterator& result_end, double starttime, double endtime ) const {
  if(timecourse_cache[type]) timecourse_cache[type]->get_markers(result_begin,result_end,starttime,endtime);
}


double SeqPlotData::get_total_duration() const {
  double result=0.0;
  for(STD_list<SeqPlotFrame>::const_iterator frameit=begin(); frameit!=end(); ++frameit) {
    result+=frameit->frameduration;
  }
  return result;
}

unsigned int SeqPlotData::numof_rec_channels() const {
  unsigned int result=1;
  CoilSensitivity* rec_coil=sim_opts.get_receiv_coil();
  if(rec_coil) result=rec_coil->get_numof_channels();
  return result;
}


JcampDxBlock& SeqPlotData::get_opts(bool include_timecourse_opts, bool include_simulation_opts) const {
  all_opts.clear();
  all_opts.set_label("Options");
  if(include_timecourse_opts) all_opts.merge(timecourse_opts);
  if(include_simulation_opts) {
    all_opts.merge(sim_opts);
    sim_opts.outdate_coil_cache(); // might be changed
  }
  return all_opts;
}

void SeqPlotData::set_coilsdir(const STD_string& coilsdir) const {
  sim_opts.TransmitterCoil.set_defaultdir(coilsdir);
  sim_opts.ReceiverCoil.set_defaultdir(coilsdir);
}


bool SeqPlotData::simulate(const STD_string& fidfile, const STD_string& samplefile, ProgressMeter* progmeter, SeqSimFeedbackAbstract* feedback) const {
  if(!create_timecourses(tcmode_plain,"",progmeter)) return false;
  return timecourse_cache[tcmode_plain]->simulate(synclist_cache,fidfile,samplefile,sim_opts,progmeter,feedback,this);
}


bool SeqPlotData::has_curves_on_channel(plotChannel chan) const {
  create_curves4qwt_cache();
  return has_curves_on_channel_cache[chan];
}
