// *****************************************************
// Copyright 2006, Charlie Poole
//
// Licensed under the Open Software License version 3.0
// *****************************************************

using System;
using NUnitLite.Framework;

namespace NUnitLite.Tests
{
    class MessageWriterTests : TestCase
    {
        public MessageWriterTests(string name) : base(name) { }

        protected TextMessageWriter writer;

        protected override void SetUp()
        {
            writer = new TextMessageWriter();
        }
    }

    [TestFixture]
    class TestMessageWriterTests : MessageWriterTests
    {
        public TestMessageWriterTests(string name) : base(name) { }

        [Test]
        public void ConnectorIsWrittenWithSurroundingSpaces()
        {
            writer.WriteConnector("and");
            Assert.That(writer.ToString(), Is.EqualTo(" and "));
        }

        [Test]
        public void PredicateIsWrittenWithTrailingSpace()
        {
            writer.WritePredicate("contains");
            Assert.That(writer.ToString(), Is.EqualTo("contains "));
        }

        [TestFixture]
        class ExpectedValueTests : ValueTests
        {
            public ExpectedValueTests(string name) : base(name) { }

            protected override void WriteValue(object obj)
            {
                writer.WriteExpectedValue(obj);
            }
        }

        [TestFixture]
        class ActualValueTests : ValueTests
        {
            public ActualValueTests(string name) : base(name) { }

            protected override void WriteValue(object obj)
            {
                writer.WriteActualValue( obj );
            }
        }

        abstract class ValueTests : MessageWriterTests
        {
            public ValueTests(string name) : base(name) { }

            protected abstract void WriteValue( object obj);

            [Test]
            public void IntegerIsWrittenAsIs()
            {
                WriteValue(42);
                Assert.That(writer.ToString(), Is.EqualTo("42"));
            }

            [Test]
            public void StringIsWrittenWithQuotes()
            {
                WriteValue("Hello");
                Assert.That(writer.ToString(), Is.EqualTo("\"Hello\""));
            }

            //[Test]
            //public void ControlCharactersInStringsAreEscaped()
            //{
            //    WriteValue("Best Wishes,\r\n\tCharlie\r\n");
            //    Assert.That(writer.ToString(), Is.EqualTo("\"Best Wishes,\\r\\n\\tCharlie\\r\\n\""));
            //}

            [Test]
            public void FloatIsWrittenWithTrailingF()
            {
                WriteValue(0.5f);
                Assert.That(writer.ToString(), Is.EqualTo("0.5f"));
            }

            [Test]
            public void FloatIsWrittenToNineDigits()
            {
                WriteValue(0.33333333333333f);
                int digits = writer.ToString().Length - 3;   // 0.dddddddddf
                Assert.That(digits, Is.EqualTo(9));
            }

            [Test]
            public void DoubleIsWrittenWithTrailingD()
            {
                WriteValue(0.5d);
                Assert.That(writer.ToString(), Is.EqualTo("0.5d"));
            }

            [Test]
            public void DoubleIsWrittenToSeventeenDigits()
            {
                WriteValue(0.33333333333333333333333333333333333333333333d);
                int digits = writer.ToString().Length - 3;
                Assert.That(digits, Is.EqualTo(17));
            }

            [Test]
            public void DecimalIsWrittenWithTrailingM()
            {
                WriteValue(0.5m);
                Assert.That(writer.ToString(), Is.EqualTo("0.5m"));
            }

            [Test]
            public void DecimalIsWrittenToTwentyNineDigits()
            {
                WriteValue(12345678901234567890123456789m);
                Assert.That(writer.ToString(), Is.EqualTo("12345678901234567890123456789m"));
            }
        }
    }
}
