#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
/* 
  Filename: users.cc
  keeps list of users
*/

#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <pwd.h>
#include <arc/StringConv.h>
#include <arc/Logger.h>
#include <arc/Utils.h>

#include <string>
#include <list>

#include "../conf/conf.h"
#include "../run/run_parallel.h"
#include "../misc/escaped.h"
#include "../jobs/states.h"
#include "../conf/environment.h"
#include "users.h"

static Arc::Logger& logger = Arc::Logger::getRootLogger();
static std::string empty_string("");

JobUser::JobUser(void) {
  control_dir="";
  unix_name=""; home=""; uid=0; gid=0;
  cache_params=NULL;
  valid=false; jobs=NULL;
  keep_finished=DEFAULT_KEEP_FINISHED;
  keep_deleted=DEFAULT_KEEP_DELETED;
  cred_plugin=NULL;
  strict_session=false;
  sharelevel=jobinfo_share_private;
}

void JobUser::SetLRMS(const std::string &lrms_name,const std::string &queue_name) {
  default_lrms=lrms_name;
  default_queue=queue_name;
}

void JobUser::SetControlDir(const std::string &dir) {
  if(dir.length() == 0) {
    control_dir=home + "/.jobstatus";
  }
  else { control_dir=dir; };
}

void JobUser::SetSessionRoot(const std::string &dir) {
  session_roots.clear();
  if(dir.length() == 0 || dir == "*") { session_roots.push_back(home + "/.jobs"); }
  else { session_roots.push_back(dir); };
}

void JobUser::SetSessionRoot(const std::vector<std::string> &dirs) {
  session_roots.clear();
  if (dirs.empty()) {
    std::string dir;
    SetSessionRoot(dir);
  } else {
    for (std::vector<std::string>::const_iterator i = dirs.begin(); i != dirs.end(); i++) {
      if (*i == "*") session_roots.push_back(home + "/.jobs");
      else session_roots.push_back(*i);
    }
  }
}

const std::string & JobUser::SessionRoot(std::string job_id) const {
  if (session_roots.size() == 0) return empty_string;
  if (session_roots.size() == 1 || job_id.empty()) return session_roots[0];
  // search for this jobid's session dir
  struct stat st;
  for (std::vector<std::string>::const_iterator i = session_roots.begin(); i != session_roots.end(); i++) {
    std::string sessiondir(*i + '/' + job_id);
    if (stat(sessiondir.c_str(), &st) == 0 && S_ISDIR(st.st_mode))
      return *i;
  }
  return empty_string; // not found
}

void JobUser::SetCacheParams(CacheConfig* params) {
  std::vector<std::string> cache_dirs = params->getCacheDirs();
  for (std::vector<std::string>::iterator i = cache_dirs.begin(); i != cache_dirs.end(); i++) {
    substitute(*i);
  }
  params->setCacheDirs(cache_dirs);
  std::vector<std::string> drain_cache_dirs = params->getDrainingCacheDirs();
  for (std::vector<std::string>::iterator i = drain_cache_dirs.begin(); i != drain_cache_dirs.end(); i++) {
    substitute(*i);
  }
  params->setDrainingCacheDirs(drain_cache_dirs);
  cache_params = params;
}

bool JobUser::CreateDirectories(void) {
  bool res = true;
  if(control_dir.length() != 0) {
    if(mkdir(control_dir.c_str(),S_IRWXU) != 0) {
      if(errno != EEXIST) res=false;
    } else {
      (chown(control_dir.c_str(),uid,gid) != 0);
      if(uid == 0) {
        chmod(control_dir.c_str(),S_IRUSR | S_IWUSR | S_IXUSR | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH);
      } else {
        chmod(control_dir.c_str(),S_IRUSR | S_IWUSR | S_IXUSR);
      };
    };
    if(mkdir((control_dir+"/logs").c_str(),S_IRWXU) != 0) {
      if(errno != EEXIST) res=false;
    } else {
      (chown((control_dir+"/logs").c_str(),uid,gid) != 0);
    };
  };
  if(session_roots.size() != 0) {
    for(std::vector<std::string>::iterator i = session_roots.begin(); i != session_roots.end(); i++) {
      if(mkdir(i->c_str(),S_IRWXU) != 0) {
        if(errno != EEXIST) res=false;
      } else {
        (chown(i->c_str(),uid,gid) != 0);
        if(uid == 0) {
          chmod(i->c_str(),S_IRUSR | S_IWUSR | S_IXUSR | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH);
        } else {
          chmod(i->c_str(),S_IRUSR | S_IWUSR | S_IXUSR);
        };
      };
    };
  };
  return res;
}

/*
  %R - session root
  %r - list of session roots
  %C - control dir
  %c - list of control dirs
  %U - username
  %u - userid
  %g - groupid
  %H - home dir
  %Q - default queue
  %L - default lrms
  %W - installation path
  %G - globus path
*/

bool JobUser::substitute(std::string& param) const {
  std::string::size_type curpos=0;
  for(;;) {
    if(curpos >= param.length()) break;
    std::string::size_type pos = param.find('%',curpos);
    if(pos == std::string::npos) break;
    pos++; if(pos>=param.length()) break;
    if(param[pos] == '%') { curpos=pos+1; continue; };
    std::string to_put;
    switch(param[pos]) {
      case 'R': to_put=SessionRoot(); break;
      case 'C': to_put=ControlDir(); break;
      case 'U': to_put=UnixName(); break;
      case 'H': to_put=Home(); break;
      case 'Q': to_put=DefaultQueue(); break;
      case 'L': to_put=DefaultLRMS(); break;
      case 'u': to_put=Arc::tostring(get_uid()); break;
      case 'g': to_put=Arc::tostring(get_gid()); break;
      case 'W': to_put=nordugrid_loc(); break;
      case 'G': to_put=globus_loc(); break;
      default: to_put=param.substr(pos-1,2);
    };
    curpos=pos+1+(to_put.length() - 2);
    param.replace(pos-1,2,to_put);
  };
  return true;
}

bool JobUsers::substitute(std::string& param) const {
  std::string session_roots = "";
  std::string control_dirs = "";
  for(JobUsers::const_iterator i = begin();i!=end();++i) {
    std::string tmp_s;
    tmp_s = i->SessionRoot();
    make_escaped_string(tmp_s);
    tmp_s=tmp_s+" ";
    if(session_roots.find(tmp_s) == std::string::npos) session_roots+=tmp_s;
    tmp_s = i->ControlDir();
    make_escaped_string(tmp_s);
    tmp_s=tmp_s+" ";
    if(control_dirs.find(tmp_s) == std::string::npos) control_dirs+=tmp_s;
  };
  std::string::size_type curpos=0;
  for(;;) {
    if(curpos >= param.length()) break;
    std::string::size_type pos = param.find('%',curpos);
    if(pos == std::string::npos) break;
    pos++; if(pos>=param.length()) break;
    if(param[pos] == '%') { curpos=pos+1; continue; };
    std::string to_put;
    switch(param[pos]) {
      case 'r': to_put=session_roots; break;
      case 'c': to_put=control_dirs; break;
      default: to_put=param.substr(pos-1,2);
    };
    curpos=pos+1+(to_put.length() - 2);
    param.replace(pos-1,2,to_put);
  };
  return true;
}

JobUser::JobUser(uid_t uid_,RunPlugin* cred) {
  struct passwd pw_;
  struct passwd *pw;
  char buf[BUFSIZ];
  uid=uid_;
  valid=false;
  cred_plugin=cred;
  /* resolve name */
  if(uid_ == 0) {
    unix_name="";  
    gid=0;
    home="/tmp";
    valid=true;
  }
  else {
    getpwuid_r(uid_,&pw_,buf,BUFSIZ,&pw);
    if(pw != NULL) {
      unix_name=pw->pw_name;  
      gid=pw->pw_gid;
      home=pw->pw_dir;
      valid=true;
    };
  };
  jobs=NULL;
  cache_params=NULL;
  SetControlDir("");
  SetSessionRoot("");
  SetLRMS("","");
  keep_finished=DEFAULT_KEEP_FINISHED;
  keep_deleted=DEFAULT_KEEP_DELETED;
  strict_session=false;
  sharelevel=jobinfo_share_private;
}

JobUser::JobUser(const std::string &u_name,RunPlugin* cred) {
  struct passwd pw_;
  struct passwd *pw;
  char buf[BUFSIZ];
  unix_name=u_name;  
  cred_plugin=cred;
  valid=false;
  /* resolve name */
  if(u_name.length() == 0) {
    uid=0;  
    gid=0;
    home="/tmp";
    valid=true;
  }
  else {
    getpwnam_r(u_name.c_str(),&pw_,buf,BUFSIZ,&pw);
    if(pw != NULL) {
      uid=pw->pw_uid;
      gid=pw->pw_gid;
      home=pw->pw_dir;
      valid=true;
    };
  };
  SetControlDir("");
  SetSessionRoot("");
  SetLRMS("","");
  jobs=NULL;
  cache_params=NULL;
  keep_finished=DEFAULT_KEEP_FINISHED;
  keep_deleted=DEFAULT_KEEP_DELETED;
  strict_session=false;
  sharelevel=jobinfo_share_private;
}

JobUser::JobUser(const JobUser &user) {
  uid=user.uid; gid=user.gid;
  unix_name=user.unix_name;  
  control_dir=user.control_dir;
  home=user.home;
  jobs=user.jobs;
  session_roots=user.session_roots;
  default_lrms=user.default_lrms;
  default_queue=user.default_queue;
  valid=user.valid;
  keep_finished=user.keep_finished;
  keep_deleted=user.keep_deleted;
  cache_params=user.cache_params;
  cred_plugin=user.cred_plugin;
  strict_session=user.strict_session;
  sharelevel=user.sharelevel;
}

JobUser::~JobUser(void) { 
  delete cache_params;
}

JobUsers::JobUsers(void) {
}

JobUsers::~JobUsers(void) {
}

JobUsers::iterator JobUsers::AddUser(const std::string &unix_name,RunPlugin* cred_plugin,const std::string &control_dir, const std::vector<std::string> *session_roots) {
  JobUser user(unix_name,cred_plugin);
  user.SetControlDir(control_dir);
  if(session_roots) user.SetSessionRoot(*session_roots);
  if(user.is_valid()) { return users.insert(users.end(),user); };
  return users.end();
}

std::string JobUsers::ControlDir(iterator user) {
  if(user == users.end()) return std::string("");
  return (*user).ControlDir();
}

JobUsers::iterator JobUsers::find(const std::string user) {
  iterator i;
  for(i=users.begin();i!=users.end();++i) {
    if((*i) == user) break;
  };
  return i;
}

std::string JobUsers::ControlDir(const std::string user) {
  for(iterator i=users.begin();i!=users.end();++i) {
    if((*i) == user) return (*i).ControlDir();
  };
  return std::string("");
}

#ifndef NO_GLOBUS_CODE

/* change effective user - real switch is done only if running as root */
bool JobUser::SwitchUser(bool su) const {
  std::string uid_s = Arc::tostring(uid);
  if(!Arc::SetEnv("USER_ID",uid_s)) if(!su) return false;
  if(!Arc::SetEnv("USER_NAME",unix_name)) if(!su) return false;
  /* set proper umask */
  umask(0177);
  if(!su) return true;
  uid_t cuid;
  if(((cuid=getuid()) != 0) && (uid != 0)) {
    if(cuid != uid) return false;
  };
  if(uid != 0) {
    setgid(gid); /* this is not an error if group failed, not a big deal */
    if(setuid(uid) != 0) return false;
  };
  return true;
}

bool JobUser::run_helpers(void) {
//  if(unix_name.length() == 0) { /* special users can not run helpers */
//    return true;  
//  };
  bool started = true;
  for(std::list<JobUserHelper>::iterator i=helpers.begin();i!=helpers.end();++i) {
    started &= i->run(*this);
  };
  return started;
}

bool JobUsers::run_helpers(void) {
  for(iterator i=users.begin();i!=users.end();++i) {
    i->run_helpers();
  };
  return true;
}
#endif // NO_GLOBUS_CODE

JobUserHelper::JobUserHelper(const std::string &cmd) {
  command=cmd;
  proc=NULL;
}
  
JobUserHelper::~JobUserHelper(void) {
#ifndef NO_GLOBUS_CODE
    if(proc != NULL) {
      delete proc;
      proc=NULL;
    };
#endif
}

#ifndef NO_GLOBUS_CODE
bool JobUserHelper::run(JobUser &user) {
    if(proc != NULL) {
      if(proc->Running()) {
        return true; /* it is already/still running */
      };
      delete proc;
      proc=NULL;
    };
    /* start/restart */
    if(command.length() == 0) return true;  /* has anything to run ? */
    char* args[100]; /* up to 98 arguments should be enough */
    std::string args_s = command;
    std::string arg_s;
    int n;
    for(n=0;n<99;n++) {
      arg_s=config_next_arg(args_s);
      if(arg_s.length() == 0) break;
      args[n]=strdup(arg_s.c_str());
    };
    args[n]=NULL;
    logger.msg(Arc::VERBOSE,"Starting helper process (%s): %s",
               user.UnixName().c_str(),command.c_str());
    std::string helper_id="helper."+user.UnixName();
    bool started=RunParallel::run(user,helper_id.c_str(),args,&proc);
    for(n=0;n<99;n++) {
      if(args[n] == NULL) break;
      free(args[n]);
    };
    if(started) return true;
    if(proc && (*proc)) return true;
    if(proc) { delete proc; proc=NULL; };
    logger.msg(Arc::ERROR,"Helper process start failed (%s): %s",
               user.UnixName().c_str(),command.c_str());
    /* start failed */
    /* doing nothing - maybe in the future */
    return false;
}
#endif

