# -*- coding: UTF-8 -*-

from __future__ import absolute_import

from PyQt4.QtCore import QObject
from PyQt4.QtCore import SIGNAL
from PyQt4.QtCore import pyqtSignal

from ninja_ide.core import settings
from ninja_ide.gui.main_panel import itab_item
from ninja_ide.gui.main_panel import main_container
from ninja_ide.gui import actions
from ninja_ide.gui.explorer import explorer_container


###############################################################################
# PLUGINS SERVICES
###############################################################################


class MainService(QObject):
    """
    Main Interact whith NINJA-IDE
    """
    #SIGNALS
    editorKeyPressEvent = pyqtSignal("QEvent")
    fileSaved = pyqtSignal("QString")
    currentTabChanged = pyqtSignal("QString")
    executeFile = pyqtSignal("QString")
    executeProject = pyqtSignal("QString")

    def __init__(self):
        QObject.__init__(self)
        self._main = main_container.MainContainer()
        self._action = actions.Actions()
        #Connect signals
        self.connect(self._main, SIGNAL("editorKeyPressEvent(QEvent)"),
            self._keyPressEvent)
        self.connect(self._main, SIGNAL("fileSaved(QString)"),
            self._fileSaved)
        self.connect(self._main, SIGNAL("currentTabChanged(QString)"),
            self._currentTabChanged)
        self.connect(self._action, SIGNAL("executeFile(QString)"),
            self._executeFile)
        self.connect(self._action, SIGNAL("executeProject(QString)"),
            self._executeProject)

    def add_menu(self, menu, lang=".py"):
        """
        Add an *extra context menu* to the editor context menu
        """
        itab_item.ITabItem.add_extra_menu(menu, lang=lang)

    def get_opened_documents(self):
        """
        Returns the opened documents
        """
        documents_data = self._main.get_opened_documents()
        #returns ONLY names!
        return [doc_data[0] for doc_list in documents_data
                    for doc_data in doc_list]

    def add_editor(self, fileName="", content=None, syntax=None):
        """
        Create a new editor
        """
        return self._main.add_editor(fileName=fileName, content=content,
                                        syntax=syntax)

    def get_editor(self):
        """
        Returns the actual editor (instance of ninja_ide.gui.editor.Editor)
        This method could return None
        """
        return self._main.get_actual_editor()

    def get_editor_path(self):
        """
        Returns the actual editor path
        This method could return None if there is not an editor
        """
        editor = self._main.get_actual_editor()
        if editor:
            return editor.ID
        return None

    def get_text(self):
        """
        Returns the plain text of the current editor
        or None if thre is not an editor.
        """
        editor = self._main.get_actual_editor()
        if editor:
            return editor.get_text()
        return

    def get_selected_text(self):
        """
        Returns the selected text of and editor.
        This method could return None
        """
        editor = self._main.get_actual_editor()
        if editor:
            return editor.textCursor().selectedText()
        return None

    def insert_text(self, text):
        """
        Insert text in the current cursor position
        @text: string
        """
        editor = self._main.get_actual_editor()
        if editor:
            editor.textCursor().insertText(text)

    def jump_to_line(self, lineno):
        """
        Jump to a specific line in the current editor
        """
        self._main.editor_jump_to_line(lineno=lineno)

    def get_lines_count(self):
        """
        Returns the count of lines in the current editor
        """
        editor = self._main.get_actual_editor()
        if editor:
            return editor.get_lines_count()
        return None

    def save_file(self):
        """
        Save the actual file
        """
        self._main.save_file()

    def open_files(self, files, mainTab=True):
        """
        Open many files
        """
        self._main.open_files(self, files, mainTab=mainTab)

    def open_file(self, fileName='', cursorPosition=0,
                    positionIsLineNumber=False):
        """
        Open a single file, if the file is already open it get focus
        """
        self._main.open_file(fileName=fileName, cursorPosition=cursorPosition,
                                positionIsLineNumber=positionIsLineNumber)

    def open_image(self, filename):
        """
        Open a single image
        """
        self._main.open_image(filename)

    def get_actual_tab(self):
        """
        Returns the actual widget
        (instance of ninja_ide.gui.main_panel.tab_widget.TabWidget)
        """
        return self._main.get_actual_widget()

    #SIGNALS
    def _keyPressEvent(self, qEvent):
        """
        Emit the signal when a key is pressed
        @event: QEvent
        """
        self.editorKeyPressEvent.emit(qEvent)

    def _fileSaved(self, fileName):
        self.fileSaved.emit(fileName)

    def _currentTabChanged(self, fileName):
        self.currentTabChanged.emit(fileName)

    def _executeFile(self, fileName):
        self.executeFile.emit(fileName)

    def _executeProject(self, projectName):
        self.executeProject.emit(projectName)


class ToolbarService(QObject):
    """
    Interact with the Toolbar
    """
    def __init__(self, toolbar):
        QObject.__init__(self)
        self._toolbar = toolbar

    def add_action(self, action):
        """
        Add an action to the Toolbar
        @action: Should be an instance(or subclass) of QAction
        """
        self._toolbar.addAction(action)


class MenuAppService(QObject):
    """
    Interact with the Plugins Menu
    """
    def __init__(self, plugins_menu):
        QObject.__init__(self)
        self._plugins_menu = plugins_menu

    def add_menu(self, menu):
        """
        Add an extra menu to the Plugin Menu of NINJA
        """
        self._plugins_menu.addMenu(menu)

    def add_action(self, action):
        """
        Add an action to the Plugin Menu of NINJA
        """
        self._plugins_menu.addAction(action)


#class ProjectTypeService(QObject):
#    """
#    Interact with the New Project Wizard
#    """
#    def __init__(self):
#        QObject.__init__(self)
#
#    def set_project_type_handler(self, project_type, project_type_handler):
#        """
#        Add a new Project Type and the handler for it
#        example:
#            foo_project_handler = FooProjectHandler(...)
#            set_project_type_handler('Foo Project', foo_project_handler)
#        Then 'Foo Project' will appear in the New Project wizard
#            and foo_project_handler instance controls the wizard
#
#        Note: project_type_handler SHOULD have a special interface see
#        ninja_ide.core.plugin_interfaces
#        """
#        settings.set_project_type_handler(project_type, project_type_handler)
#
#
#class TreeSymbolsService(QObject):
#    """
#    Interact with the symbols tree
#    """
#    def __init__(self):
#        QObject.__init__(self)
#
#    def set_symbols_handler(self, file_extension, symbols_handler):
#        """
#        Add a new Symbol's handler for the given file extension
#        example:
#            cpp_symbols_handler = CppSymbolHandler(...)
#            set_symbols_handler('.cpp', cpp_symbols_handler)
#        Then all symbols in .cpp files will be handle by cpp_symbols_handler
#
#        Note: symbols_handler SHOULD have a special interface see
#        ninja_ide.core.plugin_interfaces
#        """
#        settings.set_symbols_handler(file_extension, symbols_handler)


class MiscContainerService(QObject):

    def __init__(self, miscContainer):
        QObject.__init__(self)
        self._misc = miscContainer

    def add_widget(self, widget, icon_path, description):
        self._misc.add_to_stack(widget, icon_path, description)


class ExplorerService(QObject):

    def __init__(self):
        QObject.__init__(self)
        self._explorer = explorer_container.ExplorerContainer()

    def get_tree_projects(self):
        """
        Returns the projects tree
        """
        return self._explorer._treeProjects

    def get_current_project_item(self):
        """
        Returns the current item of the tree projects
        this methos is a shortcut of self.get_tree_projects().currentItem()
        """
        return self._explorer._treeProjects.currentItem()

    def get_tree_symbols(self):
        """
        Returns the symbols tree
        """
        return self._explorer._treeSymbols

    def set_symbols_handler(self, file_extension, symbols_handler):
        """
        Add a new Symbol's handler for the given file extension
        example:
            cpp_symbols_handler = CppSymbolHandler(...)
            set_symbols_handler('.cpp', cpp_symbols_handler)
        Then all symbols in .cpp files will be handle by cpp_symbols_handler

        Note: symbols_handler SHOULD have a special interface see
        ninja_ide.core.plugin_interfaces
        """
        settings.set_symbols_handler(file_extension, symbols_handler)

    def set_project_type_handler(self, project_type, project_type_handler):
        """
        Add a new Project Type and the handler for it
        example:
            foo_project_handler = FooProjectHandler(...)
            set_project_type_handler('Foo Project', foo_project_handler)
        Then 'Foo Project' will appear in the New Project wizard
            and foo_project_handler instance controls the wizard

        Note: project_type_handler SHOULD have a special interface see
        ninja_ide.core.plugin_interfaces
        """
        settings.set_project_type_handler(project_type, project_type_handler)

    def add_tab(self, tab, title):
        """
        Add a tab with the given title
        @tab: Should be an instance (or subclass )of QTabWidget
        @title: Name of the tab string
        """
        self._explorer.addTab(tab, title)

    def get_actual_project(self):
        """
        Returns the path of the opened projects
        """
        return self._explorer.get_actual_project()

    def add_project_menu(self, menu, lang='all'):
        """
        Add an extra menu to the project explorer for the files
        with the given extension.
        @lang: String with file extension format (py, php, json)
        """
        self._explorer._treeProjects.add_extra_menu(menu, lang=lang)
