# -*- coding: utf-8 -*-
#
#  Copyright (C) 2003-2011 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

import logging
import os
import hashlib
import logging

import numpy
import gtk


class TransparentWindow(gtk.Window):
    __gsignals__ = {'screen-changed': 'override', }

    def __init__(self, type=gtk.WINDOW_TOPLEVEL):
        gtk.Window.__init__(self, type)
        self.set_app_paintable(True)
        self.screen_changed()
        self.truncated = False
        self.mask_offset = (0, 0)
        fixed = gtk.Fixed()
        fixed.show()
        gtk.Window.add(self, fixed) # XXX
        self.__fixed = fixed
        self.__mask = None
        self.__child = None
        self.__position = (0, 0)
        self.connect_after('size_allocate', self.size_allocate)

    @property
    def mask(self):
        return self.__mask

    @mask.setter
    def mask(self, mask):
        x, y = self.mask_offset
        self.__mask = mask
        if self.is_composited():
            self.input_shape_combine_mask(mask, x, y)
        else:
            self.shape_combine_mask(mask, x, y)

    @property
    def fixed(self): # read only
        return self.__fixed

    def add(self, widget):
        self.fixed.put(widget, 0, 0)
        self.__child = widget

    def remove(self, widget):
        self.fixed.remove(widget)
        self.__child = None

    def destroy(self):
        gtk.Window.remove(self, self.fixed)
        self.fixed.destroy()
        gtk.Window.destroy(self)

    def screen_changed(self, old_screen=None):
        screen = self.get_screen()
        if self.is_composited():
            colormap = screen.get_rgba_colormap()
            self.supports_alpha = True
        else:
            logging.debug('screen does NOT support alpha.\n')
            colormap = screen.get_rgb_colormap()
            self.supports_alpha = False
        self.set_colormap(colormap)

    def size_allocate(self, widget, event):
        new_x, new_y = self.__position
        gtk.Window.move(self, new_x, new_y)

    def reset_truncate(self):
        self.fixed.move(self.__child, 0, 0)
        self.set_size_request(-1, -1)
        if self.is_composited():
            self.input_shape_combine_mask(self.mask, 0, 0)
        else:
            self.shape_combine_mask(self.mask, 0, 0)
        self.mask_offset = (0, 0)
        self.truncated = False

    def resize_move(self, x, y, xoffset=0, yoffset=0):
        ### XXX: does not work propery for gtk.WINDOW_POPUP
        ##if self.type == gtk.WINDOW_POPUP:
        ##    gtk.Window.move(self, x, y)
        ##    return
        w, h = self.__child.get_size_request()
        left, top, scrn_w, scrn_h = get_workarea()
        new_x = min(max(x + xoffset, left - w + 1), left + scrn_w - 1)
        new_y = min(max(y + yoffset, top - h + 1), top + scrn_h - 1)
        new_w = w
        new_h = h
        offset_x = 0
        offset_y = 0
        truncate = False
        if new_x < left:
            offset_x = new_x - left
            new_x = left
            new_w += offset_x
            truncate = True
        if new_x + w > left + scrn_w:
            new_w -= new_x + w - (left + scrn_w)
            truncate = True
        if new_y < top:
            offset_y = new_y - top
            new_y = top
            new_h += offset_y
            truncate = True
        if new_y + h > top + scrn_h:
            new_h -= new_y + h - (top + scrn_h)
            truncate = True
        if not truncate:
            if self.truncated:
                self.reset_truncate()
            else:
                gtk.Window.move(self, new_x, new_y) # XXX
        else:
            self.fixed.move(self.__child, offset_x, offset_y)
            self.set_size_request(new_w, new_h)
            if self.is_composited():
                self.input_shape_combine_mask(self.mask, offset_x, offset_y)
            else:
                self.shape_combine_mask(self.mask, offset_x, offset_y)
            self.truncated = True
            self.mask_offset = (offset_x, offset_y)
        self.__position = (new_x, new_y)
        self.__child.queue_draw()


def get_png_size(path):
    if not path or not os.path.exists(path):
        return 0, 0
    head, tail = os.path.split(path)
    basename, ext = os.path.splitext(tail)
    ext = ext.lower()
    if ext == '.dgp':
        buf = get_DGP_IHDR(path)
    elif ext == '.ddp':
        buf = get_DDP_IHDR(path)
    else:
        buf = get_png_IHDR(path)
    assert buf[0:8] == '\x89PNG\r\n\x1a\n' # png format
    assert buf[12:16] == 'IHDR' # name of the first chunk in a PNG datastream
    w = buf[16:20]
    h = buf[20:24]
    width = (ord(w[0]) << 24) + (ord(w[1]) << 16) + (ord(w[2]) << 8) + ord(w[3])
    height = (ord(h[0]) << 24) + (ord(h[1]) << 16) + (ord(h[2]) << 8) + ord(h[3])
    return width, height

def get_DGP_IHDR(path):
    head, tail = os.path.split(path)
    filename = tail
    m_half = hashlib.md5(filename[:len(filename) / 2]).hexdigest()
    m_full = hashlib.md5(filename).hexdigest()
    tmp = ''.join((m_full, filename))
    key = ''
    j = 0
    for i in range(len(tmp)):
        value = ord(tmp[i]) ^ ord(m_half[j])
        if not value:
            break
        key = ''.join((key, chr(value)))
        j += 1
        if j >= len(m_half):
            j = 0
    key_length = len(key)
    if key_length == 0: # not encrypted
        logging.warning(''.join((filename, ' generates a null key.')))
        return get_png_IHDR(path)
    key = ''.join((key[1:], key[0]))
    key_pos = 0
    buf = ''
    with open(path, 'rb') as f:
        for i in range(24):
            c = f.read(1)
            buff = ''.join((buf, chr(ord(c) ^ ord(key[key_pos]))))
            key_pos += 1
            if key_pos >= key_length:
                key_pos = 0
    return buf

def get_DDP_IHDR(path):
    size = os.path.getsize(path)
    key = size << 2    
    buf = ''
    with open(path, 'rb') as f:
        for i in range(24):
            c = f.read(1)
            key = (key * 0x08088405 + 1) & 0xffffffff
            buf = ''.join((buf, chr((ord(c) ^ key >> 24) & 0xff)))
    return buf

def get_png_IHDR(path):
    with open(path, 'rb') as f:
        buf = f.read(24)
    return buf

def create_blank_pixbuf(width, height):
    pixbuf = gtk.gdk.Pixbuf(gtk.gdk.COLORSPACE_RGB, True, 8, width, height)
    pixbuf.fill(0xffffffffL)
    return pixbuf

def create_pixbuf_from_DGP_file(path):
    head, tail = os.path.split(path)
    filename = tail
    m_half = hashlib.md5(filename[:len(filename) / 2]).hexdigest()
    m_full = hashlib.md5(filename).hexdigest()
    tmp = ''.join((m_full, filename))
    key = ''
    j = 0
    for i in range(len(tmp)):
        value = ord(tmp[i]) ^ ord(m_half[j])
        if not value:
            break
        key = ''.join((key, chr(value)))
        j += 1
        if j >= len(m_half):
            j = 0
    key_length = len(key)
    if key_length == 0: # not encrypted
        logging.warning(''.join((filename, ' generates a null key.')))
        pixbuf = gtk.gdk.pixbuf_new_from_file(filename)
        return pixbuf
    key = ''.join((key[1:], key[0]))
    key_pos = 0
    loader = gtk.gdk.PixbufLoader('png')
    with open(path, 'rb') as f:
        while 1:
            c = f.read(1)
            if c == '':
                break
            loader.write(chr(ord(c) ^ ord(key[key_pos])), 1)
            key_pos += 1
            if key_pos >= key_length:
                key_pos = 0
    pixbuf = loader.get_pixbuf()
    loader.close()
    return pixbuf

def create_pixbuf_from_DDP_file(path):
    with open(path, 'rb') as f:
        buf = f.read()
    key = len(buf) << 2
    loader = gtk.gdk.PixbufLoader('png')
    for i in range(len(buf)):
        key = (key * 0x08088405 + 1) & 0xffffffff
        loader.write(chr((ord(buf[i]) ^ key >> 24) & 0xff), 1)
    pixbuf = loader.get_pixbuf()
    loader.close()
    return pixbuf

def create_pixbuf_from_file(path, is_pnr=True, use_pna=False):
    head, tail = os.path.split(path)
    basename, ext = os.path.splitext(tail)
    ext = ext.lower()
    if ext == '.dgp':
        pixbuf = create_pixbuf_from_DGP_file(path)
    elif ext == '.ddp':
        pixbuf = create_pixbuf_from_DDP_file(path)
    else:
        pixbuf = gtk.gdk.pixbuf_new_from_file(path)
    if is_pnr:
        array = pixbuf.get_pixels_array()
        if not pixbuf.get_has_alpha():
            r, g, b = array[0][0]
            pixbuf = pixbuf.add_alpha(True, chr(r), chr(g), chr(b))
        else:
            ar = numpy.frombuffer(array, numpy.uint8)
            alen = pixbuf.get_width() * pixbuf.get_height()
            ar.shape = alen, 4
            rgba = ar[0]
            index = numpy.ravel(ar == rgba)
            index.shape = alen, 4
            ar[numpy.all(index, 1)] = 0
    if use_pna:
        path = os.path.join(head, ''.join((basename, '.pna')))
        if os.path.exists(path):
            assert pixbuf.get_has_alpha()
            pna_pixbuf = gtk.gdk.pixbuf_new_from_file(path)
            pna_array = pna_pixbuf.get_pixels_array()
            assert pna_pixbuf.get_bits_per_sample() / 8 == 1
            pixbuf_array = pixbuf.get_pixels_array()
            pixbuf_array[:,:,3] = pna_array[:,:,0]
    return pixbuf

def create_pixmap_from_file(path):
    pixbuf = create_pixbuf_from_file(path)
    pixmap, mask = pixbuf.render_pixmap_and_mask(1)
    return pixmap, mask

def get_workarea():
    scrn = gtk.gdk.screen_get_default()
    root = scrn.get_root_window()
    if not hasattr(scrn, 'supports_net_wm_hint') or \
            not scrn.supports_net_wm_hint('_NET_CURRENT_DESKTOP') or \
            not scrn.supports_net_wm_hint('_NET_WORKAREA'):
        left, top, width, height, depth = root.get_geometry()
    else:
        index = root.property_get('_NET_CURRENT_DESKTOP')[2][0] * 4
        left, top, width, height = root.property_get('_NET_WORKAREA')[2][index:index+4]
    return left, top, width, height
