/*
** SlashTDP demo
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <SDL/SDL.h>
#include <slashtdp/slashtdp.h>

using namespace SlashTDP;

SDL_Surface	*screen;
unsigned short	*pixels;
unsigned int	pitch;
int		red;
int		green;
int		blue;

int		collideSegment = -1;
int		highSegment = -1;
Vector		collidePoint, highPoint, startPoint;

World		world;
Body		*body;

char		*map[] = {
	"xxxxxxxxxxxxxxxxxxxx",
	"x        x         x",
	"x   x       x  xxx x",
	"x xxx       x      x",
	"x         xxxxx    x",
	"x                  x",
	"x    xxxx     x    x",
	"x           xxxx   x",
	"x             xxxxxx",
	"x      xxxx        x",
	"x                  x",
	"xxxxxxx   x        x",
	"x        xxx  xxx  x",
	"x  x    xxxxx   x  x",
	"xxxxxxxxxxxxxxxxxxxx"
};

extern "C" {
unsigned long	rdtsc_h;
unsigned long	rdtsc_l;
};
unsigned long long rdtsc_off = 0;
unsigned long long rdtsc()
{
	asm("rdtsc");
	asm("movl %edx, rdtsc_h");
	asm("movl %eax, rdtsc_l");
	return (((unsigned long long)rdtsc_h << 32)|rdtsc_l)-rdtsc_off;
}

void drawPixel(int x, int y, int color)
{
	if (x >= 0 && x < 640 && y >= 0 && y < 480)
		pixels[y*pitch + x] = color;
}

void drawLine(int x1, int y1, int x2, int y2, int color)
{
	float	len = hypot(x2 - x1, y2 - y1);
	float	x, y, dx, dy;
	
	dx = (x2 - x1)/len;
	dy = (y2 - y1)/len;
	
	x = x1;
	y = y1;
	
	for (int i=0;i<(int)len;i++,x+=dx,y+=dy)
		drawPixel((int)x, (int)y, color);
	
	drawPixel(x1, y1, color);
	drawPixel(x2, y2, color);
}

void drawLine(const Segment &seg, int color)
{
	drawLine((int)seg.a.x, (int)seg.a.y, (int)seg.b.x, (int)seg.b.y,
		color); 
}

void drawWorld()
{
	memset(pixels, 0, 640*480*2);
	int	v = 50;
	for (uint i=0;i<world.segments;i++,v+=55)
	{
		if (v > 255)
			v = 50;
		drawLine(world.segment[i],
			collideSegment==i?blue:(
			highSegment==i?SDL_MapRGB(screen->format, 255, 0, 128):
			SDL_MapRGB(screen->format, 0, v, 0)));
		Vector	n = (world.segment[i].b - world.segment[i].a).perpendicular().normalized()*14.0f;
		Vector	c = (world.segment[i].b - world.segment[i].a)*0.5f;
		c += world.segment[i].a;
		drawLine(c.x, c.y, c.x + n.x, c.y + n.y, red);
	}
}

void drawBody(const Body &body)
{
	for (uint i=0;i<body.segments;i++)
	{
		drawLine(body.segment[i] + body.center,
			body.onGround?red:green);
		drawLine(body.center.x, body.center.y,
			body.center.x + body.velocity.x*4.0f,
			body.center.y + body.velocity.y*4.0f, blue);
	}
	for (uint i=0;i<body.points;i++)
	{
		Vector	p = body.point[i] + body.center;
		drawPixel(p.x, p.y, 0xFFFF);
	}
}

Vector	V(120, 50);

void draw()
{
	SDL_LockSurface(screen);
	pixels = (unsigned short*)screen->pixels;
	pitch = screen->pitch/2;
	
	drawWorld();
	for (Body *b=world.firstBody;b;b=b->next)
		drawBody(*b);
	
	drawLine(Segment(startPoint, highPoint), SDL_MapRGB(screen->format,
		64, 64, 64));

	drawPixel(collidePoint.x, collidePoint.y, SDL_MapRGB(screen->format,
		255, 0, 255));
	drawPixel(highPoint.x, highPoint.y, SDL_MapRGB(screen->format,
		255, 255, 0));
	drawPixel(startPoint.x, startPoint.y, SDL_MapRGB(screen->format,
		43, 255, 43));
		
	
	SDL_UnlockSurface(screen);
	SDL_Flip(screen);
}

void initWorld()
{
	for (int y=0;y<15;y++)
		for (int x=0;x<20;x++)
			if (map[y][x] == 'x')
				world.addBox(x*32, y*32, x*32+31, y*32+31);
	//world.addSegment(Segment(40, 240, 190, 320));
	world.optimize();
	
	body = world.createBody(Vector(50, 50));
	body->addBox(-10, -10, 10, 10);
	
	Body	*bod = world.createBody(Vector(220, 50));
	bod->addBox(-12, -12, 12, 12);
	bod->applyForce(Vector(12, 0));
	bod->bouncing = true;
	bod->bounce = 0.6f;
}

void run()
{
	SDL_Event	e;
	bool		running = true;
	
	initWorld();
	
	while (running)
	{
		static uint	lastTicks = SDL_GetTicks();
		uint		ticks = SDL_GetTicks();
		static bool	movingLeft = false, movingRight = false;
		
		if (ticks - lastTicks > 25)
		{
			float	fv;
			ticks = lastTicks;
			unsigned long long	cycles = rdtsc();
			
			if (movingLeft && body->velocity.x > -4.0)
			{
				if (body->velocity.x + (-1.0f) > -4.0f)
					fv = -1.0f;
				else
					fv = -fabs(body->velocity.x + 4.0f);
				body->applyForce(Vector(fv, 0));
			}
			else if (movingRight && body->velocity.x < 4.0f)
			{
				if (body->velocity.x + 1.0f < 4.0f)
					fv = 1.0f;
				else
					fv = 4.0 - body->velocity.x;
				body->applyForce(Vector(fv, 0));
			}
			
			world.move();
			
			char	buff[32];
			sprintf(buff, "%llu cycles", rdtsc()-cycles);
			SDL_WM_SetCaption(buff, buff);
		}
		
		draw();
		while (SDL_PollEvent(&e))
		{
			switch (e.type)
			{
				case SDL_QUIT:
					running = false;
					break;
				case SDL_KEYDOWN:
					if (e.key.keysym.sym == SDLK_d)
						V.x++;
					if (e.key.keysym.sym == SDLK_a)
						V.x--;
					if (e.key.keysym.sym == SDLK_s)
						V.y--;
					if (e.key.keysym.sym == SDLK_w)
						V.y++;
					if (e.key.keysym.sym == SDLK_RIGHT)
						movingRight = true;
					if (e.key.keysym.sym == SDLK_LEFT)
						movingLeft = true;
					if (e.key.keysym.sym == SDLK_UP)
					{
						if (body->onGround)
						{
							body->applyForce(Vector(0, -5));
							body->onGround = false;
						}
					}
					if (e.key.keysym.sym == SDLK_DOWN)
						body->applyForce(Vector(0, 2));
					if (e.key.keysym.sym == SDLK_ESCAPE)
						running = false;
					break;
				case SDL_KEYUP:
					if (e.key.keysym.sym == SDLK_LEFT)
						movingLeft = false;
					if (e.key.keysym.sym == SDLK_RIGHT)
						movingRight = false;
					break;
			}
		}
	}
}

int main()
{
	SDL_Init(SDL_INIT_VIDEO);
	screen = SDL_SetVideoMode(640, 480, 16, SDL_SWSURFACE);
	red = SDL_MapRGB(screen->format, 255, 0, 0);
	green = SDL_MapRGB(screen->format, 0, 255, 0);
	blue = SDL_MapRGB(screen->format, 64, 64, 255);

	unsigned long long	c = rdtsc();
	for (uint i=0;i<1000;i++)
		rdtsc();
	rdtsc_off = (rdtsc()-c)/1000;
	
	run();
	
	SDL_Quit();
	return 0;
}

