/*
 	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com).
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


#ifndef REPSTORE_ASN1_H
#define REPSTORE_ASN1_H

#include <mVector.h>
#include <ASN1/NewPKIObject.h>
#include <openssl/asn1.h>
#include <ASN1/Asn1Helper.h>

/*!
	This structure represents the signature of an entry.
*/
typedef struct st_REP_STORE_SIG
{
	ASN1_BIT_STRING * signature;		//!< The signature.
	X509_ALGOR * sig_alg;				//!< The algo.
}REP_STORE_SIG;
DECLARE_ASN1_ITEM(REP_STORE_SIG)


/*!
  This class is the representation of REP_STORE_SIG
*/
class RepStoreSig : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	RepStoreSig();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	RepStoreSig(const RepStoreSig & other);

	/*! \brief This is the destructor.
	 */
	~RepStoreSig();


	/*! \brief This function converts a REP_STORE_SIG to a RepStoreSig.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const REP_STORE_SIG * Datas);

	/*! \brief This function converts a RepStoreSig to a REP_STORE_SIG.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(REP_STORE_SIG ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of REP_STORE_SIG.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets X509_ALGOR::sig_alg.
	 *  \param c_sigAlg [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_sigAlg(const X509_ALGOR * c_sigAlg);

	/*! \brief This function returns X509_ALGOR::sig_alg.
	 *  \return The value of X509_ALGOR::sig_alg.
	 */
	const X509_ALGOR * get_sigAlg() const;

	/*! \brief This function returns X509_ALGOR::sig_alg.
	 *  \return The value of X509_ALGOR::sig_alg.
	 */
	X509_ALGOR * get_sigAlg();


	/*! \brief This function sets ASN1_BIT_STRING::signature.
	 *  \param c_signature [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_signature(const ASN1_BIT_STRING * c_signature);

	/*! \brief This function returns ASN1_BIT_STRING::signature.
	 *  \return The value of ASN1_BIT_STRING::signature.
	 */
	const ASN1_BIT_STRING * get_signature() const;

	/*! \brief This function returns ASN1_BIT_STRING::signature.
	 *  \return The value of ASN1_BIT_STRING::signature.
	 */
	ASN1_BIT_STRING * get_signature();


	/*! \brief This funciton generates the PEM representation of the local instance.
	 *  \param PemDatas [OUT] the PEM output.
	 *  \return true on success, false on failure.
	 */
	bool to_PEM(mString & PemDatas) const;

	/*! \brief This function loads the local instance from a PEM representation.
	 *  \param PemDatas [IN] the PEM input.
	 *  \return true on success, false on failure.
	 */
	bool from_PEM(const mString & PemDatas);


	/*! \brief This function copies a RepStoreSig to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const RepStoreSig & other);
	/*! \brief This member is an empty instance of RepStoreSig.
	 */
	static RepStoreSig EmptyInstance;

private:
	X509_ALGOR * m_sigAlg;
	ASN1_BIT_STRING * m_signature;
	void resetAll();
	void freeAll();
protected:
};


#endif // #ifndef REPSTORE_ASN1_H
