/* pbuild.c
 
   libpbuild public functions.

   Copyright (C) 2007, 2008, 2009, 2010 Eloy Paris

   This is part of Network Expect (nexp)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <math.h>

#include "pbuild-priv.h"

/*
 * Returns the size in bytes of a PDU. The size of a PDU is:
 *
 * PDU size = size of PDU header + size of PDU options + size of PDU payload
 */
size_t
pb_len(const GNode *pdu)
{
    struct node_data *node_data;

    node_data = pdu->data;

    return   node_data->_data_pdu.hdr_len
	   + node_data->_data_pdu.opts_len
	   + node_data->_data_pdu.payload_len;
}

/*
 * Builds a PDU.
 *
 * The PDU is built by calling the builder functions stored in the pdu
 * structure passed as a parameter. The PDU is built at the address
 * pointed to by the "dest" parameter.
 */
size_t
pb_build(const GNode *pdu, uint8_t *dest, uint8_t *prev_pdu)
{
    uint8_t *opts_ptr;
    size_t hdr_len, real_pdu_size, opts_len, payload_len;
    const pdu_t *p;
    const GNode *next_pdu;
    size_t only_first;
    guint noptions, i;
    struct option_data *option_data;
    struct node_data *node_data;

    p = ( (struct node_data *) pdu->data)->_data_pdu.template;

    hdr_len = ( (struct node_data *) pdu->data)->_data_pdu.hdr_len;
    payload_len = ( (struct node_data *) pdu->data)->_data_pdu.payload_len;
    opts_len = ( (struct node_data *) pdu->data)->_data_pdu.opts_len;

    /* Zero out the PDU header so all fields are 0 by default */
    memset(dest, 0, hdr_len);

    /* Build header - first pass */
    if (p->build)
	p->build(pdu, dest); /* Pass 1 builder supplied */
    else
	_pb_generic_build(pdu, dest); /* No pass 1 builder; use generic */

    /* Put the payload in place */
    if ( (next_pdu = pb_nextpdu(pdu) ) )
	pb_build(next_pdu, dest + hdr_len + opts_len, dest);

    /* Build options */
    noptions = ( (struct node_data *) pdu->data)->_data_pdu.opts_data
	       ? ( (struct node_data *) pdu->data)->_data_pdu.opts_data->len
	       : 0;

    for (opts_ptr = dest + hdr_len, i = 0; i < noptions; i++) {
	node_data = pdu->data;
	option_data = g_ptr_array_index(node_data->_data_pdu.opts_data, i);

	if (option_data->template->build)
	    /* User-provided option builder; let's use that one */
	    option_data->template->build(option_data->option, opts_ptr, dest,
					 prev_pdu);
	else
	    /* No user-provided option builder; let's use the generic builder */
	    _pb_generic_build(option_data->option, opts_ptr);

	opts_ptr += option_data->len;
    }

    /* Build header - second pass */
    if (p->postbuild)
	p->postbuild(pdu, dest, prev_pdu);

    real_pdu_size = hdr_len + opts_len + payload_len;

    only_first = num_next(_pb_pdata(pdu, "only-first") );

    return (only_first != 0 && real_pdu_size > only_first)
	   ? only_first : real_pdu_size;
}

static gboolean
destroy_traverse_func(GNode *node, gpointer data _U_)
{
    struct node_data *node_data;
    guint i;

    node_data = node->data;

    free(node_data->name); /* Allocated via strdup() by the lexer */

    if (node_data->type == PDU_NODE_PARM) {
	switch (node_data->_data_parm.field->type) {
	case PDU_FTYPE_BRACED_ARGS:
	case PDU_FTYPE_BIT:
	case PDU_FTYPE_BOOL:
	    /* Nothing */
	    break;
	case PDU_FTYPE_NUMTYPE:
	case PDU_FTYPE_BITFIELD:
	case PDU_FTYPE_UINT8:
	case PDU_FTYPE_UINT16:
	case PDU_FTYPE_UINT32:
	case PDU_FTYPE_IP:
	    num_destroy(node_data->_data_parm.data);
	    break;
	case PDU_FTYPE_UINT:
	    free(node_data->_data_parm.data);
	    break;
	case PDU_FTYPE_MACADDR:
	case PDU_FTYPE_IP6ADDR:
	    free(node_data->_data_parm.data);
	    break;
	case PDU_FTYPE_FIXEDP32:
	case PDU_FTYPE_FIXEDP64:
	    free(node_data->_data_parm.data);
	    break;
	case PDU_FTYPE_DATA:
	    free( ( (struct payload *) node_data->_data_parm.data)->data);
	    free(node_data->_data_parm.data);
	    break;
	}
    } else if (node_data->type == PDU_NODE_PDU) {
	/* Free options data, if present */
	if (node_data->_data_pdu.opts_data != NULL) {
	    for (i = 0; i < node_data->_data_pdu.opts_data->len; i++)
		free(g_ptr_array_index(node_data->_data_pdu.opts_data, i) );

	    g_ptr_array_free(node_data->_data_pdu.opts_data, TRUE);
	}
    }

    free(node_data);

    return FALSE;
}

/*
 * Releases all the structures allocated by a PDU.
 */
void
pb_destroy(GNode *pdu)
{
    g_node_traverse(pdu, G_PRE_ORDER, G_TRAVERSE_ALL, -1,
		    destroy_traverse_func, NULL);

    g_node_destroy(pdu);
}

#if 0
/*
 * Recursively dumps a PDU.
 */
void
pdu_dump(pdu_t *p, int verbose, int level)
{
    int i, noptions;
    pdu_option_t *option;
    char *prefix;

    prefix = pdu_make_prefix(level);

    printf("%sPDU #%d: %s\n", prefix, level, p->name);

    if (verbose) {
	printf("%s  Header size: %d bytes\n"
	       "%s  Number of bytes to include: ",
	       prefix, p->hdr_len,
	       prefix);

	if (p->only_first != 0)
	    printf("only first %u\n", p->only_first);
	else
	    printf("entire PDU\n");

	if (p->header_dumper)
	    p->header_dumper(p, prefix);
    }

    /* First we count the number of options */
    for (noptions = 0, option = p->pdu_options; option; option=option->next)
	noptions++;
    printf("%s  Number of options/extension headers: %d\n", prefix, noptions);

    /* Now we print each option name */
    for (i = 0, option = p->pdu_options; option; option = option->next, i++) {
	printf("%s    Option %2d: %s\n", prefix, i, option->name);
	if (verbose) {
	    printf("%s      Size: %d %s\n", prefix, option->len,
		   option->len == 1 ? "byte" : "bytes");
	    if (option->option_dumper)
		option->option_dumper(option, prefix);
	}
    }

    printf("%s", prefix);
    if (p->next) {
	printf("  Payload: PDU #%d\n", level + 1);
	pdu_dump(p->next, verbose, level + 1);
    } else
	printf("  Payload: this PDU has no payload (or the PDU is a payload.)\n");
}
#endif

static gboolean
dump_traverse_func(GNode *node, gpointer data _U_)
{
    struct node_data *node_data;
    static char *node_types[] = {
	[PDU_NODE_PDU] = "PDU",
	[PDU_NODE_PARM] = "parm",
	[PDU_NODE_BRACED_PARMS] = "braced-parms"
    };
    double d;
    struct fixedp_num *fpn;

    node_data = node->data;

    printf("%s%s (type = %s",
	   _pb_make_prefix(g_node_depth(node) - 1), node_data->name,
	   node_types[node_data->type]);

    if (node_data->type == PDU_NODE_PARM) {
	switch (node_data->_data_parm.field->type) {
	case PDU_FTYPE_BRACED_ARGS:
	    /* Nothing */
	    break;
	case PDU_FTYPE_NUMTYPE:
	case PDU_FTYPE_BITFIELD:
	case PDU_FTYPE_UINT8:
	case PDU_FTYPE_UINT16:
	case PDU_FTYPE_UINT32:
	case PDU_FTYPE_IP:
	    printf(", value = %s)\n", num_info(node_data->_data_parm.data) );
	    break;
	case PDU_FTYPE_UINT:
	    printf(", value = %lu)\n", *(unsigned long *) node_data->_data_parm.data);
	    break;
	case PDU_FTYPE_BIT:
	case PDU_FTYPE_BOOL:
	    printf(", value = %s)\n", node_data->_data_parm.data
				      ? "true" : "false");
	    break;
	case PDU_FTYPE_MACADDR:
	    printf(", value = %s)\n",
		   _pb_mac_to_ascii(node_data->_data_parm.data) );
	    break;
	case PDU_FTYPE_DATA:
	    printf(", payload)\n");
	    break;
	case PDU_FTYPE_IP6ADDR:
	    printf(", value = %s)\n", _pb_ip6addr2str(*(ip6_addr_t *) node_data->_data_parm.data) );
	    break;
	case PDU_FTYPE_FIXEDP32:
	    fpn = node_data->_data_parm.data;
	    d = fpn->int_part + fpn->frac_part/pow(2, 16);
	    printf(", value = %f)\n", d);
	    break;
	case PDU_FTYPE_FIXEDP64:
	    fpn = node_data->_data_parm.data;
	    d = fpn->int_part + fpn->frac_part/pow(2, 32);
	    printf(", value = %f)\n", d);
	    break;
	}
    } else if (node_data->type == PDU_NODE_PDU) {
	printf(", hdr: %zu, opts: %zu, payload: %zu)\n",
	       node_data->_data_pdu.hdr_len,
	       node_data->_data_pdu.opts_len,
	       node_data->_data_pdu.payload_len);

    } else
	printf(")\n");

    return FALSE;
}

void
pb_dumppdu(const GNode *tree)
{
    struct node_data *node_data;

    node_data = tree->data;

    g_node_traverse( (GNode *) tree, G_PRE_ORDER, G_TRAVERSE_ALL, -1,
		    dump_traverse_func, NULL);
}

void
pdu_list(void)
{
    printf("\
            PDU Name                 Documented in\n\
------------------------------     -----------------\n");

#if 0
    for (i = 0; pdus[i]; i++)
	printf("%3s     %4s\n",
	       pdus[i]->name,
	       pdus[i]->documented_in);
#endif
}

const char *
pb_getname(const GNode *pdu)
{
    return ( (struct node_data *) pdu->data)->name;
}

/*
 * Returns the next PDU in the PDU tree received as parameter. This
 * next PDU is also known as the payload of the PDU received as parameter.
 */
const GNode *
pb_nextpdu(const GNode *pdu)
{
    const GNode *node;

    node = g_node_last_child( (GNode *) pdu);

    /*
     * "node" can be either NULL (if "pdu" has no children) or non-NULL
     * (if "pdu" has children.) If non-NULL then we need to check the
     * node type to make sure the node is really a PDU node.
     */
    return node && ( (struct node_data *) node->data)->type == PDU_NODE_PDU
	   ? node : NULL;
}

static gboolean
calc_permutations(GNode *node, gpointer data)
{
    struct node_data *node_data;
    int nvalues;

    node_data = node->data;

    /*
     * If node is a parameter node and the type of parameter
     * is a numspec (number or IP address) and the number of
     * values in the numspec is > 1 then we update the permutation
     * count.
     */
    if (node_data->type == PDU_NODE_PARM) {
	switch (node_data->_data_parm.field->type) {
	case PDU_FTYPE_NUMTYPE:
	case PDU_FTYPE_BITFIELD:
	case PDU_FTYPE_UINT8:
	case PDU_FTYPE_UINT16:
	case PDU_FTYPE_UINT32:
	case PDU_FTYPE_IP:
	    if ( (nvalues = num_nvalues(node_data->_data_parm.data) ) > 1)
		*(int *) data *= nvalues;
	    break;
	default:
	    ;
	}
    }

    return FALSE; /* So g_node_traverse() continues */
}

/*
 * Calculates the number of packets that a PDU will generate. This depends on
 * the cartesian product of all the numspecs that change.
 */
int
pb_permutation_count(const GNode *pdu)
{
    int permutation_count;

    permutation_count = 1;

    g_node_traverse( (GNode *) pdu, G_IN_ORDER, G_TRAVERSE_LEAVES, -1,
		    calc_permutations, &permutation_count);

    return permutation_count;
}
