/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2010 Oracle and/or its affiliates. All rights reserved.
 *
 * Oracle and Java are registered trademarks of Oracle and/or its affiliates.
 * Other names may be trademarks of their respective owners.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */
 

package org.netbeans.modules.j2me.cdc.platform.system;

import java.awt.EventQueue;
import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileUtil;
import org.openide.util.RequestProcessor;
import org.openide.windows.IOProvider;
import org.openide.windows.InputOutput;
import org.openide.windows.OutputWriter;

/**
 * Utility to call external executable.
 *
 * @author Ondrej Nekola
 */
public class ExternalProcessing {

    static private final Logger logger = Logger.getLogger(ExternalProcessing.class.getCanonicalName());

    /**
     * Calls an executable and blocks until done.
     *
     * @param executable file tu execute
     * @param workdir working directory set to the execution
     * @param params list of parameters given to the executable
     * @return (standard) output generated by the executable
     * @throws java.io.IOException when execution fails or program exists with nonzero retun value
     */
    static public String callExternal(final FileObject executable, final FileObject workdir, final String... params) throws IOException {
        StringBuffer sb = new StringBuffer();
        try {
            final File executableFile = FileUtil.toFile(executable);
            final String[] command = new String[params.length + 1];
            command[0] = executableFile.getPath();

            for (int i = 0; i < params.length; i++) {
                command[i + 1] = params[i];
            }

            final File workdirFile = FileUtil.toFile(workdir);

            final Process process = Runtime.getRuntime().exec(command, null, workdirFile);
            StreamReader ior = new StreamReader(process.getInputStream(), sb);
            EatingReader irr = new EatingReader(process.getErrorStream());

            ior.join();
            irr.join();
            int exitValue = process.waitFor();
            process.getOutputStream().close();
            if (exitValue != 0) {
                throw new IOException();
            }
        } catch (InterruptedException ex) {
            logger.log(Level.WARNING, ex.getLocalizedMessage(), ex);
        }
        return sb.toString();

    }

    /**
     * The same as @see#callExternal, but it shows NB output window.
     * @param executable
     * @param workDirFo
     * @param params
     * @throws java.io.IOException
     */
    static public void callExternalShowOutput(final FileObject executable, final FileObject workDirFo, final String... params) throws IOException {

        final File executableFile = FileUtil.toFile(executable);
        final String[] command = new String[params.length + 1];
        command[0] = executableFile.getPath();

        for (int i = 0; i < params.length; i++) {
            command[i + 1] = params[i];
        }


        if (EventQueue.isDispatchThread()) {
            throw new IllegalStateException("Cannot be called from Event Dispatch Thread");
        }

        File workDir = FileUtil.toFile(workDirFo);
        final Process process = Runtime.getRuntime().exec(command, null, workDir);

        //Get the standard out
        InputStream out = new BufferedInputStream(process.getInputStream(), 8192);
        //Get the standard in
        InputStream err = new BufferedInputStream(process.getErrorStream(), 8192);

        //Create readers for each
        final Reader outReader = new BufferedReader(new InputStreamReader(out));
        final Reader errReader = new BufferedReader(new InputStreamReader(err));

        StringBuilder sb = new StringBuilder();
        for (String string : command) {
            if (sb.length() > 0) {
                sb.append(" ");
            }
            sb.append(string);
        }

        InputOutput io = IOProvider.getDefault().getIO(sb.toString(), false);

        io.select();

        io.getOut().println(sb); //XXX
        io.getOut().println(); //XXXd

        OutHandler processSystemOut = new OutHandler(outReader, io.getOut());
        OutHandler processSystemErr = new OutHandler(errReader, io.getErr());

        //Get two different threads listening on the output & err
        RequestProcessor.getDefault().post(processSystemOut);
        RequestProcessor.getDefault().post(processSystemErr);

        try {
            //Hang this thread until the process exits
            process.waitFor();
        } catch (InterruptedException ex) {
            logger.log(Level.SEVERE, "Interupted", ex);
        }
        processSystemOut.close();
        processSystemErr.close();

        io.getOut().println("Process exited with exit code " + process.exitValue());
    }


    //The runnable that will handle an output stream, piping it to the output window:
    static class OutHandler implements Runnable {

        private Reader out;
        private OutputWriter writer;

        public OutHandler(Reader out, OutputWriter writer) {
            this.out = out;
            this.writer = writer;
        }

        public void run() {
            while (true) {
                try {
                    while (!out.ready()) {
                        try {
                            Thread.currentThread().sleep(200);
                        } catch (InterruptedException e) {
                            close();
                            return;
                        }
                    }
                    if (!readOneBuffer() || Thread.currentThread().isInterrupted()) {
                        close();
                        return;
                    }
                } catch (IOException ioe) {
                    //Stream already closed, this is fine
                    return;
                }
            }
        }

        private boolean readOneBuffer() throws IOException {
            char[] cbuf = new char[255];
            int read;
            while ((read = out.read(cbuf)) != -1) {
                writer.write(cbuf, 0, read);
            }
            return read != -1;
        }

        private void close() {
            try {
                out.close();
            } catch (IOException ioe) {
                logger.log(Level.SEVERE, "Failed to close stream", ioe);
            } finally {
                writer.close();
            }
        }
    }
}
