/*
 * NASPRO - The NASPRO Architecture for Sound Processing
 * LV2 bridging helper library
 *
 * Copyright (C) 2007-2011 NASPRO Bridge it development team
 *
 * See the COPYING file for license conditions.
 */

#include "internal.h"

_NABRIT_DEF char
nabrit_util_filter_by_suffix(const void *value, void *opaque)
{
	nacore_fs_dir_entry entry;
	const char *suffix;
	const char *s;
	size_t len_s, len_sfx;

	entry = (nacore_fs_dir_entry)value;
	suffix = (const char *)opaque;

	s = nacore_fs_dir_entry_get_name(entry);

	len_s = strlen(s);
	len_sfx = strlen(suffix);

	if (len_s < len_sfx)
		return 0;

	return (strcmp(s + len_s - len_sfx, suffix) == 0);
}

_NABRIT_DEF int
nabrit_util_load_all_in_env_path(nabrit_bridge bridge, const char *name,
				 nacore_filter_cb dir_entry_filter_cb,
				 void *dir_entry_filter_opaque,
				 nacore_op_with_msg_cb load_cb,
				 void *load_opaque,
				 nacore_msg_context msg_context,
				 void *msg_opaque)
{
	const char *path;
	nacore_list prefixes;
	nacore_list_elem elem;
	const char *p;
	nacore_msg_context ctx;
	int errsv;

	path = nacore_env_get(name);
	if (path == NULL)
	  {
		errsv = errno;
		if (errsv == ENOENT)
			nacore_msg_text(msg_context, nacore_msg_severity_err,
					msg_opaque, "The %s environment "
					"variable is not set", name);
		else
			nacore_msg_text(msg_context, nacore_msg_severity_err,
					msg_opaque, "Could not get the value "
					"of environment variable %s", name);
		goto path_err;
	  }

	prefixes = nacore_env_path_prefixes_split(path);
	if (prefixes == NULL)
	  {
		errsv = errno;
		nacore_msg_text(msg_context, nacore_msg_severity_err,
				msg_opaque,"Error while splitting path "
				"prefixes");
		goto prefixes_err;
	  }

	nacore_env_free(path);

	for (elem = nacore_list_get_head(prefixes); elem != NULL;
	     elem = nacore_list_elem_get_next(prefixes, elem))
	  {
		p = nacore_list_elem_get_value(prefixes, elem);

		ctx = nacore_msg_status_begin(msg_context, msg_opaque,
			"Loading all plugin libraries in %s", p);

		nabrit_util_load_all_in_dir(bridge, p, dir_entry_filter_cb,
					    dir_entry_filter_opaque, load_cb,
					    load_opaque, msg_context,
					    msg_opaque);

		nacore_msg_status_end(ctx, nacore_msg_result_ok);
	  }

	nacore_list_free(prefixes, NULL, NULL);

	return 0;

prefixes_err:
	nacore_env_free(path);
path_err:
	return errsv;
}

_NABRIT_DEF int
nabrit_util_load_all_in_dir(nabrit_bridge bridge, const char *dirname,
			    nacore_filter_cb dir_entry_filter_cb,
			    void *dir_entry_filter_opaque,
			    nacore_op_with_msg_cb load_cb, void *load_opaque,
			    nacore_msg_context msg_context, void *msg_opaque)
{
	nacore_fs_dir dir;
	nacore_fs_dir_entry entry;
	const char *name;
	char *filename;
	nacore_msg_context ctx;
	int errsv;

	dir = nacore_fs_dir_open(dirname);
	if (dir == NULL)
	  {
		errsv = errno;
		nacore_msg_text(msg_context, nacore_msg_severity_err,
				msg_opaque, "Could not open directory %s",
				dirname);
		return errsv;
	  }

	for (entry = nacore_fs_dir_get_next_entry(dir); entry != NULL;
	     nacore_fs_dir_entry_free(entry),
	     entry = nacore_fs_dir_get_next_entry(dir))
	  {
		name = nacore_fs_dir_entry_get_name(entry);

		if (dir_entry_filter_cb != NULL)
			if (dir_entry_filter_cb(entry, dir_entry_filter_opaque)
			    == 0)
			  {
				nacore_msg_text(msg_context,
					nacore_msg_severity_info, msg_opaque,
					"%s%s%s skipped", dirname,
					nacore_fs_dir_sep, name);
				continue;
			  }

		nacore_asprintf(&filename, "%s%s%s", dirname, nacore_fs_dir_sep,
				name);
		if (filename == NULL)
		  {
			nacore_msg_text(msg_context, nacore_msg_severity_err,
					msg_opaque, "Not enough memory");
			continue;
		  }

		ctx = nacore_msg_status_begin(msg_context, msg_opaque,
			"Loading plugin library %s", filename);

		load_cb(filename, ctx, msg_opaque, load_opaque);

		nacore_msg_status_end(ctx, nacore_msg_result_ok);
	  }

	nacore_fs_dir_close(dir);

	return 0;
}
