--------------------------------------------------------------------------
--                                                                      --
--           Copyright: Copyright (C) 2000-2010 CNRS/IN2P3              --
--                                                                      --
-- Narval framework is free  software; you can redistribute  it and/or  --
-- modify  it   under  terms  of  the  GNU General  Public  License as  --
-- published  by  the  Free Software Foundation; either version  2, or  --
-- (at your option) any later version. Narval framework is distributed  --
-- in the hope  that  they will  be useful, but  WITHOUT ANY WARRANTY;  --
-- without even the implied warranty of  MERCHANTABILITY or FITNESS FOR --
-- A PARTICULAR PURPOSE. See the  GNU. General Public License for more  --
-- details. You should have received  a copy of the GNU General Public  --
-- License distributed with Narval; see file COPYING. If not, write to  --
-- the Free Software  Foundation,  Inc., 51 Franklin St,  Fifth Floor,  --
-- Boston, MA 02110-1301 USA.                                           --
--------------------------------------------------------------------------
with Ada.Exceptions;
with System.RPC;

with Narval.Partitions;

with Log4ada.Loggers;
with Log4ada.Appenders.Consoles;
with Log4ada.Appenders.Annex_E;

package body Narval.Narval_Naming_Registry is

   Partitions_List : Partitions.Partition_Shell_Type;
   Logger : aliased Log4ada.Loggers.Logger_Type;
   Console : aliased Log4ada.Appenders.Consoles.Console_Type;
   Remote_Appender : aliased Log4ada.Appenders.Annex_E.Remote_Appender_Type;

   --------------------------
   -- Ajouter_Sous_Systeme --
   --------------------------

   procedure Add_Sub_System
     (Sub_System_Name : String;
      Host_Name : String;
      Config : Configurator.Abstract_Actors_Coordination.Configuration_Access)
   is
   begin
      if Partitions_List.Exist (Sub_System_Name) then
         Ada.Exceptions.Raise_Exception
           (Sub_System_Name_Already_Used'Identity,
            Sub_System_Name & " is already used by another subsystem");
      end if;
      Partitions_List.Add (Name => Sub_System_Name,
                           Config => Config,
                           Host_Name => Host_Name);
      Log4ada.Loggers.Info_Out (Logger'Access, "new sub system " &
                                Sub_System_Name & " subcribed from " &
                                Host_Name);
   end Add_Sub_System;

   --------------------------
   -- Enlever_Sous_Systeme --
   --------------------------

   procedure Remove_Sub_System (Sub_System_Name : String) is
   begin
      Partitions_List.Remove (Name => Sub_System_Name);
      Log4ada.Loggers.Info_Out (Logger'Access, "sub system " &
                                Sub_System_Name & " removed");
   end Remove_Sub_System;

   ---------------------------------
   -- Recuperer_Noms_Sous_Systeme --
   ---------------------------------

   function Get_Sub_Systems_Names return String is
   begin
      return Partitions_List.Name_All_Partitions;
   end Get_Sub_Systems_Names;

   ----------------------------
   -- Configuration_Courante --
   ----------------------------

   function Get_Configuration
     (Name : String)
      return Configurator.Abstract_Actors_Coordination.Configuration_Access
   is
   begin
      return Partitions_List.Configuration (Name);
   end Get_Configuration;

   -------------------------------------
   -- Nombre_Configurations_Presentes --
   -------------------------------------

   function Number_Of_Sub_Systems return Natural is
   begin
      return Partitions_List.Number_Of_Partitions;
   end Number_Of_Sub_Systems;

   ---------------------
   -- Is_This_The_End --
   ---------------------

   This_Is_The_End : Boolean := False;
   pragma Atomic (This_Is_The_End);

   function Is_This_The_End return Boolean is
   begin
      return This_Is_The_End;
   end Is_This_The_End;

   -----------------------
   -- Finir_Application --
   -----------------------

   procedure End_Registry_Service is
   begin
      if Partitions_List.Number_Of_Partitions /= 0 then
         raise Unable_To_End_Registry_Service;
      end if;
      Log4ada.Loggers.Warn_Out (Logger'Access,
                                "narval naming registry received end order");
      This_Is_The_End := True;
   end End_Registry_Service;

   Local_Event_Receiver :
     Events_Receivers.Events_Receivers_Class_Access := null;
   pragma Atomic (Local_Event_Receiver);

   ---------------
   -- Subscribe --
   ---------------

   procedure Subscribe
     (Event_Receiver : Events_Receivers.Events_Receivers_Class_Access) is
      use type Events_Receivers.Events_Receivers_Class_Access;
      Events_Receiver_Alive : Boolean := True;
      Need_Trig : Boolean := False;
   begin
      if Local_Event_Receiver /= null then
         --  test if event receiver is alive
         begin
            Events_Receiver_Alive := Events_Receivers.Event_Receiver_Is_Alive
              (Local_Event_Receiver);
         exception
            when System.RPC.Communication_Error =>
               Log4ada.Appenders.Annex_E.Disable (Remote_Appender'Access);
               Events_Receiver_Alive := False;
               Need_Trig := True;
         end;
         if Events_Receiver_Alive then
            Log4ada.Loggers.Warn_Out (Logger'Access,
                                      "somebody try to launch" &
                                      " a second central log");
            Ada.Exceptions.Raise_Exception
              (Already_Existent_Receiver'Identity,
               "cannot subscribe two event receiver");
         end if;
      end if;
      Local_Event_Receiver := Event_Receiver;
      Log4ada.Appenders.Annex_E.Set_Receiver (Remote_Appender, Event_Receiver);
      Log4ada.Appenders.Annex_E.Enable (Remote_Appender'Access);
      Log4ada.Loggers.Info_Out (Logger'Access, "remote logging on");
      if Need_Trig then
         Partitions_List.Trig_Event ("log_reload");
      end if;
   end Subscribe;

   ------------------------
   -- Get_Event_Receiver --
   ------------------------

   function Get_Event_Receiver
     return Events_Receivers.Events_Receivers_Class_Access is
      use type Events_Receivers.Events_Receivers_Class_Access;
   begin
      if Local_Event_Receiver = null then
         raise No_Event_Receiver;
      end if;
      return Local_Event_Receiver;
   end Get_Event_Receiver;

begin
   Log4ada.Loggers.Set_Name (Logger'Access, "narval_naming_registry");
   Log4ada.Loggers.Set_Level (Logger'Access, Log4ada.All_Level);
   Log4ada.Loggers.Add_Appender (Logger'Access, Console'Access);
   Log4ada.Loggers.Add_Appender (Logger'Access, Remote_Appender'Access);
   Log4ada.Appenders.Annex_E.Disable (Remote_Appender'Access);
end Narval.Narval_Naming_Registry;
