--------------------------------------------------------------------------
--                                                                      --
--           Copyright: Copyright (C) 2000-2010 CNRS/IN2P3              --
--                                                                      --
-- Narval framework is free  software; you can redistribute  it and/or  --
-- modify  it   under  terms  of  the  GNU General  Public  License as  --
-- published  by  the  Free Software Foundation; either version  2, or  --
-- (at your option) any later version. Narval framework is distributed  --
-- in the hope  that  they will  be useful, but  WITHOUT ANY WARRANTY;  --
-- without even the implied warranty of  MERCHANTABILITY or FITNESS FOR --
-- A PARTICULAR PURPOSE. See the  GNU. General Public License for more  --
-- details. You should have received  a copy of the GNU General Public  --
-- License distributed with Narval; see file COPYING. If not, write to  --
-- the Free Software  Foundation,  Inc., 51 Franklin St,  Fifth Floor,  --
-- Boston, MA 02110-1301 USA.                                           --
--------------------------------------------------------------------------
with Ada.Exceptions;
with Ada.Text_IO;
with Ada.Characters.Handling;
with McKae.XML.EZ_Out.String_Stream;
with GNAT.Sockets;

with DOM.Core.Attrs;
with DOM.Core.Nodes;
with DOM.Readers;

with Input_Sources.File;
with Input_Sources.Strings;
with Unicode.CES.Utf8;

package body Narval.Configurator.Actors_Description is

   use DOM.Core.Nodes;
   use type DOM.Core.Node;
   use DOM.Core.Attrs;

   procedure Check_Inputs (Actor : Actor_Description_Type);
   procedure Fill_Inputs (Actor : Actor_Description_Type);
   procedure Check_Inputs (Actor_Position : Actors_Description_Vector.Cursor);

   ----------
   -- Fill --
   ----------

   procedure Fill (Actor : in out Actor_Description_Type;
                   Informations : String) is
      Input : Input_Sources.Strings.String_Input;
      String_Reader : DOM.Readers.Tree_Reader;
      String_Input : aliased String := Informations;
   begin
      Input_Sources.Strings.Open (String_Input'Unchecked_Access,
                                  Unicode.CES.Utf8.Utf8_Encoding,
                                  Input);
      DOM.Readers.Parse (String_Reader, Input);
      Input_Sources.Strings.Close (Input);
      Fill (DOM.Core.Nodes.First_Child
              (DOM.Readers.Get_Tree (String_Reader)),
            Actor);
   end Fill;

   procedure Fill (Node_Root : DOM.Core.Node;
                   Actor : in out Actor_Description_Type) is
      Attribute : DOM.Core.Attr;
      List : DOM.Core.Node_List;
      Node : DOM.Core.Node;
      Receiver_Name : Unbounded_String;
      Receiver : Receiver_Descriptor_Access;
   begin
      Attribute := Get_Named_Item (Attributes (Node_Root), "input_buffers");
      if Attribute /= null then
         Actor.Declared_Inputs := Natural'Value (Value (Attribute));
      end if;
      Attribute := Get_Named_Item (Attributes (Node_Root), "output_buffers");
      if Attribute /= null then
         Actor.Declared_Outputs := Natural'Value (Value (Attribute));
      end if;
      if Node_Name (Node_Root) = "producer" then
         Actor.Kind := A_Producer;
      elsif Node_Name (Node_Root) = "consumer" then
         Actor.Kind := A_Consumer;
      elsif Node_Name (Node_Root) = "stand_alone_actor" then
         Actor.Kind := A_Stand_Alone_Actor;
      elsif Node_Name (Node_Root) = "intermediary" then
         Actor.Kind := An_Intermediary;
      end if;
      List := Child_Nodes (Node_Root);
      for I in 0 .. Length (List) - 1 loop
         Node := Item (List, I);
         declare
            Xml_Node_Name : constant String :=
              Ada.Characters.Handling.To_Lower (Node_Name (Node));
         begin
            if Xml_Node_Name = "name" then
               Actor.Name := To_Unbounded_String
                 (Node_Value (First_Child (Node)));
            elsif Xml_Node_Name = "rank" then
               Actor.Rank_Number := Rank_Type'Value
                 (Node_Value (First_Child (Node)));
            elsif Xml_Node_Name = "hostname" then
               if Node_Value (First_Child (Node)) = "localhost" then
                  Actor.Host_Name := To_Unbounded_String
                    (GNAT.Sockets.Host_Name);
               else
                  Actor.Host_Name := To_Unbounded_String
                    (Node_Value (First_Child (Node)));
               end if;
            elsif Xml_Node_Name = "binary_code" then
               Actor.Process_Name := To_Unbounded_String
                 (Node_Value (First_Child (Node)));
            elsif Xml_Node_Name = "output_buffer_name" then
               Senders_Vector.Append
                 (Actor.Outputs,
                  new Sender_Descriptor_Type'
                    (Name => To_Unbounded_String
                       (Node_Value (First_Child (Node))),
                     Size => 0,
                     Port => Null_Unbounded_String,
                     Consumers_Number => 0));
            elsif Xml_Node_Name = "size" then
               Attribute := Get_Named_Item (Attributes (Node),
                                            "output_buffer");
               if Attribute = null then
                  raise Configuration_Failed with
                    "size must have output_buffer attribute";
               end if;
               declare
                  Cursor : constant Senders_Vector.Cursor :=
                    Find_Output (Actor.Outputs,
                                 To_Unbounded_String (Value (Attribute)));
                  use type Senders_Vector.Cursor;
               begin
                  if Cursor = Senders_Vector.No_Element then
                     raise Configuration_Failed
                       with Value (Attribute) & " unknown output_buffer";
                  end if;
                  Senders_Vector.Element (Cursor).Size :=
                    Positive'Value (Node_Value (First_Child (Node)));
               end;
            elsif Xml_Node_Name = "port" then
               Attribute := Get_Named_Item (Attributes (Node),
                                            "output_buffer");
               if Attribute = null then
                  raise Configuration_Failed with
                    "port must have output_buffer attribute";
               end if;
               declare
                  Cursor : constant Senders_Vector.Cursor :=
                    Find_Output (Actor.Outputs,
                                 To_Unbounded_String (Value (Attribute)));
                  use type Senders_Vector.Cursor;
               begin
                  if Cursor = Senders_Vector.No_Element then
                     raise Configuration_Failed
                       with Value (Attribute) & " unknown output_buffer";
                  end if;
                  Senders_Vector.Element (Cursor).Port :=
                    To_Unbounded_String (Node_Value (First_Child (Node)));
               end;
            elsif Xml_Node_Name = "consumers_number" then
               if Actor.Parent_Vector /= null then
                  raise Configuration_Failed with
                    "consumers_number tag is reserved for " &
                    "communication, not configuration";
               end if;
               Attribute := Get_Named_Item (Attributes (Node),
                                            "output_buffer");
               if Attribute = null then
                  raise Configuration_Failed with
                    "consumers_number must have output_buffer attribute";
               end if;
               declare
                  Cursor : constant Senders_Vector.Cursor :=
                    Find_Output (Actor.Outputs,
                                 To_Unbounded_String (Value (Attribute)));
                  use type Senders_Vector.Cursor;
               begin
                  if Cursor = Senders_Vector.No_Element then
                     raise Configuration_Failed
                       with Value (Attribute) & " unknown output_buffer";
                  end if;
                  Senders_Vector.Element (Cursor).Consumers_Number :=
                    Actor_Numbering_Type'Value
                    (Node_Value (First_Child (Node)));
               end;
            elsif Xml_Node_Name = "argument" then
               String_Vector.Append (Actor.Arguments,
                                     To_Unbounded_String
                                       (Node_Value (First_Child (Node))));
            elsif Xml_Node_Name = "debug" or Xml_Node_Name = "log_level" then
               Actor.Log_Level := Log4ada.Level_Type'Value
                 (Node_Value (First_Child (Node)));
            elsif Xml_Node_Name = "data_source" then
               Attribute := Get_Named_Item (Attributes (Node),
                                            "source_buffer");
               if Attribute = null then
                  raise Configuration_Failed
                    with "data source need source buffer";
               end if;
               Receiver_Name := To_Unbounded_String (Value (Attribute));
               Attribute := Get_Named_Item (Attributes (Node), "source_port");
               if Attribute = null then
                  raise Configuration_Failed
                    with "data source need source port";
               end if;
               Receiver := new Receiver_Descriptor_Type;
               Receiver.Name := Receiver_Name;
               Receiver.Port := To_Unbounded_String (Value (Attribute));
               Receiver.Data_Source := To_Unbounded_String
                 (Node_Value (First_Child (Node)));
               Receivers_Vector.Append (Actor.Inputs, Receiver);
               if Actor.Parent_Vector = null then
                  Attribute := Get_Named_Item (Attributes (Node),
                                               "size");
                  if Attribute = null then
                     raise Configuration_Failed
                       with "data source need size in this context";
                  end if;
                  Receiver.Size := Natural'Value (Value (Attribute));
               end if;
            elsif Xml_Node_Name = "service" then
               raise Configuration_Failed
                 with "service tag no longer supported";
            elsif Xml_Node_Name = "#text" then
               null;
            else
               raise Configuration_Failed
                 with "fill_actor " & Xml_Node_Name & " : unknown tag";
            end if;
         end;
      end loop;
      if Actor.Parent_Vector /= null then
         Fill_Inputs (Actor);
      end if;
   end Fill;

   procedure Fill (Actors : in out Actors_Description_Vector.Vector;
                   File_Name : String) is
      File_Reader : DOM.Readers.Tree_Reader;
      File : Input_Sources.File.File_Input;
   begin
      Input_Sources.File.Open (File_Name, File);
      DOM.Readers.Parse (File_Reader, File);
      Input_Sources.File.Close (File);
      Fill (DOM.Readers.Get_Tree (File_Reader), Actors);
   end Fill;

   procedure Fill (Node_Root : DOM.Core.Node;
                   Actors : in out Actors_Description_Vector.Vector) is
      List : DOM.Core.Node_List;
      Node : DOM.Core.Node;
      Node_Child : constant DOM.Core.Node := First_Child (Node_Root);
      Actor : Actor_Description_Access;
   begin
      List := Child_Nodes (Node_Child);
      for I in 0 .. Length (List) - 1 loop
         Node := Item (List, I);
         declare
            Xml_Node_Name : constant String :=
              Ada.Characters.Handling.To_Lower (Node_Name (Node));
         begin
            if Xml_Node_Name = "producer" or
              Xml_Node_Name = "consumer" or
              Xml_Node_Name = "intermediary" or
              Xml_Node_Name = "stand_alone_actor" then
               Actor := new Actor_Description_Type;
               Actor.Task_Register := new Register_Synchro_Task;
               Actor.Parent_Vector := Actors'Unchecked_Access;
               Fill (Node, Actor.all);
               Actors_Description_Vector.Append (Actors, Actor);
            elsif Xml_Node_Name = "#text" then
               null;
            else
               raise Configuration_Failed
                 with "fill_vector : " & Xml_Node_Name & " unknown tag";
            end if;
         end;
      end loop;
   end Fill;

   procedure Check_Inputs (Actor : Actor_Description_Type) is
      Cursor : Receivers_Vector.Cursor;
      use type Receivers_Vector.Cursor;
   begin
      Cursor := Receivers_Vector.First (Actor.Inputs);
      loop
         exit when Cursor = Receivers_Vector.No_Element;
         Cursor := Receivers_Vector.Next (Cursor);
         raise Program_Error;
      end loop;
   end Check_Inputs;

   procedure Check_Inputs
     (Actor_Position : Actors_Description_Vector.Cursor) is
      Actor : constant Actor_Description_Access :=
        Actors_Description_Vector.Element (Actor_Position);
   begin
      Check_Inputs (Actor.all);
   end Check_Inputs;

   procedure Fill_Inputs (Actor : Actor_Description_Type) is
      Cursor : Receivers_Vector.Cursor;
      use type Receivers_Vector.Cursor;
   begin
      Cursor := Receivers_Vector.First (Actor.Inputs);
      loop
         exit when Cursor = Receivers_Vector.No_Element;
         declare
            Input : constant Receiver_Descriptor_Access :=
              Receivers_Vector.Element (Cursor);
         begin
            declare
               Data_Source_Actor : constant Actor_Description_Access :=
                 Find_Actor (Actor.Parent_Vector.all, Input.Data_Source);
               Output_Cursor : constant Senders_Vector.Cursor :=
                 Find_Output (Data_Source_Actor.Outputs,
                              Input.Name);
               use type Senders_Vector.Cursor;
            begin
               if Output_Cursor = Senders_Vector.No_Element then
                  raise Configuration_Failed with
                    "output named :" & To_String (Input.Name) &
                    ", doen't exist for data source :" &
                    To_String (Input.Data_Source);
               end if;
               if Senders_Vector.Element
                 (Output_Cursor).Port /= Input.Port then
                  raise Configuration_Failed with
                    "input named :" & To_String (Input.Name) &
                    " use " & To_String (Input.Port) & " instead of " &
                    To_String (Senders_Vector.Element (Output_Cursor).Port);
               end if;
               Input.Size := Senders_Vector.Element (Output_Cursor).Size;
               Senders_Vector.Element (Output_Cursor).Consumers_Number :=
                 Senders_Vector.Element (Output_Cursor).Consumers_Number + 1;
            end;
         exception
            when Find_Failure =>
               raise Configuration_Failed with
                 "input named :" & To_String (Input.Name) &
                 ", data source not found :" & To_String (Input.Data_Source);
         end;
         Cursor := Receivers_Vector.Next (Cursor);
      end loop;
   end Fill_Inputs;

   --------------------
   -- Validity_Check --
   --------------------

   procedure Validity_Check
     (Actor_Position : Actors_Description_Vector.Cursor) is
      Actor : constant Actor_Description_Access :=
        Actors_Description_Vector.Element (Actor_Position);
   begin
      Validity_Check (Actor.all);
   end Validity_Check;

   procedure Validity_Check (Actor : Actor_Description_Type)
   is
      use Ada.Exceptions;
      Inputs_Length : constant Natural :=
        Natural (Receivers_Vector.Length (Actor.Inputs));
      Outputs_Length : constant Natural :=
        Natural (Senders_Vector.Length (Actor.Outputs));
   begin
      if Actor.Name = Null_Unbounded_String then
         raise Actor_Without_Name;
      end if;
      if Actor.Host_Name = Null_Unbounded_String then
         Raise_Exception (Actor_Without_Host_Name'Identity, "actor : " &
                            To_String (Actor.Name));
      end if;
      if Actor.Process_Name = Null_Unbounded_String then
         Raise_Exception (Actor_Without_Process_Name'Identity, "actor : " &
                            To_String (Actor.Name));
      end if;
      if not Actor.Check_All then
         return;
      end if;
      if Actor.Declared_Inputs /= Inputs_Length then
         raise Configuration_Failed with
           "declared inputs" & Actor.Declared_Inputs'Img &
           "/= of inputs length" & Inputs_Length'Img;
      end if;
      if Actor.Declared_Outputs /= Outputs_Length then
         raise Configuration_Failed with
           "declared outputs" & Actor.Declared_Inputs'Img &
           "/= of outputs length" & Outputs_Length'Img;
      end if;
      case Actor.Kind is
         when A_Stand_Alone_Actor =>
            null;
         when A_Producer =>
            Senders_Vector.Iterate (Actor.Outputs, Validity_Check'Access);
         when A_Consumer =>
            Receivers_Vector.Iterate (Actor.Inputs, Validity_Check'Access);
         when An_Intermediary =>
            Senders_Vector.Iterate (Actor.Outputs, Validity_Check'Access);
            Receivers_Vector.Iterate (Actor.Inputs, Validity_Check'Access);
      end case;
   end Validity_Check;

   -----------------
   -- Rank_Calcul --
   -----------------

   procedure Rank_Calcul
     (Actor_Position : Actors_Description_Vector.Cursor)
   is
      Actor : constant Actor_Description_Access :=
        Actors_Description_Vector.Element (Actor_Position);
      Cursor : Receivers_Vector.Cursor;
      use type Receivers_Vector.Cursor;
      Producer_Position : Actors_Description_Vector.Cursor;
      Current_Rank : Rank_Type;
      Max_Rank : Rank_Type := -1;
   begin
      if Actor.Rank_Calcul_Running then
         raise Actors_Loop;
      end if;
      if Actor.Rank_Number /= -1 then
         return;
      end if;
      if Actor.Kind = A_Producer then
         Actor.Rank_Number := 1;
         return;
      end if;
      Actor.Rank_Calcul_Running := True;
      Cursor := Receivers_Vector.First (Actor.Inputs);
      loop
         exit when Cursor = Receivers_Vector.No_Element;
         Producer_Position := Find_Actor
           (Actor.Parent_Vector.all,
            Receivers_Vector.Element (Cursor).Data_Source);
         if Actors_Description_Vector.Element
           (Producer_Position).Rank_Number = -1 then
            Rank_Calcul (Producer_Position);
         end if;
         Current_Rank := Actors_Description_Vector.Element
           (Producer_Position).Rank_Number + 1;
         if Current_Rank > Max_Rank then
            Max_Rank := Current_Rank;
         end if;
         Cursor := Receivers_Vector.Next (Cursor);
      end loop;
      if Max_Rank = -1 then
         Max_Rank := 1;
      end if;
      Actor.Rank_Number := Max_Rank;
      Actor.Rank_Calcul_Running := False;
   end Rank_Calcul;

   -------------
   -- Display --
   -------------

   procedure Display (Actor : Actor_Description_Type) is
      procedure Display_Output (Output_Position : Senders_Vector.Cursor);
      procedure Display_Input (Input_Position : Receivers_Vector.Cursor);
      procedure Display_Output (Output_Position : Senders_Vector.Cursor) is
         Output : constant Sender_Descriptor_Access :=
           Senders_Vector.Element (Output_Position);
         use type Senders_Vector.Cursor;
      begin
         Ada.Text_IO.Put_Line ("output named :" & To_String (Output.Name) &
                                 Output.Size'Img & "," &
                                 To_String (Output.Port) &
                                 Output.Consumers_Number'Img);
      end Display_Output;
      procedure Display_Input (Input_Position : Receivers_Vector.Cursor) is
         Input : constant Receiver_Descriptor_Access :=
           Receivers_Vector.Element (Input_Position);
         use type Receivers_Vector.Cursor;
      begin
         Ada.Text_IO.Put_Line ("input named :" &
                                 To_String (Input.Name) & Input.Size'Img &
                                 "," & To_String (Input.Port));
      end Display_Input;
   begin
      Ada.Text_IO.Put_Line ("name : " & To_String (Actor.Name));
      Ada.Text_IO.Put_Line ("kind : " & Actor.Kind'Img);
      Ada.Text_IO.Put_Line ("rank :" & Actor.Rank_Number'Img);
      Senders_Vector.Iterate (Actor.Outputs, Display_Output'Access);
      Receivers_Vector.Iterate (Actor.Inputs, Display_Input'Access);
   end Display;

   procedure Display (Actor_Position : Actors_Description_Vector.Cursor) is
      Actor : constant Actor_Description_Access :=
        Actors_Description_Vector.Element (Actor_Position);
   begin
      Display (Actor.all);
   end Display;

   procedure Validity_Check (Sender_Position : Senders_Vector.Cursor) is
      Output : constant Sender_Descriptor_Access :=
        Senders_Vector.Element (Sender_Position);
   begin
      if Output.Name = Null_Unbounded_String then
         raise Unamed_Buffer;
      end if;
      if Output.Consumers_Number = 0 then
         raise Data_Source_Unused;
      end if;
      if Output.Size = 0 then
         raise Buffer_Size_Unset;
      end if;
   end Validity_Check;

   procedure Validity_Check (Receiver_Position : Receivers_Vector.Cursor) is
      Input : constant Receiver_Descriptor_Access :=
        Receivers_Vector.Element (Receiver_Position);
   begin
      if Input.Data_Source = Null_Unbounded_String then
         raise Data_Source_Missing;
      end if;
   end Validity_Check;

   --------------------
   -- Validity_Check --
   --------------------

   procedure Validity_Check (Actors : Actors_Description_Vector.Vector) is
   begin
      Actors_Description_Vector.Iterate (Actors, Check_Inputs'Access);
      Actors_Description_Vector.Iterate (Actors, Validity_Check'Access);
   end Validity_Check;

   -----------------
   -- Rank_Calcul --
   -----------------

   procedure Rank_Calcul (Actors : Actors_Description_Vector.Vector) is
   begin
      Actors_Description_Vector.Iterate (Actors, Rank_Calcul'Access);
   end Rank_Calcul;

   -------------
   -- Display --
   -------------

   procedure Display (Actors : Actors_Description_Vector.Vector) is
   begin
      Actors_Description_Vector.Iterate (Actors, Display'Access);
   end Display;

   function Find_Actor (Actors : Actors_Description_Vector.Vector;
                        Actor_Name : Unbounded_String)
                       return Actor_Description_Access is
      Cursor : Actors_Description_Vector.Cursor;
      use type Actors_Description_Vector.Cursor;
   begin
      Cursor := Find_Actor (Actors, Actor_Name);
      if Cursor = Actors_Description_Vector.No_Element then
         raise Find_Failure;
      end if;
      return Actors_Description_Vector.Element (Cursor);
   end Find_Actor;

   function Selection (Item : Receiver_Descriptor_Access;
                       Select_On : Unbounded_String) return Boolean is
   begin
      return Item.Name = Select_On;
   end Selection;

   function Selection (Item : Sender_Descriptor_Access;
                       Select_On : Unbounded_String) return Boolean is
   begin
      return Item.Name = Select_On;
   end Selection;

   function Selection (Item : Actor_Description_Access;
                       Select_On : Unbounded_String) return Boolean is
   begin
      return Item.Name = Select_On;
   end Selection;

   task body Register_Synchro_Task is
      Actor_Pointer : Actors.Actor_Class_Access := null;
   begin
      loop
         select
            accept Post (Pointer : Actors.Actor_Class_Access) do
               Actor_Pointer := Pointer;
            end Post;
         or
            terminate;
         end select;
         loop
            select
               accept Clear;
               Actor_Pointer := null;
               exit;
            or
               accept Get_Pointer (Pointer : out Actors.Actor_Class_Access) do
                  Pointer := Actor_Pointer;
               end Get_Pointer;
            or
               terminate;
            end select;
         end loop;
      end loop;
   end Register_Synchro_Task;

   function To_String (Actor : Actor_Description_Type) return String is
      use McKae.XML.EZ_Out.String_Stream.String_Buffering;
      use McKae.XML.EZ_Out.String_Stream.XML_String_Buffer;
      Xml_Buffer : String_Buffer;
      Kind_String : Unbounded_String;
      Attributes : Attributes_List (1 .. 2);
      Output_Length : constant Natural :=
        Natural (Senders_Vector.Length (Actor.Outputs));
      Input_Length : constant Natural :=
        Natural (Receivers_Vector.Length (Actor.Inputs));
      procedure Add_Argument (Argument_Position : String_Vector.Cursor);
      procedure Add_Input (Input_Position : Receivers_Vector.Cursor);
      procedure Add_Output (Output_Position : Senders_Vector.Cursor);
      procedure Add_Argument (Argument_Position : String_Vector.Cursor) is
         use String_Vector;
      begin
         Output_Element (Xml_Buffer,
                         "argument",
                         To_String (Element (Argument_Position)));
      end Add_Argument;
      procedure Add_Input (Input_Position : Receivers_Vector.Cursor) is
         use Receivers_Vector;
         Input : constant Receiver_Descriptor_Access :=
           Element (Input_Position);
      begin
         Output_Element (Xml_Buffer,
                         "data_source",
                         To_String (Input.Data_Source),
                         ("source_port" = To_String (Input.Port),
                          "source_buffer" = To_String (Input.Name),
                          "size" = Input.Size));
      end Add_Input;
      procedure Add_Output (Output_Position : Senders_Vector.Cursor) is
         use Senders_Vector;
         Output : constant Sender_Descriptor_Access :=
           Element (Output_Position);
      begin
         Output_Element (Xml_Buffer,
                         "output_buffer_name",
                         To_String (Output.Name));
         Output_Element (Xml_Buffer,
                         "size",
                         Output.Size'Img,
                         ("output_buffer" = Output.Name));
         Output_Element (Xml_Buffer,
                         "port",
                         To_String (Output.Port),
                         ("output_buffer" = Output.Name));
         Output_Element (Xml_Buffer,
                         "consumers_number",
                         Output.Consumers_Number'Img,
                         ("output_buffer" = Output.Name));
      end Add_Output;
   begin
      case Actor.Kind is
         when A_Producer =>
            Kind_String := To_Unbounded_String ("producer");
            Attributes := (1 => "output_buffers" = Output_Length,
                           2 => "input_buffers" = 0);
         when A_Consumer =>
            Kind_String := To_Unbounded_String ("consumer");
            Attributes := (1 => "output_buffers" = 0,
                           2 => "input_buffers" = Input_Length);
         when An_Intermediary =>
            Kind_String := To_Unbounded_String ("intermediary");
            Attributes := (1 => "output_buffers" = Output_Length,
                           2 => "input_buffers" = Input_Length);
         when A_Stand_Alone_Actor =>
            Kind_String := To_Unbounded_String ("stand_alone_actor");
      end case;
      Start_Element (Xml_Buffer, To_String (Kind_String));
      Output_Element (Xml_Buffer, "name", To_String (Actor.Name), Attributes);
      String_Vector.Iterate (Actor.Arguments, Add_Argument'Access);
      Output_Element (Xml_Buffer, "rank", Actor.Rank_Number'Img);
      Output_Element (Xml_Buffer, "log_level", Actor.Log_Level'Img);
      Receivers_Vector.Iterate (Actor.Inputs, Add_Input'Access);
      Senders_Vector.Iterate (Actor.Outputs, Add_Output'Access);
      End_Element (Xml_Buffer, To_String (Kind_String));
      return Get_String (Xml_Buffer);
   end To_String;

   procedure Free (Receivers : in out Receivers_Vector.Vector) is
      Cursor : Receivers_Vector.Cursor;
      use type Receivers_Vector.Cursor;
      Receiver : Receiver_Descriptor_Access;
   begin
      Cursor := Receivers_Vector.First (Receivers);
      loop
         exit when Cursor = Receivers_Vector.No_Element;
         Receiver := Receivers_Vector.Element (Cursor);
         Free (Receiver);
         Cursor := Receivers_Vector.Next (Cursor);
      end loop;
      Receivers_Vector.Clear (Receivers);
   end Free;

   procedure Free (Senders : in out Senders_Vector.Vector) is
      Cursor : Senders_Vector.Cursor;
      use type Senders_Vector.Cursor;
      Sender : Sender_Descriptor_Access;
   begin
      Cursor := Senders_Vector.First (Senders);
      loop
         exit when Cursor = Senders_Vector.No_Element;
         Sender := Senders_Vector.Element (Cursor);
         Free (Sender);
         Cursor := Senders_Vector.Next (Cursor);
      end loop;
      Senders_Vector.Clear (Senders);
   end Free;

   procedure Free (Actors : in out Actors_Description_Vector.Vector) is
      Cursor : Actors_Description_Vector.Cursor;
      use type Actors_Description_Vector.Cursor;
   begin
      Cursor := Actors_Description_Vector.First (Actors);
      loop
         exit when Cursor = Actors_Description_Vector.No_Element;
         declare
            Actor : Actor_Description_Access :=
              Actors_Description_Vector.Element (Cursor);
         begin
            Free (Actor.Inputs);
            Free (Actor.Outputs);
            String_Vector.Clear (Actor.Arguments);
            Free (Actor.Task_Register);
            Free (Actor);
         end;
         Cursor := Actors_Description_Vector.Next (Cursor);
      end loop;
      Actors_Description_Vector.Clear (Actors);
   end Free;

   function "<" (Left, Right : Actor_Description_Access) return Boolean is
   begin
      return Left.Rank_Number < Right.Rank_Number;
   end "<";

end Narval.Configurator.Actors_Description;
