--------------------------------------------------------------------------
--                                                                      --
--           Copyright: Copyright (C) 2000-2010 CNRS/IN2P3              --
--                                                                      --
-- Narval framework is free  software; you can redistribute  it and/or  --
-- modify  it   under  terms  of  the  GNU General  Public  License as  --
-- published  by  the  Free Software Foundation; either version  2, or  --
-- (at your option) any later version. Narval framework is distributed  --
-- in the hope  that  they will  be useful, but  WITHOUT ANY WARRANTY;  --
-- without even the implied warranty of  MERCHANTABILITY or FITNESS FOR --
-- A PARTICULAR PURPOSE. See the  GNU. General Public License for more  --
-- details. You should have received  a copy of the GNU General Public  --
-- License distributed with Narval; see file COPYING. If not, write to  --
-- the Free Software  Foundation,  Inc., 51 Franklin St,  Fifth Floor,  --
-- Boston, MA 02110-1301 USA.                                           --
--------------------------------------------------------------------------
with System.Storage_Elements;

package body Narval.Actors.Actives.Consumers.Data_Rate is

   use Log4ada.Loggers;

   type Data_Rate_Info_Type is record
      Difference : Duration;
      Data_Rate_Bytes_Per_Second : Long_Float;
      Data_Rate_Mega_Bytes_Per_Second : Long_Float;
   end record;

   function Data_Rate (Object : access Data_Rate_Consumer_Type)
                  return Data_Rate_Info_Type;

   task body Data_Rate_Calculation_Task_Type is
      Infos : Data_Rate_Info_Type;
   begin
      loop
         select
            accept Start;
         or
            terminate;
         end select;
         loop
            select
               accept Stop;
               exit;
            or
               delay 2.0;
               Infos := Data_Rate (Wrapper);
               Wrapper.Data_Rate_Parameter.Long_Float_Value :=
                 Infos.Data_Rate_Mega_Bytes_Per_Second;
               Info_Out (Wrapper.Logger'Access,
                         "en " & Infos.Difference'Img & " secondes, debit :" &
                         Infos.Data_Rate_Bytes_Per_Second'Img &
                         " en Octets/secondes," &
                         Infos.Data_Rate_Mega_Bytes_Per_Second'Img &
                         " en MOctets/secondes");
            end select;
         end loop;
      end loop;
   end Data_Rate_Calculation_Task_Type;

   function Data_Rate (Object : access Data_Rate_Consumer_Type)
                  return Data_Rate_Info_Type is
      Function_Return : Data_Rate_Info_Type;
      use type Ada.Calendar.Time;
   begin
      Object.Stop_Date := Ada.Calendar.Clock;
      Function_Return.Difference := Object.Stop_Date -
        Object.Start_Date;
      Function_Return.Data_Rate_Bytes_Per_Second :=
        Object.Number_Of_Bytes_Received /
        Long_Float (Function_Return.Difference);
      Function_Return.Data_Rate_Mega_Bytes_Per_Second :=
        Function_Return.Data_Rate_Bytes_Per_Second /
        Long_Float (Narval.Mega_Bytes);
      return Function_Return;
   end Data_Rate;

   -----------------
   -- Initialiser --
   -----------------

   procedure On_Initialise
     (Object : access Data_Rate_Consumer_Type) is
      use Parameters;
      use Ada.Strings.Unbounded;
   begin
      Actives.On_Initialise (Active_Actor_Access (Object));
      Object.Data_Rate_Parameter :=
        new Parameters.Parameter_Type'(Container_Kind => Long_Float_Type,
                                       Mode => Read_Only,
                                       Monitor => Always,
                                       Run_Parameter => False,
                                       Editor => None,
                                       Name => To_Unbounded_String ("debit"),
                                       Long_Float_Value => 0.0);

      Parameter_Vector_Package.Append (Object.Parameters_List,
                                 Object.Data_Rate_Parameter);
      Object.Inputs (1).Memory.Toggle_Dumping_Mode
        (Event_Based_Dumping => False);
      Object.Waiting_Delay := 0.1;
   end On_Initialise;

   ----------------------------
   -- Travail_En_Acquisition --
   ----------------------------

   procedure Buffer_Handling
     (Object : access Data_Rate_Consumer_Type) is
   begin
      declare
         use Protected_Memory;
         use type System.Storage_Elements.Storage_Offset;
         Data : Raw_Access_Type;
         Manipulation_Buffer : Buffer_Enumeration_Type;
      begin
         for I in Object.Inputs'Range loop
            select
               Object.Inputs (I).Memory.Lock_Buffer
                 (Manipulation_Buffer);
               begin
                  Data :=
                    Object.Inputs
                    (I).Memory.Get_Buffer_Structure
                    (Manipulation_Buffer);
                  Object.Number_Of_Bytes_Received :=
                    Object.Number_Of_Bytes_Received +
                    Long_Float (Data.Size);
                  Object.Inputs
                    (I).Memory.Unlock_Buffer (Manipulation_Buffer);
               exception
                  when others =>
                     Object.Inputs
                       (I).Memory.Unlock_Buffer (Manipulation_Buffer);
               end;
            or
               delay Object.Waiting_Delay;
            end select;
         end loop;
      end;
   exception
      when E : others =>
         Fatal_Out (Object.Logger'Access, "travail debit", E);
   end Buffer_Handling;

   procedure On_Start
     (Object : access Data_Rate_Consumer_Type) is
   begin
      Object.Start_Date := Ada.Calendar.Clock;
      Object.Number_Of_Bytes_Received := 0.0;
      Consumers.On_Start (Consumer_Access (Object));
      Object.Data_Rate_Calculation_Task.Start;
   end On_Start;

   procedure On_Stop (Object : access Data_Rate_Consumer_Type) is
      Infos : Data_Rate_Info_Type;
   begin
      Object.Data_Rate_Calculation_Task.Stop;
      Consumers.On_Stop (Consumer_Access (Object));
      Infos := Data_Rate (Object);
      Info_Out (Object.Logger'Access,
                "nombre octets recus : " &
                Object.Number_Of_Bytes_Received'Img &
                " en " & Infos.Difference'Img & " secondes, debit :" &
                Infos.Data_Rate_Bytes_Per_Second'Img & " en Octets/secondes," &
                Infos.Data_Rate_Mega_Bytes_Per_Second'Img &
                " en MOctets/secondes");
   exception
      when E : others =>
         Fatal_Out (Object.Logger'Access, "Sur_Arreter", E);
   end On_Stop;

   procedure On_Suspend
     (Object : access Data_Rate_Consumer_Type) is
   begin
      Consumers.On_Suspend (Consumer_Access
                                   (Object));
      Object.Pause_On := True;
   end On_Suspend;

end Narval.Actors.Actives.Consumers.Data_Rate;
